<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Models\UserModuleAccess;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use Inertia\Response;

class AuthenticatedSessionController extends Controller
{
    /**
     * Show the login page.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('auth/login', [
            'canResetPassword' => Route::has('password.request'),
            'status' => $request->session()->get('status'),
        ]);
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();
        $request->session()->regenerate();

        $user = Auth::user();

        // Known module slugs your app supports for routing decisions
        $known = ['primary', 'dental', 'chronics', 'secondary', 'medicals'];

        // SUPERADMIN → treat as multiple modules → go to Kiosk
        if ($user->role?->value === 'superadmin') {
            // Clear any prior selection; Kiosk will let them choose
            $request->session()->forget('active_module');
            return redirect()->route('kiosk.index');
        }

        // Fetch the user's module slugs via the pivot table
        $slugs = UserModuleAccess::query()
            ->where('user_id', $user->id)
            ->whereHas('module') // ensure module exists
            ->with('module:id,slug')
            ->get()
            ->pluck('module.slug')
            ->filter(fn ($s) => in_array($s, $known, true))
            ->values()
            ->all();

        // Unique & normalized
        $slugs = array_values(array_unique(array_map('strval', $slugs)));

        // Decide redirect target
        if (count($slugs) === 1) {
            $only = $slugs[0];
            $request->session()->put('active_module', $only);

            // Send them directly to that module's dashboard
            return match ($only) {
                'dental'   => redirect()->route('dental.dashboard'),
                'chronics' => redirect()->route('chronics.dashboard'),
                // If you add dashboards for these, swap these to their routes.
                'secondary' => redirect()->route('kiosk.index')->with('info', 'Secondary module coming soon.'),
                'medicals'  => redirect()->route('kiosk.index')->with('info', 'Medicals module coming soon.'),
                default    => redirect()->route('dashboard'), // 'primary'
            };
        }

        if (count($slugs) > 1) {
            // Multiple choices → Kiosk (don’t preselect)
            $request->session()->forget('active_module');
            return redirect()->route('kiosk.index');
        }

        // No explicit access rows → default to primary
        $request->session()->put('active_module', 'primary');
        return redirect()->route('dashboard');
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/login');
    }
}
