<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

// OpenSpout v4
use OpenSpout\Reader\XLSX\Reader as XLSXReader;
use OpenSpout\Reader\CSV\Reader as CSVReader;
use OpenSpout\Reader\CSV\Options as CSVOptions;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;
use OpenSpout\Common\Entity\Row as SpoutRow;

class BulkUploadController extends Controller
{
    // ================== BULK IMPORT (UI) ==================
    public function bulkImportForm()
    {
        $companies = Company::orderBy('name')->get(['id', 'name']);

        return inertia('Patients/BulkImport', [
            'templates' => [
                'patients' => [
                    'headers' => [
                        'first_name', 'middle_name', 'surname', 'employee_number', 'company',
                        'date_of_birth', 'email', 'gender', 'id_number', 'phone',
                        'home_address', 'work_area', 'suburb', 'marital_status',
                        'emergency_contact_name', 'emergency_contact_relation', 'emergency_contact_phone',
                        'allergies', 'is_smoker', 'occupation',
                        'medical_aid_name', 'medical_aid_number',
                    ],
                ],
                'dependents' => [
                    'headers' => [
                        'first_name', 'middle_name', 'surname', 'relationship', 'phone', 'gender', 'date_of_birth', 'id_number', 'email',
                        'parent_employee_number', 'parent_id_number', 'parent_id',
                        'home_address', 'work_area', 'suburb', 'marital_status',
                        'emergency_contact_name', 'emergency_contact_relation', 'emergency_contact_phone',
                        'allergies', 'is_smoker', 'occupation',
                        'medical_aid_name', 'medical_aid_number',
                    ],
                ],
            ],
            'companies' => $companies,
        ]);
    }

    // ================== BULK IMPORT (PREVIEW) ==================
    public function bulkImportPreview(Request $request)
    {
        $request->validate([
            'mode'        => 'required|in:patients,dependents',
            'file'        => 'required|file|mimes:xlsx,csv,txt',
            'company_id'  => 'nullable|exists:companies,id',
        ]);

        [$headers, $rows] = $this->readSpreadsheetToArrays($request->file('file'));

        $mode    = $request->string('mode')->toString();
        $preview = [];
        $valid   = 0;
        $invalid = 0;

        foreach ($rows as $i => $rowValues) {
            $row = $this->mapRowByHeaders($headers, $rowValues);

            if ($mode === 'patients') {
                [$ok, $errs, $payload] = $this->validatePatientRow($row);
            } else {
                [$ok, $errs, $payload] = $this->validateDependentRow($row);
            }

            $preview[] = [
                'index'  => $i + 2,
                'data'   => $payload,
                'errors' => $errs,
                'raw'    => $row,
            ];

            $ok ? $valid++ : $invalid++;
        }

        return response()->json([
            'columns' => $headers,
            'rows'    => $preview,
            'stats'   => ['total' => count($rows), 'valid' => $valid, 'invalid' => $invalid],
        ]);
    }

    // ================== BULK IMPORT (COMMIT) ==================
    public function bulkImportCommit(Request $request)
    {
        $request->validate([
            'mode'        => 'required|in:patients,dependents',
            'file'        => 'required|file|mimes:xlsx,csv,txt',
            'company_id'  => 'required_if:mode,patients|nullable|exists:companies,id',
        ], [
            'company_id.required_if' => 'Please select a company for Employees import.',
        ]);

        [$headers, $rows] = $this->readSpreadsheetToArrays($request->file('file'));

        $mode            = $request->string('mode')->toString();
        $targetCompanyId = $request->integer('company_id') ?: null;

        $created        = 0;
        $failed         = 0;

        $invalidRows    = []; // [ ['row_index'=>int, 'raw'=>[], 'errors'=>[]] ]
        $duplicateRows  = []; // [ ['row_index'=>int, 'raw'=>[], 'reason'=>string, 'match'=>Patient] ]

        DB::beginTransaction();
        try {
            foreach ($rows as $i => $rowValues) {
                $row = $this->mapRowByHeaders($headers, $rowValues);

                if ($mode === 'patients') {
                    // 1) Validate payload shape (phone defaults handled here)
                    [$ok, $errs, $payload] = $this->validatePatientRow($row);
                    if (!$ok) {
                        $failed++;
                        $invalidRows[] = ['row_index' => $i + 2, 'raw' => $row, 'errors' => $errs];
                        continue;
                    }

                    // 2) Require selected company and force it
                    if (!$targetCompanyId) {
                        $failed++;
                        $invalidRows[] = [
                            'row_index' => $i + 2,
                            'raw' => $row,
                            'errors' => ['No company selected for employees import']
                        ];
                        continue;
                    }
                    $payload['company_id'] = $targetCompanyId;

                    // 3) Duplicate checks
                    $empNo = $row['employee_number'] ?? null;
                    $natId = $row['id_number'] ?? null;

                    // (a) Duplicate within SAME COMPANY by employee_number
                    $existingByEmp = null;
                    if ($empNo) {
                        $existingByEmp = Patient::query()
                            ->whereNull('parent_patient_id')
                            ->where('company_id', $targetCompanyId)
                            ->where('employee_number', $empNo)
                            ->first();
                    }

                    // (b) Duplicate globally by id_number (kept)
                    $existingById = null;
                    if ($natId) {
                        $existingById = Patient::query()
                            ->whereNull('parent_patient_id')
                            ->where('id_number', $natId)
                            ->first();
                    }

                    if ($existingByEmp || $existingById) {
                        $failed++;
                        $existing = $existingByEmp ?: $existingById;
                        $reason   = $this->duplicateReason($existing, $empNo, $natId, $targetCompanyId);

                        $duplicateRows[] = [
                            'row_index' => $i + 2,
                            'raw'       => $row,
                            'reason'    => $reason,
                            'match'     => $existing,
                        ];
                        continue;
                    }

                    // 4) Create account holder
                    Patient::create($payload);
                    $created++;
                } else {
                    // Dependents flow (phone defaults handled here)
                    [$ok, $errs, $payload] = $this->validateDependentRow($row);
                    if (!$ok) {
                        $failed++;
                        $invalidRows[] = ['row_index' => $i + 2, 'raw' => $row, 'errors' => $errs];
                        continue;
                    }

                    $parentId = $this->resolveParentId(
                        $row['parent_id'] ?? null,
                        $row['parent_employee_number'] ?? null,
                        $row['parent_id_number'] ?? null
                    );
                    if (!$parentId) {
                        $failed++;
                        $invalidRows[] = [
                            'row_index' => $i + 2,
                            'raw' => $row,
                            'errors' => ['Parent not found by id / employee_number / id_number']
                        ];
                        continue;
                    }
                    $payload['parent_patient_id'] = $parentId;

                    Patient::create($payload);
                    $created++;
                }
            }

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'ok'      => false,
                'message' => 'Import failed: ' . $e->getMessage(),
            ], 500);
        }

        // Summaries for Swal
        $invalidSummary   = $this->summarizeInvalidReasons($invalidRows);
        $duplicateSummary = $this->summarizeDuplicateReasons($duplicateRows);

        // Examples (first 5)
        $invalidExamples = array_slice(array_map(function ($r) {
            return ['row_index' => $r['row_index'], 'errors' => $r['errors']];
        }, $invalidRows), 0, 5);

        $duplicateExamples = array_slice(array_map(function ($r) {
            return ['row_index' => $r['row_index'], 'reason' => $r['reason']];
        }, $duplicateRows), 0, 5);

        // Artifacts (route URLs)
        $invalidUrl   = !empty($invalidRows)   ? $this->writeInvalidXlsxAndGetUrl($headers, $invalidRows, $mode)      : null;
        $duplicateUrl = !empty($duplicateRows) ? $this->writeDuplicatesXlsxAndGetUrl($headers, $duplicateRows, $mode) : null;

        return response()->json([
            'ok'                        => true,
            'created'                   => $created,
            'failed'                    => $failed,
            'errors'                    => [],
            'invalid_download_url'      => $invalidUrl,
            'duplicates_download_url'   => $duplicateUrl,
            'invalid_reasons_summary'   => $invalidSummary,
            'duplicate_reasons_summary' => $duplicateSummary,
            'invalid_examples'          => $invalidExamples,
            'duplicate_examples'        => $duplicateExamples,
        ]);
    }

    // ================== SECURE ARTIFACT DOWNLOAD ==================
    public function downloadArtifact(Request $request, string $type, string $filename)
    {
        if (!in_array($type, ['invalid', 'duplicates'], true)) abort(404);
        if (!preg_match('/^[-A-Za-z0-9_.]+$/', $filename))    abort(404);

        $relative = "bulk-import/{$type}/{$filename}";
        if (!Storage::disk('public')->exists($relative)) abort(404);

        return Storage::disk('public')->download($relative);
    }

    // ================== Helpers ==================

    private function readSpreadsheetToArrays(UploadedFile $file): array
    {
        $ext = strtolower($file->getClientOriginalExtension() ?: $file->guessExtension() ?: '');

        if ($ext === 'xlsx') {
            $reader = new XLSXReader();
        } else {
            $csvOptions = new CSVOptions();
            $csvOptions->fieldDelimiter = ',';
            $csvOptions->fieldEnclosure = '"';
            $csvOptions->encoding       = 'UTF-8';
            $reader = new CSVReader($csvOptions);
        }

        $reader->open($file->getRealPath());

        $headers = [];
        $rows    = [];

        foreach ($reader->getSheetIterator() as $sheet) {
            foreach ($sheet->getRowIterator() as $idx => $row) {
                $cells = $row->getCells();
                $vals  = array_map(function ($cell) {
                    $v = $cell->getValue();
                    return is_string($v) ? trim($v) : $v;
                }, $cells);

                if ($idx === 1) {
                    $headers = array_map(fn($h) => strtolower(trim((string) $h)), $vals);
                    continue;
                }

                if (count(array_filter($vals, fn($v) => $v !== null && $v !== '')) === 0) continue;

                $rows[] = $vals;
            }
            break; // first sheet only
        }

        $reader->close();

        return [$headers, $rows];
    }

    private function mapRowByHeaders(array $headers, array $vals): array
    {
        $out = [];
        foreach ($headers as $i => $key) {
            $out[$key] = $vals[$i] ?? null;
        }
        return $out;
    }

    private function coalesce(array $row, array $keys)
    {
        foreach ($keys as $k) {
            if (array_key_exists($k, $row) && $row[$k] !== null && $row[$k] !== '') {
                return $row[$k];
            }
        }
        return null;
    }

    private function normalizeName(?string $v): ?string
    {
        if ($v === null) return null;
        $v = preg_replace('/\s+/u', ' ', trim($v));
        return Str::title($v);
    }

    private function parseDateFlexible($v): ?string
    {
        if ($v === null || $v === '') return null;

        if (is_numeric($v)) {
            try {
                $dt = Carbon::createFromTimestampUTC(((float)$v - 25569) * 86400);
                return $dt->toDateString();
            } catch (\Throwable $e) {}
        }

        $s = trim((string) $v);

        foreach (['Y-m-d', 'Y/m/d'] as $fmt) {
            $dt = Carbon::createFromFormat($fmt, $s, 'UTC');
            if ($dt !== false) return $dt->toDateString();
        }
        foreach (['d/m/Y', 'm/d/Y'] as $fmt) {
            try {
                $dt = Carbon::createFromFormat($fmt, $s, 'UTC');
                if ($dt && $dt->format($fmt) === $s) return $dt->toDateString();
            } catch (\Throwable $e) {}
        }
        foreach (['d-m-Y', 'm-d-Y'] as $fmt) {
            try {
                $dt = Carbon::createFromFormat($fmt, $s, 'UTC');
                if ($dt && $dt->format($fmt) === $s) return $dt->toDateString();
            } catch (\Throwable $e) {}
        }

        try {
            return Carbon::parse($s)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function resolveParentId($parentId, $employeeNumber, $idNumber): ?int
    {
        if ($parentId) {
            $p = Patient::whereNull('parent_patient_id')->find($parentId);
            if ($p) return $p->id;
        }
        if ($employeeNumber) {
            $p = Patient::whereNull('parent_patient_id')->where('employee_number', $employeeNumber)->first();
            if ($p) return $p->id;
        }
        if ($idNumber) {
            $p = Patient::whereNull('parent_patient_id')->where('id_number', $idNumber)->first();
            if ($p) return $p->id;
        }
        return null;
    }

    private function validatePatientRow(array $row): array
    {
        $errs = [];

        $first  = $this->normalizeName($row['first_name'] ?? null);
        $last   = $this->normalizeName($row['surname'] ?? null);
        $middle = $this->normalizeName($row['middle_name'] ?? null);

        // NEW: default phone
        $phoneInput = trim((string) ($row['phone'] ?? ''));
        $phone = $phoneInput !== '' ? $phoneInput : '263';

        if (!$first) $errs[] = 'first_name is required';
        if (!$last)  $errs[] = 'surname is required';
        // phone no longer required-error; we default to '263'

        $gender = $row['gender'] ?? null;
        if ($gender) {
            if (strtoupper((string)$gender) === 'M') $gender = 'Male';
            elseif (strtoupper((string)$gender) === 'F') $gender = 'Female';

            if (!in_array($gender, ['Male', 'Female', 'Other'], true)) {
                $errs[] = "gender must be Male, Female or Other";
            }
        }

        $dob = $this->parseDateFlexible($row['date_of_birth'] ?? null);
        if (($row['date_of_birth'] ?? null) && !$dob) $errs[] = "date_of_birth could not be parsed";

        $medicalAidProvider = $this->coalesce($row, ['medical_aid_provider', 'medical_aid_name', 'medical_aid']);
        $medicalAidNumber   = $this->coalesce($row, ['medical_aid_number', 'medical_aid_no']);

        $payload = [
            'first_name'                 => $first,
            'middle_name'                => $middle,
            'surname'                    => $last,
            'employee_number'            => $row['employee_number'] ?? null,
            'date_of_birth'              => $dob,
            'email'                      => $row['email'] ?? null,
            'gender'                     => $gender ?: null,
            'id_number'                  => $row['id_number'] ?? null,
            'phone'                      => $phone, // NEW: defaulted if blank
            'emergency_contact_name'     => $row['emergency_contact_name'] ?? null,
            'emergency_contact_relation' => $row['emergency_contact_relation'] ?? null,
            'emergency_contact_phone'    => $row['emergency_contact_phone'] ?? null,
            'allergies'                  => $row['allergies'] ?? null,
            'is_smoker'                  => isset($row['is_smoker']) ? in_array(strtolower((string)$row['is_smoker']), ['1', 'yes', 'true'], true) : null,
            'occupation'                 => $row['occupation'] ?? null,
            'home_address'               => $row['home_address'] ?? null,
            'work_area'                  => $row['work_area'] ?? null,
            'suburb'                     => $row['suburb'] ?? null,
            'marital_status'             => $row['marital_status'] ?? null,
            'medical_aid_provider'       => $medicalAidProvider ?: null,
            'medical_aid_number'         => $medicalAidNumber ?: null,
        ];

        return [count($errs) === 0, $errs, $payload];
    }

    private function validateDependentRow(array $row): array
    {
        $errs = [];

        $first  = $this->normalizeName($row['first_name'] ?? null);
        $last   = $this->normalizeName($row['surname'] ?? null);
        $middle = $this->normalizeName($row['middle_name'] ?? null);

        // NEW: default phone
        $phoneInput = trim((string) ($row['phone'] ?? ''));
        $phone = $phoneInput !== '' ? $phoneInput : '263';

        if (!$first) $errs[] = 'first_name is required';
        if (!$last)  $errs[] = 'surname is required';
        // phone no longer required-error; we default to '263'

        $relationship = $row['relationship'] ?? 'Other';

        $gender = $row['gender'] ?? null;
        if ($gender) {
            if (strtoupper((string)$gender) === 'M') $gender = 'Male';
            elseif (strtoupper((string)$gender) === 'F') $gender = 'Female';

            if (!in_array($gender, ['Male', 'Female', 'Other'], true)) {
                $errs[] = "gender must be Male, Female or Other";
            }
        }

        $dob = $this->parseDateFlexible($row['date_of_birth'] ?? null);
        if (($row['date_of_birth'] ?? null) && !$dob) $errs[] = "date_of_birth could not be parsed";

        $medicalAidProvider = $this->coalesce($row, ['medical_aid_provider', 'medical_aid_name', 'medical_aid']);
        $medicalAidNumber   = $this->coalesce($row, ['medical_aid_number', 'medical_aid_no']);

        $payload = [
            'first_name'                 => $first,
            'middle_name'                => $middle,
            'surname'                    => $last,
            'relationship'               => $relationship,
            'date_of_birth'              => $dob,
            'email'                      => $row['email'] ?? null,
            'gender'                     => $gender ?: null,
            'id_number'                  => $row['id_number'] ?? null,
            'phone'                      => $phone, // NEW: defaulted if blank
            'emergency_contact_name'     => $row['emergency_contact_name'] ?? null,
            'emergency_contact_relation' => $row['emergency_contact_relation'] ?? null,
            'emergency_contact_phone'    => $row['emergency_contact_phone'] ?? null,
            'allergies'                  => $row['allergies'] ?? null,
            'is_smoker'                  => isset($row['is_smoker']) ? in_array(strtolower((string)$row['is_smoker']), ['1', 'yes', 'true'], true) : null,
            'occupation'                 => $row['occupation'] ?? null,
            'home_address'               => $row['home_address'] ?? null,
            'work_area'                  => $row['work_area'] ?? null,
            'suburb'                     => $row['suburb'] ?? null,
            'marital_status'             => $row['marital_status'] ?? null,
            'medical_aid_provider'       => $medicalAidProvider ?: null,
            'medical_aid_number'         => $medicalAidNumber ?: null,
        ];

        return [count($errs) === 0, $errs, $payload];
    }

    /**
     * Build a duplicate reason with company-awareness.
     */
    private function duplicateReason(Patient $existing, ?string $empNo, ?string $natId, ?int $targetCompanyId = null): string
    {
        if ($empNo && $existing->employee_number === $empNo && $targetCompanyId && (int)$existing->company_id === (int)$targetCompanyId) {
            return 'Employee number already exists in the selected company';
        }
        if ($natId && $existing->id_number === $natId) {
            return 'National ID already exists';
        }
        return 'Duplicate detected';
    }

    /** Aggregate counts of invalid reasons. */
    private function summarizeInvalidReasons(array $invalidRows): array
    {
        $counts = [];
        foreach ($invalidRows as $r) {
            $errs = $r['errors'] ?? [];
            foreach ($errs as $e) {
                $counts[$e] = ($counts[$e] ?? 0) + 1;
            }
        }
        arsort($counts);
        $out = [];
        foreach ($counts as $reason => $count) {
            $out[] = ['reason' => (string) $reason, 'count' => (int) $count];
        }
        return array_slice($out, 0, 20);
    }

    /** Aggregate counts of duplicate reasons. */
    private function summarizeDuplicateReasons(array $duplicateRows): array
    {
        $counts = [];
        foreach ($duplicateRows as $r) {
            $reason = $r['reason'] ?? 'Duplicate';
            $counts[$reason] = ($counts[$reason] ?? 0) + 1;
        }
        arsort($counts);
        $out = [];
        foreach ($counts as $reason => $count) {
            $out[] = ['reason' => (string) $reason, 'count' => (int) $count];
        }
        return array_slice($out, 0, 20);
    }

    /**
     * Build an XLSX of invalid rows and return a controller route URL.
     */
    private function writeInvalidXlsxAndGetUrl(array $headers, array $invalidRows, string $mode): ?string
    {
        if (empty($invalidRows)) return null;

        $xlsxHeaders   = array_values($headers);
        $xlsxHeaders[] = 'error_messages';
        $xlsxHeaders[] = 'source_row_index';

        $filename = 'bulk-invalid-' . $mode . '-' . time() . '.xlsx';
        $relative = 'bulk-import/invalid/' . $filename;

        Storage::disk('public')->makeDirectory('bulk-import/invalid');
        $fullPath = Storage::disk('public')->path($relative);

        $writer = new XLSXWriter();
        $writer->openToFile($fullPath);
        $writer->addRow(SpoutRow::fromValues($xlsxHeaders));

        foreach ($invalidRows as $bad) {
            $rowIndex = $bad['row_index'] ?? null;
            $raw      = $bad['raw'] ?? [];
            $errs     = $bad['errors'] ?? [];

            $line = [];
            foreach ($headers as $h) {
                $line[] = $raw[$h] ?? null;
            }
            $line[] = is_array($errs) ? implode('; ', $errs) : (string)$errs;
            $line[] = $rowIndex;

            $writer->addRow(SpoutRow::fromValues($line));
        }

        $writer->close();

        return route('bulk.import.download', ['type' => 'invalid', 'filename' => $filename]);
    }

    /**
     * Build an XLSX for duplicates and return a controller route URL.
     */
    private function writeDuplicatesXlsxAndGetUrl(array $headers, array $duplicateRows, string $mode): ?string
    {
        if (empty($duplicateRows)) return null;

        $xlsxHeaders   = array_values($headers);
        $xlsxHeaders[] = 'duplicate_reason';
        $xlsxHeaders[] = 'matched_patient_id';
        $xlsxHeaders[] = 'matched_full_name';
        $xlsxHeaders[] = 'matched_employee_number';
        $xlsxHeaders[] = 'matched_id_number';
        $xlsxHeaders[] = 'source_row_index';

        $filename = 'bulk-duplicates-' . $mode . '-' . time() . '.xlsx';
        $relative = 'bulk-import/duplicates/' . $filename;

        Storage::disk('public')->makeDirectory('bulk-import/duplicates');
        $fullPath = Storage::disk('public')->path($relative);

        $writer = new XLSXWriter();
        $writer->openToFile($fullPath);
        $writer->addRow(SpoutRow::fromValues($xlsxHeaders));

        foreach ($duplicateRows as $dup) {
            $rowIndex = $dup['row_index'] ?? null;
            $raw      = $dup['raw'] ?? [];
            /** @var Patient|null $m */
            $m        = $dup['match'] ?? null;
            $reason   = $dup['reason'] ?? 'Duplicate';

            $line = [];
            foreach ($headers as $h) {
                $line[] = $raw[$h] ?? null;
            }
            $line[] = $reason;
            $line[] = $m?->id;
            $line[] = $m ? trim(($m->first_name ?? '') . ' ' . ($m->surname ?? '')) : null;
            $line[] = $m?->employee_number;
            $line[] = $m?->id_number;
            $line[] = $rowIndex;

            $writer->addRow(SpoutRow::fromValues($line));
        }

        $writer->close();

        return route('bulk.import.download', ['type' => 'duplicates', 'filename' => $filename]);
    }
}
