<?php
namespace App\Http\Controllers;

use App\Models\CentralStore;
use App\Models\Clinic;
use App\Models\ClinicMedicationStock;
use App\Models\ClinicStockReceipt;
use App\Models\Log as AuditLog;
use App\Models\Medication;
use App\Models\MedicationBatch;
use App\Models\MedicationRequest;
use App\Models\StockTransfer;
use App\Models\Supplier; // <= add this at the top with the other use statements

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class CentralStoreController extends Controller
{
    public function index(Request $request)
    {
        $centralStores = CentralStore::query()
            ->when($request->input('search'), function ($query, $search) {
                $query->where('name', 'like', "%{$search}%")
                    ->orWhere('location', 'like', "%{$search}%");
            })
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('CentralStores/Index', [
            'centralStores' => $centralStores,
            'filters'       => $request->only('search'),
        ]);
    }

    /**
     * Display a listing of received batches.
     */

    public function receivedBatchesIndex(Request $request, CentralStore $centralStore)
    {
        $search       = $request->input('search');
        $medicationId = $request->input('medication_id');
        $supplierId   = $request->input('supplier_id');
        $startDate    = $request->input('start_date');
        $endDate      = $request->input('end_date');
        $expired      = $request->input('expired');
        $expiringSoon = $request->input('expiring_soon');

        $batches = MedicationBatch::with(['medication:id,name', 'supplier:id,name'])
            ->where('central_store_id', $centralStore->id)
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('batch_number', 'like', "%{$search}%")
                        ->orWhereHas('medication', fn($q) => $q->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('supplier', fn($q) => $q->where('name', 'like', "%{$search}%"));
                });
            })
            ->when($medicationId, function ($query, $medicationId) {
                $query->where('medication_id', $medicationId);
            })
            ->when($supplierId, function ($query, $supplierId) {
                $query->where('supplier_id', $supplierId);
            })
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                $query->whereBetween('received_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($expired !== null, function ($query) use ($expired) {
                $query->where('expiry_date', $expired ? '<' : '>=', now());
            })
            ->when($expiringSoon !== null, function ($query) use ($expiringSoon) {
                $query->whereBetween('expiry_date', [
                    now(),
                    now()->addDays(30),
                ]);
            })
            ->latest('updated_at') // Changed from 'received_date' to 'created_at'
            ->paginate(10)
            ->appends($request->all());

        $medications = Medication::select('id', 'name')->orderBy('name')->get();
        $suppliers   = Supplier::select('id', 'name')->orderBy('name')->get();

        return inertia('CentralStores/ReceivedBatchesIndex', [
            'batches'        => $batches,
            'medications'    => $medications,
            'suppliers'      => $suppliers,
            'filters'        => $request->only([
                'search',
                'medication_id',
                'supplier_id',
                'start_date',
                'end_date',
                'expired',
                'expiring_soon',
            ]),
            'centralStoreId' => $centralStore->id,
        ]);
    }

    /**
     * Show the form for creating a new central store.
     */
    public function create()
    {
        return Inertia::render('CentralStores/Create');
    }

    /**
     * Store a newly created central store in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'        => 'required|string|max:255',
            'location'    => 'nullable|string|max:255',
            'address'     => 'nullable|string|max:1000',
            'latitude'    => 'nullable|numeric|between:-90,90',
            'longitude'   => 'nullable|numeric|between:-180,180',
            'description' => 'nullable|string',
        ]);

        $centralStore = CentralStore::create($validated);

        // Create log entry
        AuditLog::create([
            'user_id'       => Auth::id(),
            'user_name'     => Auth::user()->name ?? 'System',
            'action'        => 'create',
            'loggable_type' => CentralStore::class,
            'loggable_id'   => $centralStore->id,
            'description'   => 'Created a new central store: ' . $centralStore->name,
            'old_values'    => null,
            'new_values'    => $validated,
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        return redirect()
            ->route('central-stores.index')
            ->with('success', 'Central Store created successfully.');
    }

    /**
     * Display the specified central store.
     */

    public function show(CentralStore $centralStore)
    {
        // Fetch all medication batches associated with the central store that are close to expiring
        $batches = MedicationBatch::where('central_store_id', $centralStore->id)
            ->with('medication', 'supplier')                            // Eager load related models for better performance
            ->whereBetween('expiry_date', [now(), now()->addDays(300)]) // Filter batches expiring within the next 30 days
            ->latest('expiry_date')                                     // Order by expiry_date in descending order
            ->paginate(10)                                              // Paginate the results
            ->withQueryString();                                        // Preserve query string parameters in pagination links

        return Inertia::render('CentralStores/Show', [
            'centralStore' => $centralStore,
            'batches'      => $batches,
        ]);
    }

    /**
     * Show the form for editing the specified central store.
     */
    public function edit(CentralStore $centralStore)
    {
        return Inertia::render('CentralStores/Edit', [
            'centralStore' => $centralStore,
        ]);
    }

    /**
     * Update the specified central store in storage.
     */
    public function update(Request $request, CentralStore $centralStore)
    {
        $validated = $request->validate([
            'name'        => 'required|string|max:255',
            'location'    => 'nullable|string|max:255',
            'address'     => 'nullable|string|max:1000',
            'latitude'    => 'nullable|numeric|between:-90,90',
            'longitude'   => 'nullable|numeric|between:-180,180',
            'description' => 'nullable|string',
        ]);

        // Capture old values before updating
        $oldValues = $centralStore->getOriginal();

        // Update store
        $centralStore->update($validated);

        // Capture changed fields only (optional, cleaner logs)
        $changes = $centralStore->getChanges();

        // Create log entry
        AuditLog::create([
            'user_id'       => Auth::id(),
            'user_name'     => Auth::user()->name ?? 'System',
            'action'        => 'update',
            'loggable_type' => CentralStore::class,
            'loggable_id'   => $centralStore->id,
            'description'   => 'Updated central store: ' . $centralStore->name,
            'old_values'    => $oldValues,
            'new_values'    => $changes,
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        return redirect()
            ->route('central-stores.index')
            ->with('success', 'Central Store updated successfully.');
    }

    /**
     * Remove the specified central store from storage.
     */
    public function destroy(CentralStore $centralStore, Request $request)
    {
        // Capture old values before deleting
        $oldValues = $centralStore->toArray();

        // Delete the central store
        $centralStore->delete();

        // Create log entry
        AuditLog::create([
            'user_id'       => Auth::id(),
            'user_name'     => Auth::user()->name ?? 'System',
            'action'        => 'deleted',
            'loggable_type' => CentralStore::class,
            'loggable_id'   => $centralStore->id,
            'description'   => 'Deleted central store: ' . ($oldValues['name'] ?? 'Unknown'),
            'old_values'    => $oldValues,
            'new_values'    => null,
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        return redirect()
            ->route('central-stores.index')
            ->with('success', 'Central Store deleted successfully.');
    }

    //Batch Management Here
    public function receiveBatchForm(CentralStore $centralStore)
    {
        return Inertia::render('CentralStores/ReceiveBatch', [
            'centralStore' => $centralStore,
            'medications'  => Medication::select('id', 'name', 'dosage', 'unit', 'form')->get(),
            'suppliers'    => Supplier::select('id', 'name')->get(),
        ]);
    }

    public function receiveBatch(Request $request, CentralStore $centralStore)
    {
        try {
            // Validate the request data
            $validated = $request->validate([
                'medication_id'    => 'required|exists:medications,id',
                'supplier_id'      => 'required|exists:suppliers,id',
                'batch_number'     => 'required|string|max:255',
                'quantity'         => 'required|integer|min:1',
                'expiry_date'      => 'required|date|after:today',
                'manufacture_date' => 'nullable|date|before_or_equal:today',
                'received_date'    => 'required|date|before_or_equal:today',
                'invoice_number'   => 'nullable|string|max:255',
                'price'            => 'nullable|numeric|min:0',
                'invoice_file'     => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240', // 10MB
            ]);

            // Normalize and add store
            $validated['batch_number']     = trim($validated['batch_number']);
            $validated['central_store_id'] = $centralStore->id;

            // Handle invoice file upload if provided
            if ($request->hasFile('invoice_file')) {
                $file         = $request->file('invoice_file');
                $supplierName = Supplier::find($request->supplier_id)?->name ?? 'unknown_supplier';
                $filename     = strtolower(
                    'invoice_' .
                    preg_replace('/\s+/', '_', $request->batch_number) . '_' .
                    preg_replace('/\s+/', '_', $supplierName) . '_' .
                    time() . '.' . $file->getClientOriginalExtension()
                );
                $path                      = $file->storeAs('invoices', $filename, 'public');
                $validated['invoice_file'] = $path;
            }

            DB::beginTransaction();

            // 1) If there's a FULL MATCH of identifying fields, TOP-UP quantity
            $fullMatch = MedicationBatch::where([
                'central_store_id' => $centralStore->id,
                'medication_id'    => $validated['medication_id'],
                'supplier_id'      => $validated['supplier_id'],
                'batch_number'     => $validated['batch_number'],
                'expiry_date'      => $validated['expiry_date'],
                'manufacture_date' => $validated['manufacture_date'],
            ])->first();

            if ($fullMatch) {
                $beforeQty = $fullMatch->quantity;
                $fullMatch->increment('quantity', (int) $validated['quantity']);

                // Optionally keep latest received_date if it's later
                if (! empty($validated['received_date']) && $validated['received_date'] > ($fullMatch->received_date ?? '')) {
                    $fullMatch->received_date = $validated['received_date'];
                }

                // Optionally update invoice_number/price if provided
                if (! empty($validated['invoice_number'])) {
                    $fullMatch->invoice_number = $validated['invoice_number'];
                }
                if (array_key_exists('price', $validated) && $validated['price'] !== null) {
                    $fullMatch->price = $validated['price'];
                }
                if (! empty($validated['invoice_file'])) {
                    // if you want to keep the latest file path
                    $fullMatch->invoice_file = $validated['invoice_file'];
                }

                $fullMatch->save();

                $medicationName = Medication::find($validated['medication_id'])->name;

                AuditLog::create([
                    'user_id'       => Auth::id(),
                    'user_name'     => Auth::user()->name ?? 'System',
                    'action'        => 'update',
                    'loggable_type' => MedicationBatch::class,
                    'loggable_id'   => $fullMatch->id,
                    'description'   => 'Topped up existing medication batch #' . $fullMatch->batch_number .
                    ' of ' . $medicationName . ' for store: ' . $centralStore->name,
                    'old_values'    => ['quantity' => $beforeQty],
                    'new_values'    => ['quantity' => $fullMatch->quantity],
                    'ip_address'    => $request->ip(),
                    'user_agent'    => $request->header('User-Agent'),
                ]);

                DB::commit();

                return redirect()
                    ->route('central-stores.transfer-stock-form', [
                        'centralStore' => $centralStore->id,
                        'batch'        => $fullMatch->id,
                    ])
                    ->with('success', 'Existing batch found — quantity topped up successfully. Proceed to transfer.');
            }

            // 2) If batch_number already exists for this store but with DIFFERENT details, block and explain
            $sameNumberDifferentDetails = MedicationBatch::where('central_store_id', $centralStore->id)
                ->where('batch_number', $validated['batch_number'])
                ->where(function ($q) use ($validated) {
                    $q->where('medication_id', '!=', $validated['medication_id'])
                        ->orWhere('supplier_id', '!=', $validated['supplier_id'])
                        ->orWhere('expiry_date', '!=', $validated['expiry_date'])
                        ->orWhere('manufacture_date', '!=', $validated['manufacture_date']);
                })
                ->exists();

            if ($sameNumberDifferentDetails) {
                DB::rollBack();
                return redirect()
                    ->back()
                    ->withErrors([
                        'batch_number' => 'This batch number is already used for a different set of batch details. ' .
                        'Please use a unique batch number or adjust the details to match the existing batch if you intend to top up.',
                    ])
                    ->withInput();
            }

            // 3) Otherwise, CREATE a new batch
            $batch = MedicationBatch::create($validated);

            $medicationName = Medication::find($validated['medication_id'])->name;

            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? 'System',
                'action'        => 'create',
                'loggable_type' => MedicationBatch::class,
                'loggable_id'   => $batch->id,
                'description'   => 'Received new medication batch #' . $batch->batch_number .
                ' of ' . $medicationName .
                ' for store: ' . $centralStore->name,
                'old_values'    => null,
                'new_values'    => $validated,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();

            // ✅ Redirect to transfer stock form after receiving batch
            return redirect()
                ->route('central-stores.transfer-stock-form', [
                    'centralStore' => $centralStore->id,
                    'batch'        => $batch->id,
                ])
                ->with('success', 'Medication batch received successfully. Proceed to transfer.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('Database query failed: ' . $e->getMessage());
            DB::rollBack();
            return redirect()->back()->with('error', 'Failed to receive medication batch due to a database error.');
        } catch (\Exception $e) {
            Log::error('An error occurred: ' . $e->getMessage());
            DB::rollBack();
            return redirect()->back()->with('error', 'An unexpected error occurred. Please try again later.');
        }
    }

    /**
     * Show the form for transferring stock to clinics
     */
    public function transferStockForm(CentralStore $centralStore, MedicationBatch $batch)
    {
        $clinics = Clinic::select('id', 'name')->get();

        return Inertia::render('CentralStores/TransferStock', [
            'centralStore' => $centralStore,
            'batch'        => $batch->load('medication'),
            'clinics'      => $clinics,
            'currentStock' => $batch->quantity,
        ]);
    }

/**
 * Process the stock transfer to clinics
 */

    public function transferStock(Request $request, CentralStore $centralStore, MedicationBatch $batch)
    {
        try {
            $validated = $request->validate([
                'clinic_id'     => 'required|exists:clinics,id',
                'request_id'    => 'nullable|integer|exists:medication_requests,id',
                'quantity'      => [
                    'required',
                    'integer',
                    'min:1',
                    function ($attribute, $value, $fail) use ($batch) {
                        if ($value > $batch->quantity) {
                            $fail('The transfer quantity cannot exceed available stock.');
                        }
                    },
                ],
                'transfer_date' => 'required|date|before_or_equal:today',
            ]);

            DB::beginTransaction();

            // 1. Reduce stock in central store
            $batch->decrement('quantity', $validated['quantity']);

            // 2. Record the transfer
            $stockTransfer = StockTransfer::create([
                'from_central_store_id' => $centralStore->id,
                'to_clinic_id'          => $validated['clinic_id'],
                'medication_batch_id'   => $batch->id,
                'quantity'              => $validated['quantity'],
                'transfer_date'         => $validated['transfer_date'],
                'transferred_by'        => Auth::id(),
                'status'                => 'pending',
            ]);

            // 3. Create clinic stock receipt
            ClinicStockReceipt::create([
                'stock_transfer_id' => $stockTransfer->id,
                'clinic_id'         => $validated['clinic_id'],
                'notes'             => 'Pending approval',
            ]);

            // 4. Approve medication request if provided
            if (! empty($validated['request_id'])) {
                $medRequest = MedicationRequest::find($validated['request_id']);
                if ($medRequest) {
                    $medRequest->update([
                        'status'               => 'approved',
                        'transferred_quantity' => $validated['quantity'],
                    ]);
                }
            }

            // 5. Audit log
            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? 'System',
                'action'        => 'transfer',
                'loggable_type' => StockTransfer::class,
                'loggable_id'   => $stockTransfer->id,
                'description'   => "Transferred {$validated['quantity']} units from central store '{$centralStore->name}' (Batch #{$batch->batch_number}) to clinic ID {$validated['clinic_id']}",
                'old_values'    => ['batch_quantity_before' => $batch->quantity + $validated['quantity']],
                'new_values'    => ['batch_quantity_after' => $batch->quantity],
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();

            return redirect()
                ->route('central-stores.stock-transfers.index', $centralStore->id)
                ->with('success', 'Stock transfer initiated successfully. Awaiting approval from the receiving clinic.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Stock transfer failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to transfer stock. Please try again.');
        }
    }

    public function transferStockJson(Request $request, CentralStore $centralStore, MedicationBatch $batch)
    {
        try {
            $validated = $request->validate([
                'clinic_id'     => 'required|exists:clinics,id',
                'request_id'    => 'nullable|integer|exists:medication_requests,id',
                'quantity'      => [
                    'required',
                    'integer',
                    'min:1',
                    function ($attribute, $value, $fail) use ($batch) {
                        if ($value > $batch->quantity) {
                            $fail('The transfer quantity cannot exceed available stock.');
                        }
                    },
                ],
                'transfer_date' => 'required|date|before_or_equal:today',
            ]);

            DB::beginTransaction();

            // Reduce stock in central store
            $batch->decrement('quantity', $validated['quantity']);

            // Record transfer
            $stockTransfer = StockTransfer::create([
                'from_central_store_id' => $centralStore->id,
                'to_clinic_id'          => $validated['clinic_id'],
                'medication_batch_id'   => $batch->id,
                'quantity'              => $validated['quantity'],
                'transfer_date'         => $validated['transfer_date'],
                'transferred_by'        => Auth::id(),
                'status'                => 'pending',
            ]);

            // Create ClinicStockReceipt
            ClinicStockReceipt::create([
                'stock_transfer_id' => $stockTransfer->id,
                'clinic_id'         => $validated['clinic_id'],
                'notes'             => 'Pending approval',
            ]);

            // If request_id present, update MedicationRequest
            if (! empty($validated['request_id'])) {
                MedicationRequest::where('id', $validated['request_id'])
                    ->update([
                        'status'               => 'approved',
                        'transferred_quantity' => $validated['quantity'],
                    ]);
            }

            DB::commit();

            return response()->json([
                'message'  => 'Stock transfer initiated successfully. Awaiting approval from the receiving clinic.',
                'transfer' => $stockTransfer,
            ], 200);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Stock transfer failed: ' . $e->getMessage());

            return response()->json([
                'message' => 'Failed to transfer stock. Please try again.',
            ], 500);
        }
    }

    public function destroyReceivedBatch(Request $request, CentralStore $centralStore, MedicationBatch $batch)
    {
        // Ensure the batch belongs to this central store
        if ((int) $batch->central_store_id !== (int) $centralStore->id) {
            abort(404);
        }

        try {
            DB::beginTransaction();

            // Keep old values for audit trail
            $oldValues = $batch->toArray();

            // Gather related Stock Transfers for this batch
            $transfers   = StockTransfer::where('medication_batch_id', $batch->id)->get();
            $transferIds = $transfers->pluck('id')->all();

            // 1) Delete Clinic Stock Receipts linked to those transfers
            //    (if you already have FK ON DELETE CASCADE from receipts->transfers,
            //     you can omit this explicit delete)
            $receiptsDeleted = 0;
            if (! empty($transferIds)) {
                $receiptsDeleted = ClinicStockReceipt::whereIn('stock_transfer_id', $transferIds)->delete();
            }

            // 2) Delete Stock Transfers that originated from this batch
            $transfersDeleted = 0;
            if (! empty($transferIds)) {
                $transfersDeleted = StockTransfer::whereIn('id', $transferIds)->delete();
            }

            // 3) Delete stored invoice file if present (non-fatal on failure)
            if (! empty($batch->invoice_file)) {
                try {
                    Storage::disk('public')->delete($batch->invoice_file);
                } catch (\Throwable $e) {
                    Log::warning('Failed to delete invoice file for batch ' . $batch->id . ': ' . $e->getMessage());
                }
            }

            // 4) Finally delete the batch itself
            $batch->delete();

            // Audit log (we keep logs for traceability)
            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? 'System',
                'action'        => 'deleted',
                'loggable_type' => MedicationBatch::class,
                'loggable_id'   => $oldValues['id'] ?? null,
                'description'   => sprintf(
                    'Deleted medication batch #%s for store: %s. Also removed %d stock transfer(s) and %d clinic receipt(s).',
                    $oldValues['batch_number'] ?? 'N/A',
                    $centralStore->name,
                    $transfersDeleted,
                    $receiptsDeleted
                ),
                'old_values'    => $oldValues,
                'new_values'    => null,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();

            return redirect()
                ->route('central-stores.received-batches.index', $centralStore->id)
                ->with('success', 'Batch and all associated transfers/receipts were deleted successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Failed to delete batch ' . $batch->id . ': ' . $e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to delete batch and related records. Please try again.');
        }
    }

    public function editReceivedBatch(CentralStore $centralStore, MedicationBatch $batch)
    {
        // Ensure the batch belongs to this central store
        if ((int) $batch->central_store_id !== (int) $centralStore->id) {
            abort(404);
        }

        return Inertia::render('CentralStores/EditBatch', [
            'centralStore' => $centralStore,
            'batch'        => $batch->load(['medication:id,name,dosage,unit,form', 'supplier:id,name']),
            'medications'  => Medication::select('id', 'name', 'dosage', 'unit', 'form')->orderBy('name')->get(),
            'suppliers'    => Supplier::select('id', 'name')->orderBy('name')->get(),
        ]);
    }

    /**
     * Update a received batch that belongs to a central store.
     * - Validates inputs
     * - Prevents duplicate identity collisions
     * - Replaces invoice file safely
     * - Logs changes in AuditLog
     */
    public function updateReceivedBatch(Request $request, CentralStore $centralStore, MedicationBatch $batch)
    {
        // Ensure the batch belongs to this central store
        if ((int) $batch->central_store_id !== (int) $centralStore->id) {
            abort(404);
        }

        $validated = $request->validate([
            'medication_id'    => ['required', 'exists:medications,id'],
            'supplier_id'      => ['required', 'exists:suppliers,id'],
            'batch_number'     => ['required', 'string', 'max:255'],
            'quantity'         => ['required', 'integer', 'min:0'],
            'expiry_date'      => ['required', 'date', 'after:today'],
            'manufacture_date' => ['nullable', 'date', 'before_or_equal:today'],
            'received_date'    => ['required', 'date', 'before_or_equal:today'],
            'invoice_number'   => ['nullable', 'string', 'max:255'],
            'price'            => ['nullable', 'numeric', 'min:0'],
            'invoice_file'     => ['nullable', 'file', 'mimes:pdf,jpg,jpeg,png', 'max:10240'], // 10MB
        ]);

        // Normalize some fields
        $validated['batch_number']     = trim($validated['batch_number']);
        $validated['central_store_id'] = $centralStore->id; // keep consistent

        DB::beginTransaction();

        try {
            // === Collision checks (exclude this batch's id) ===

            // (A) Full-identity duplicate: same medication/supplier/batch#/expiry/manufacture for THIS store
            $fullIdentityDuplicate = MedicationBatch::where([
                'central_store_id' => $centralStore->id,
                'medication_id'    => $validated['medication_id'],
                'supplier_id'      => $validated['supplier_id'],
                'batch_number'     => $validated['batch_number'],
                'expiry_date'      => $validated['expiry_date'],
                'manufacture_date' => $validated['manufacture_date'],
            ])->where('id', '!=', $batch->id)->first();

            if ($fullIdentityDuplicate) {
                DB::rollBack();
                return back()
                    ->withErrors([
                        'batch_number' => 'A batch with the same medication, supplier, batch number, expiry, and manufacture date already exists in this store. Consider topping up that batch instead of editing into a duplicate.',
                    ])
                    ->withInput();
            }

            // (B) Same batch number BUT different critical details (avoid reusing batch# for different product details)
            $sameNumberDifferentDetails = MedicationBatch::where('central_store_id', $centralStore->id)
                ->where('batch_number', $validated['batch_number'])
                ->where('id', '!=', $batch->id)
                ->where(function ($q) use ($validated) {
                    $q->where('medication_id', '!=', $validated['medication_id'])
                        ->orWhere('supplier_id', '!=', $validated['supplier_id'])
                        ->orWhereDate('expiry_date', '!=', $validated['expiry_date'])
                        ->orWhereDate('manufacture_date', '!=', $validated['manufacture_date']);
                })
                ->exists();

            if ($sameNumberDifferentDetails) {
                DB::rollBack();
                return back()
                    ->withErrors([
                        'batch_number' => 'This batch number is already used for a different set of details in this store. Use a unique batch number or align the details to match the existing batch.',
                    ])
                    ->withInput();
            }

            // Keep old values for audit
            $oldValues = $batch->getOriginal();

            // Handle invoice file replacement
            if ($request->hasFile('invoice_file')) {
                // Optional: delete old file (ignore failures)
                if (! empty($batch->invoice_file)) {
                    try {Storage::disk('public')->delete($batch->invoice_file);} catch (\Throwable $e) {}
                }

                $file         = $request->file('invoice_file');
                $supplierName = Supplier::find($validated['supplier_id'])?->name ?? 'unknown_supplier';
                $filename     = strtolower(
                    'invoice_' .
                    preg_replace('/\s+/', '_', $validated['batch_number']) . '_' .
                    preg_replace('/\s+/', '_', $supplierName) . '_' .
                    time() . '.' . $file->getClientOriginalExtension()
                );
                $path                      = $file->storeAs('invoices', $filename, 'public');
                $validated['invoice_file'] = $path;
            }

            // Update batch
            $batch->update($validated);

            // For clearer logs, only log changed fields
            $changes = $batch->getChanges();

            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? 'System',
                'action'        => 'update',
                'loggable_type' => MedicationBatch::class,
                'loggable_id'   => $batch->id,
                'description'   => 'Updated medication batch #' . $batch->batch_number . ' for store: ' . $centralStore->name,
                'old_values'    => $oldValues,
                'new_values'    => $changes ?: $validated,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();

            return redirect()
                ->route('central-stores.received-batches.index', $centralStore->id)
                ->with('success', 'Batch updated successfully.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Failed to update batch ' . $batch->id . ': ' . $e->getMessage());
            return redirect()
                ->back()
                ->with('error', 'Failed to update batch. Please try again.');
        }
    }

    // app/Http/Controllers/CentralStoreController.php

    public function returnFromClinicJson(Request $request, ClinicMedicationStock $clinicStock)
    {
        $validated = $request->validate([
            'quantity'       => ['required', 'integer', 'min:1'],
            'delete_if_zero' => ['sometimes', 'boolean'],
            'reason'         => ['sometimes', 'nullable', 'string', 'max:500'],
        ]);

        $qty          = (int) $validated['quantity'];
        $deleteIfZero = (bool) ($validated['delete_if_zero'] ?? false);
        $reason       = $validated['reason'] ?? null;

        DB::beginTransaction();

        try {
            // 1) Lock the clinic stock row under the transaction
            $lockedClinicStock = ClinicMedicationStock::query()
                ->whereKey($clinicStock->id)
                ->lockForUpdate()
                ->with('medicationBatch') // we need the batch (and its central store id)
                ->firstOrFail();

            if ($qty > $lockedClinicStock->quantity) {
                DB::rollBack();
                return response()->json([
                    'message' => 'Return quantity exceeds available clinic stock.',
                ], 422);
            }

            // 2) Resolve the central store batch (must be the same medication_batch_id)
            $batch = MedicationBatch::query()
                ->whereKey($lockedClinicStock->medication_batch_id)
                ->lockForUpdate()
                ->firstOrFail();

            // Optional: ensure batch belongs to a central store
            if (empty($batch->central_store_id)) {
                DB::rollBack();
                return response()->json([
                    'message' => 'This clinic stock is not linked to a central store batch.',
                ], 422);
            }

            // 3) Snapshot old values for logs
            $oldClinicValues = $lockedClinicStock->getOriginal();
            $oldBatchValues  = $batch->getOriginal();

            // 4) Apply the return
            $lockedClinicStock->quantity -= $qty;
            $lockedClinicStock->save();

            $batch->quantity += $qty;
            $batch->save();

            // 5) Optional: delete clinic stock row if now zero
            if ($deleteIfZero && $lockedClinicStock->quantity === 0) {
                $lockedClinicStock->delete();
            }

            // 6) Audit logs (clinic side)
            $descSuffix = $reason ? (' Reason: ' . $reason) : '';

            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? 'System',
                'action'        => 'updated',
                'loggable_type' => ClinicMedicationStock::class,
                'loggable_id'   => $clinicStock->id,
                'description'   => "Returned {$qty} unit(s) from clinic stock to central store." . $descSuffix,
                'old_values'    => $oldClinicValues,
                'new_values'    => $lockedClinicStock->exists ? $lockedClinicStock->fresh()->getAttributes() : null,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            // 7) Audit logs (central store batch side)
            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? 'System',
                'action'        => 'updated',
                'loggable_type' => MedicationBatch::class,
                'loggable_id'   => $batch->id,
                'description'   => "Received {$qty} unit(s) returned from clinic stock (Batch #{$batch->batch_number})." . $descSuffix,
                'old_values'    => $oldBatchValues,
                'new_values'    => $batch->getAttributes(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();

            return response()->json([
                'message'       => 'Stock returned to central store successfully.',
                'central_store' => $batch->central_store_id,
                'batch'         => [
                    'id'               => $batch->id,
                    'quantity'         => $batch->quantity,
                    'batch_number'     => $batch->batch_number,
                    'medication_id'    => $batch->medication_id,
                    'central_store_id' => $batch->central_store_id,
                ],
                'clinic_stock'  => $lockedClinicStock->exists
                ? $lockedClinicStock->only(['id', 'clinic_id', 'medication_batch_id', 'quantity'])
                : null,
            ], 200);

        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Return to central store failed: ' . $e->getMessage());
            return response()->json([
                'message' => 'Failed to return stock to central store.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

}
