<?php

namespace App\Http\Controllers;

use App\Models\CentralStore;
use App\Models\MedicationBatch;
use Illuminate\Http\Request;
use Carbon\Carbon;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class CentralStoreReportsController extends Controller
{
    public function medicationBatchesReport(Request $request, CentralStore $centralStore)
    {

        $search       = $request->query('search');
        $medicationId = $request->query('medication_id');
        $supplierId   = $request->query('supplier_id');
        $startDate    = $request->query('start_date');   // received_date from
        $endDate      = $request->query('end_date');     // received_date to
        // Treat these as boolean flags if present/truthy
        $expired      = filter_var($request->query('expired'), FILTER_VALIDATE_BOOLEAN);
        $expSoon      = filter_var($request->query('expiring_soon'), FILTER_VALIDATE_BOOLEAN);

        $query = MedicationBatch::query()
            ->with([
                'medication:id,name,dosage,form,unit',
                'supplier:id,name',
                'centralStore:id,name',
            ])
            ->where('central_store_id', $centralStore->id)
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('batch_number', 'like', "%{$search}%")
                       ->orWhereHas('medication', fn($m) => $m->where('name', 'like', "%{$search}%"))
                       ->orWhereHas('supplier', fn($s) => $s->where('name', 'like', "%{$search}%"));
                });
            })
            ->when($medicationId, fn($q) => $q->where('medication_id', $medicationId))
            ->when($supplierId,   fn($q) => $q->where('supplier_id', $supplierId))
            ->when($startDate && $endDate, function ($q) use ($startDate, $endDate) {
                $q->whereBetween('received_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($startDate && !$endDate, fn($q) => $q->whereDate('received_date', '>=', $startDate))
            ->when(!$startDate && $endDate, fn($q) => $q->whereDate('received_date', '<=', $endDate))
            ->when($expired, fn($q) => $q->whereDate('expiry_date', '<', now()->toDateString()))
            ->when($expSoon, function ($q) {
                $q->whereBetween('expiry_date', [now()->toDateString(), now()->addDays(30)->toDateString()]);
            });

        $batches = $query
            ->orderBy('expiry_date')
            ->orderBy('received_date')
            ->get();

        $generatedAt = now()->format('Y-m-d H:i:s');

        $filters = [];
        if ($search)       $filters[] = "Search: {$search}";
        if ($medicationId) $filters[] = "Medication ID: {$medicationId}";
        if ($supplierId)   $filters[] = "Supplier ID: {$supplierId}";
        if ($startDate)    $filters[] = "From: {$startDate}";
        if ($endDate)      $filters[] = "To: {$endDate}";
        if ($expired)      $filters[] = "Expired: yes";
        if ($expSoon)      $filters[] = "Expiring Soon (≤30d): yes";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf(
            'CentralStore_Batches_%s_%s.xlsx',
            preg_replace('/\s+/', '_', $centralStore->name ?? 'CentralStore'),
            now()->format('Ymd_His')
        );

        return response()->streamDownload(function () use ($batches, $centralStore, $generatedAt, $filtersLine) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // Sheet
            $sheetName = $this->safeSheetName('Medication Batches');
            $writer->getCurrentSheet()->setName($sheetName);

            // Header
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['CENTRAL STORE — MEDICATION BATCHES'],
                ['Central Store', $centralStore->name ?? ''],
                ['Generated At', $generatedAt],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);

            // Table header
            $headers = [
                'Batch #',
                'Medication',
                'Dosage',
                'Form',
                'Unit',
                'Supplier',
                'Quantity',
                'Expiry Date',
                'Manufacture Date',
                'Received Date',
                'Invoice #',
                'Price',
            ];
            $writer->addRow(Row::fromValues($headers, $headerStyle));

            // Column widths (optional)
            $opts = $writer->getOptions();
            $opts->setColumnWidth(34, 2); // Medication
            $opts->setColumnWidth(16, 1); // Batch #
            $opts->setColumnWidth(14, 8); // Expiry

            // Rows
            $totalQty = 0;
            foreach ($batches as $b) {
                $med = $b->medication;
                $sup = $b->supplier;

                $row = [
                    $b->batch_number ?? '',
                    $med->name ?? '',
                    $med->dosage ?? '',
                    $med->form ?? '',
                    $med->unit ?? '',
                    $sup->name ?? '',
                    (int) ($b->quantity ?? 0),
                    optional($b->expiry_date)->format('Y-m-d') ?? (is_string($b->expiry_date) ? $b->expiry_date : ''),
                    optional($b->manufacture_date)->format('Y-m-d') ?? (is_string($b->manufacture_date) ? $b->manufacture_date : ''),
                    optional($b->received_date)->format('Y-m-d') ?? (is_string($b->received_date) ? $b->received_date : ''),
                    $b->invoice_number ?? '',
                    $b->price !== null ? (float) $b->price : null,
                ];

                $totalQty += (int) ($b->quantity ?? 0);
                $writer->addRow(Row::fromValues($row, $wrapStyle));
            }

            // Footer totals
            $writer->addRow(Row::fromValues([]));
            $writer->addRow(Row::fromValues(['Total Batches', $batches->count()]));
            $writer->addRow(Row::fromValues(['Total Quantity', $totalQty]));

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /* ===== helpers ===== */

    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    private function writeSheetHeader(XLSXWriter $writer, Style $titleStyle, Style $labelStyle, array $lines): void
    {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
