<?php
namespace App\Http\Controllers\Chronics;

use App\Http\Controllers\Controller;
use App\Models\ChronicDispensation;
use App\Models\ClinicMedicationStock;
use App\Models\NurseMedicationStock;
use App\Models\Patient;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Inertia\Inertia;

class ChronicDispensationController extends Controller
{
    /**
     * Show the chronic dispensing screen (NOT tied to a consultation).
     * - Lists clinic stocks for the user’s clinic
     * - Lists the nurse’s personal stocks
     * - Lists existing chronic dispensations (optionally filtered by patient)
     */

    public function patientHistory(Request $request, Patient $patient)
    {
        $perPage = (int) ($request->integer('per_page') ?: 20);

        $query = ChronicDispensation::with([
            'dispensedBy:id,name,email',

            // CLINIC side
            'clinicMedicationStock.clinic:id,name',
            // include medication_id so medication relation can resolve
            'clinicMedicationStock.medicationBatch:id,batch_number,medication_id',
            'clinicMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',

            // NURSE side
            'nurseMedicationStock.clinic:id,name',
            'nurseMedicationStock.medicationBatch:id,batch_number,medication_id',
            'nurseMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
        ])
            ->where('patient_id', $patient->id)
            ->orderByDesc('dispensed_at')
            ->orderByDesc('id');

        // Optional filters
        if ($search = $request->string('search')->toString()) {
            $query->where(function ($q) use ($search) {
                $q->whereHas('clinicMedicationStock.medicationBatch.medication', function ($mq) use ($search) {
                    $mq->where('name', 'like', "%{$search}%");
                })->orWhereHas('nurseMedicationStock.medicationBatch.medication', function ($mq) use ($search) {
                    $mq->where('name', 'like', "%{$search}%");
                });
            });
        }

        if ($start = $request->date('start_date')) {
            $query->whereDate('dispensed_at', '>=', $start);
        }
        if ($end = $request->date('end_date')) {
            $query->whereDate('dispensed_at', '<=', $end);
        }

        $paginated = $query->paginate($perPage)->withQueryString();

        // Map to a UI-friendly shape
        $items = collect($paginated->items())->map(function (ChronicDispensation $d) {
            $origin = $d->nurse_medication_stock_id ? 'nurse' : 'clinic';

            // Get the actual Medication model (no optional() wrapper)
            $medModel =
            $d->clinicMedicationStock?->medicationBatch?->medication
                ?: $d->nurseMedicationStock?->medicationBatch?->medication;

            $batch =
            $d->clinicMedicationStock?->medicationBatch?->batch_number ?? $d->nurseMedicationStock?->medicationBatch?->batch_number;

            $clinicName =
            $d->clinicMedicationStock?->clinic?->name ?? $d->nurseMedicationStock?->clinic?->name;

            return [
                'id'           => $d->id,
                'dispensed_at' => optional($d->dispensed_at)?->toDateTimeString(),
                'origin'       => $origin, // 'clinic' | 'nurse'
                'clinic_name'  => $clinicName,
                'quantity'     => $d->quantity,
                'frequency'    => $d->frequency,
                'route'        => $d->route,
                'duration'     => $d->duration,
                'notes'        => $d->notes,
                'medication'   => $medModel ? [
                    'name'   => $medModel->name,
                    'dosage' => $medModel->dosage,
                    'form'   => $medModel->form,
                    'unit'   => $medModel->unit,
                ] : null,
                'batch_number' => $batch,
                'dispensed_by' => $d->dispensedBy ? [
                    'id'    => $d->dispensedBy->id,
                    'name'  => $d->dispensedBy->name,
                    'email' => $d->dispensedBy->email,
                ] : null,
                'show_url'     => route('chronics.dispensations.show', $d->id),
            ];
        });

        return Inertia::render('Chronics/Dispensations/PatientHistory', [
            'patient'  => [
                'id'              => $patient->id,
                'full_name'       => $patient->full_name ?? trim(($patient->first_name ?? '') . ' ' . ($patient->surname ?? '')),
                'employee_number' => $patient->employee_number,
                'gender'          => $patient->gender,
                'date_of_birth'   => $patient->date_of_birth,
            ],
            'timeline' => [
                'data' => $items,
                'meta' => [
                    'current_page' => $paginated->currentPage(),
                    'last_page'    => $paginated->lastPage(),
                    'per_page'     => $paginated->perPage(),
                    'total'        => $paginated->total(),
                ],
            ],
            'filters'  => [
                'search'     => $request->string('search')->toString() ?: null,
                'start_date' => $request->input('start_date'),
                'end_date'   => $request->input('end_date'),
                'per_page'   => $perPage,
            ],
        ]);
    }

    public function show(ChronicDispensation $dispensation)
    {
        $dispensation->load([
            'patient:id,first_name,middle_name,surname,employee_number,gender,date_of_birth',
            'dispensedBy:id,name,email',
            'clinicMedicationStock.clinic:id,name',
            'clinicMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
            'clinicMedicationStock.medicationBatch:id,batch_number,expiry_date',
            'nurseMedicationStock.clinic:id,name',
            'nurseMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
            'nurseMedicationStock.medicationBatch:id,batch_number,expiry_date',
        ]);

        $origin = $dispensation->nurse_medication_stock_id ? 'nurse' : 'clinic';

        $med = optional($dispensation->clinicMedicationStock?->medicationBatch?->medication)
            ?: optional($dispensation->nurseMedicationStock?->medicationBatch?->medication);

        $batch = $dispensation->clinicMedicationStock?->medicationBatch?->batch_number ?? $dispensation->nurseMedicationStock?->medicationBatch?->batch_number;

        $expiry = $dispensation->clinicMedicationStock?->medicationBatch?->expiry_date ?? $dispensation->nurseMedicationStock?->medicationBatch?->expiry_date;

        $clinicName = $dispensation->clinicMedicationStock?->clinic?->name ?? $dispensation->nurseMedicationStock?->clinic?->name;

        return Inertia::render('Chronics/Dispensations/Show', [
            'dispensation' => [
                'id'           => $dispensation->id,
                'patient'      => [
                    'id'          => $dispensation->patient->id,
                    'full_name'   => trim(($dispensation->patient->first_name ?? '') . ' ' . ($dispensation->patient->middle_name ?? '') . ' ' . ($dispensation->patient->surname ?? '')),
                    'gender'      => $dispensation->patient->gender,
                    'dob'         => $dispensation->patient->date_of_birth,
                    'employee_no' => $dispensation->patient->employee_number,
                ],
                'origin'       => $origin, // 'clinic' | 'nurse'
                'clinic_name'  => $clinicName,
                'quantity'     => $dispensation->quantity,
                'frequency'    => $dispensation->frequency,
                'route'        => $dispensation->route,
                'duration'     => $dispensation->duration, // string|null (matches your validation)
                'notes'        => $dispensation->notes,
                'dispensed_at' => optional($dispensation->dispensed_at)?->toDateTimeString(),
                'dispensed_by' => $dispensation->dispensedBy ? [
                    'id'    => $dispensation->dispensedBy->id,
                    'name'  => $dispensation->dispensedBy->name,
                    'email' => $dispensation->dispensedBy->email,
                ] : null,
                'medication'   => $med ? [
                    'name'   => $med->name,
                    'dosage' => $med->dosage,
                    'form'   => $med->form,
                    'unit'   => $med->unit,
                ] : null,
                'batch_number' => $batch,
                'expiry_date'  => $expiry,
            ],
            // Handy links for the page
            'links'        => [
                'patient_history' => route('chronics.patients.dispensations.history', $dispensation->patient_id),
                'patient_show'    => route('patients.show', $dispensation->patient_id),
            ],
        ]);
    }

    public function showForm(Patient $patient, Request $request)
    {
        // Basic guards (same style as your consult method)
        $user = Auth::user();
        if (! $user) {
            abort(403, 'Unauthorized.');
        }
        if (! $user->clinic_id) {
            abort(403, 'Your account is not assigned to a clinic.');
        }

        $search   = (string) $request->get('search', '');
        $clinicId = (int) $user->clinic_id;

        // Clinic stock for the user's primary clinic
        $clinicMedicationStocks = ClinicMedicationStock::with(['medicationBatch.medication'])
            ->where('clinic_id', $clinicId)
            ->where('quantity', '>', 0)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->orderByDesc('id')
            ->paginate(10)
            ->withQueryString();

        // Nurse stock for the signed-in nurse at their clinic
        $nurseMedicationStocks = NurseMedicationStock::with(['nurse', 'clinic', 'medicationBatch.medication'])
            ->where('nurse_id', Auth::id())
            ->where('clinic_id', $clinicId)
            ->where('quantity', '>', 0)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->orderByDesc('id')
            ->paginate(10)
            ->withQueryString();

        // Chronic dispensations (history) for this patient
        $dispensations = ChronicDispensation::with([
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy:id,name',
        ])
            ->where('patient_id', $patient->id)
            ->latest('dispensed_at')
            ->get();

        // Reuse your option sets
        $frequencyOptions = [
            'Stat'               => 'Stat',
            'As needed'          => 'As needed',
            'At bedtime'         => 'At bedtime',
            'Every morning'      => 'Every morning',
            'Every evening'      => 'Every evening',
            'Once daily'         => 'Once daily',
            'Twice daily'        => 'Twice daily',
            'Three times daily'  => 'Three times daily',
            'Four times daily'   => 'Four times daily',
            'Every 15 minutes'   => 'Every 15 minutes',
            'Every 30 minutes'   => 'Every 30 minutes',
            'Every hour'         => 'Every hour',
            'Every 2 hours'      => 'Every 2 hours',
            'Every 3 hours'      => 'Every 3 hours',
            'Every 4 hours'      => 'Every 4 hours',
            'Every 6 hours'      => 'Every 6 hours',
            'Every 8 hours'      => 'Every 8 hours',
            'Every 12 hours'     => 'Every 12 hours',
            'Every other day'    => 'Every other day',
            'Every 48 hours'     => 'Every 48 hours',
            'Every 72 hours'     => 'Every 72 hours',
            'Every 96 hours'     => 'Every 96 hours',
            'Twice weekly'       => 'Twice weekly',
            'Three times weekly' => 'Three times weekly',
            'Weekly'             => 'Weekly',
            'Monthly'            => 'Monthly',
        ];

        $routeOptions = [
            'Oral'          => 'Oral',
            'Sublingual'    => 'Sublingual',
            'Topical'       => 'Topical',
            'Inhalation'    => 'Inhalation',
            'Intramuscular' => 'Intramuscular',
            'Intravenous'   => 'Intravenous',
            'Subcutaneous'  => 'Subcutaneous',
            'Rectal'        => 'Rectal',
            'Vaginal'       => 'Vaginal',
            'Ophthalmic'    => 'Ophthalmic',
            'Otic'          => 'Otic',
            'Nasal'         => 'Nasal',
        ];

        // IMPORTANT: render a chronic-specific page so your TSX can expect `patient` instead of `consultation`
        return Inertia::render('Chronics/Medications/MedicationDispense', [
            'patient'                => $patient->load('company'),
            'clinicMedicationStocks' => $clinicMedicationStocks,
            'nurseMedicationStocks'  => $nurseMedicationStocks,
            'dispensations'          => $dispensations,
            'frequencyOptions'       => $frequencyOptions,
            'routeOptions'           => $routeOptions,
            'filters'                => ['search' => $search],
        ]);
    }

    public function index(Request $request)
    {
        $user = Auth::user();
        if (! $user || ! $user->clinic_id) {
            abort(403, 'Your account is not assigned to a clinic.');
        }

        $search    = $request->input('search');
        $patientId = $request->integer('patient_id');

        // Clinic medication stocks for the AUTHENTICATED USER'S clinic
        $clinicMedicationStocks = ClinicMedicationStock::with(['medicationBatch.medication'])
            ->where('clinic_id', $user->clinic_id)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->paginate(10)
            ->withQueryString();

        // Nurse medication stocks for the authenticated nurse in their clinic
        $nurseMedicationStocks = NurseMedicationStock::with(['nurse', 'clinic', 'medicationBatch.medication'])
            ->where('nurse_id', Auth::id())
            ->where('clinic_id', $user->clinic_id)
            ->where('quantity', '>', 0)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->paginate(10)
            ->withQueryString();

        // Existing chronic dispensations (optionally filtered by patient)
        $dispensations = ChronicDispensation::with([
            'patient:id,first_name,middle_name,surname',
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy:id,name',
        ])
            ->when($patientId, fn($q) => $q->where('patient_id', $patientId))
            ->latest('dispensed_at')
            ->latest()
            ->paginate(15)
            ->withQueryString();

        return Inertia::render('Chronics/MedicationDispense', [ // <- adjust to your actual view
            'clinicMedicationStocks' => $clinicMedicationStocks,
            'nurseMedicationStocks'  => $nurseMedicationStocks,
            'dispensations'          => $dispensations,
            'patients'               => Patient::select('id', 'first_name', 'surname')->orderBy('surname')->get(),
            'filters'                => $request->only('search', 'patient_id'),
        ]);
    }

    /**
     * Store ONE chronic dispensation (no consultation).
     */

    public function storeBatch(Patient $patient, Request $request)
    {
        $data = $request->validate([
            'items'              => 'required|array|min:1',
            'items.*.stock_type' => 'required|string|in:clinic,nurse',
            'items.*.stock_id'   => 'required|integer',
            'items.*.quantity'   => 'required|numeric|min:1',
            'items.*.frequency'  => 'nullable|string|max:255',
            'items.*.route'      => 'nullable|string|max:255',
            'items.*.duration'   => 'nullable|string|max:255',
            'items.*.notes'      => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            foreach ($data['items'] as $item) {
                if ($item['stock_type'] === 'clinic') {
                    /** @var ClinicMedicationStock $stock */
                    $stock = ClinicMedicationStock::whereKey($item['stock_id'])
                        ->lockForUpdate()
                        ->firstOrFail();
                    $stockRelationKey = 'clinic_medication_stock_id';
                } else {
                    /** @var NurseMedicationStock $stock */
                    $stock = NurseMedicationStock::whereKey($item['stock_id'])
                        ->lockForUpdate()
                        ->firstOrFail();
                    $stockRelationKey = 'nurse_medication_stock_id';
                }

                if ($stock->quantity < $item['quantity']) {
                    throw ValidationException::withMessages([
                        'items' => ["Only {$stock->quantity} items available for stock ID {$stock->id}"],
                    ]);
                }

                // Upsert by patient + stock
                $where = [
                    'patient_id'      => $patient->id,
                    $stockRelationKey => $stock->id,
                ];

                /** @var ChronicDispensation|null $disp */
                $disp      = ChronicDispensation::where($where)->lockForUpdate()->latest('dispensed_at')->first();
                $oldValues = $disp?->getOriginal();

                if ($disp) {
                    $disp->quantity     = (float) $disp->quantity + (float) $item['quantity'];
                    $disp->frequency    = $item['frequency'] ?? $disp->frequency;
                    $disp->route        = $item['route'] ?? $disp->route;
                    $disp->duration     = $item['duration'] ?? $disp->duration;
                    $disp->notes        = $item['notes'] ?? $disp->notes;
                    $disp->dispensed_by = Auth::id();
                    $disp->dispensed_at = now();
                    $disp->save();
                } else {
                    $disp = ChronicDispensation::create(array_merge($where, [
                        'quantity'     => $item['quantity'],
                        'frequency'    => $item['frequency'] ?? null,
                        'route'        => $item['route'] ?? null,
                        'duration'     => $item['duration'] ?? null,
                        'notes'        => $item['notes'] ?? null,
                        'dispensed_by' => Auth::id(),
                        'dispensed_at' => now(),
                    ]));
                }

                // Decrement the stock
                $stock->decrement('quantity', $item['quantity']);

                // Optional logging
                \App\Models\Log::create([
                    'user_id'     => Auth::id(),
                    'user_name'   => Auth::user()->name ?? null,
                    'action'      => $oldValues ? 'updated' : 'created',
                    'description' => "Chronic dispensation ID {$disp->id} (batch) for patient " . ($patient->full_name ?? "{$patient->first_name} {$patient->surname}"),
                    'loggable_type' => ChronicDispensation::class,
                    'loggable_id' => $disp->id,
                    'old_values' => $oldValues,
                    'new_values' => $disp->toArray(),
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->header('User-Agent'),
                ]);
            }

            DB::commit();
            return back()->with('success', 'Chronic medications dispensed successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return back()->withErrors(['error' => 'Failed to dispense chronic medications. Please try again.']);
        }
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'stock_type' => 'required|string|in:clinic,nurse',
            'stock_id'   => 'required|integer',
            'quantity'   => 'required|integer|min:1',
            'patient_id' => 'required|exists:patients,id',
            'frequency'  => 'nullable|string|max:255',
            'route'      => 'nullable|string|max:255',
            'duration'   => 'nullable|string|max:255', // string to allow "30 days", "3 months"
            'notes'      => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // 1) Lock the stock row so only one request decrements at a time
            if ($validated['stock_type'] === 'clinic') {
                /** @var ClinicMedicationStock $stock */
                $stock = ClinicMedicationStock::whereKey($validated['stock_id'])
                    ->lockForUpdate()
                    ->firstOrFail();
                $stockRelationKey = 'clinic_medication_stock_id';
            } else {
                /** @var NurseMedicationStock $stock */
                $stock = NurseMedicationStock::whereKey($validated['stock_id'])
                    ->lockForUpdate()
                    ->firstOrFail();
                $stockRelationKey = 'nurse_medication_stock_id';
            }

            if ($stock->quantity < $validated['quantity']) {
                throw ValidationException::withMessages([
                    'quantity' => "Only {$stock->quantity} items available",
                ]);
            }

            // 2) Upsert behavior for chronic: group by patient + stock
            $where = [
                'patient_id'      => $validated['patient_id'],
                $stockRelationKey => $stock->id,
            ];

            $disp      = ChronicDispensation::where($where)->lockForUpdate()->latest('dispensed_at')->first();
            $oldValues = $disp?->getOriginal();

            if ($disp) {
                // Update + increment in one locked scope
                $disp->quantity     = (float) $disp->quantity + (float) $validated['quantity'];
                $disp->frequency    = $validated['frequency'] ?? $disp->frequency;
                $disp->route        = $validated['route'] ?? $disp->route;
                $disp->duration     = $validated['duration'] ?? $disp->duration;
                $disp->notes        = $validated['notes'] ?? $disp->notes;
                $disp->dispensed_by = Auth::id();
                $disp->dispensed_at = now();
                $disp->save();
            } else {
                $disp = ChronicDispensation::create([
                    'patient_id'      => $validated['patient_id'],
                    $stockRelationKey => $stock->id,
                    'quantity'        => $validated['quantity'],
                    'frequency'       => $validated['frequency'] ?? null,
                    'route'           => $validated['route'] ?? null,
                    'duration'        => $validated['duration'] ?? null,
                    'notes'           => $validated['notes'] ?? null,
                    'dispensed_by'    => Auth::id(),
                    'dispensed_at'    => now(),
                ]);
            }

            // 3) Decrement stock
            $stock->decrement('quantity', $validated['quantity']);

            // 4) Logging
            $patient     = Patient::find($validated['patient_id']);
            $patientName = $patient ? ($patient->first_name . ' ' . $patient->surname) : 'Unknown Patient';

            \App\Models\Log::create([
                'user_id'     => Auth::id(),
                'user_name'   => Auth::user()->name ?? null,
                'action'      => $oldValues ? 'updated' : 'created',
                'description' => "Chronic dispensation ID {$disp->id} for patient {$patientName}",
                'loggable_type' => ChronicDispensation::class,
                'loggable_id'   => $disp->id,
                'old_values'    => $oldValues,
                'new_values'    => $disp->toArray(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();
            return back()->with('success', 'Chronic medication dispensed successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return back()->withErrors(['error' => 'Failed to dispense chronic medication. Please try again.']);
        }
    }

    /**
     * Store MANY chronic dispensations in one go.
     */

    /**
     * Paginated “All Chronic Dispensations” (with filters).
     */
    public function allDispensations(Request $request)
    {
        $filters = $request->only('search', 'start_date', 'end_date', 'patient_id', 'dispensed_by');

        $dispensations = ChronicDispensation::with([
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy:id,name',
            'patient:id,first_name,surname',
        ])
            ->when($filters['search'] ?? null, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->whereHas('clinicMedicationStock.medicationBatch.medication', function ($q2) use ($search) {
                        $q2->where('name', 'like', "%{$search}%");
                    })
                        ->orWhereHas('nurseMedicationStock.medicationBatch.medication', function ($q2) use ($search) {
                            $q2->where('name', 'like', "%{$search}%");
                        })
                        ->orWhereHas('patient', function ($q2) use ($search) {
                            $q2->where('first_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%");
                        });
                });
            })
            ->when($filters['start_date'] ?? null, fn($q, $date) => $q->whereDate('dispensed_at', '>=', $date))
            ->when($filters['end_date'] ?? null, fn($q, $date) => $q->whereDate('dispensed_at', '<=', $date))
            ->when($filters['patient_id'] ?? null, fn($q, $patientId) => $q->where('patient_id', $patientId))
            ->when($filters['dispensed_by'] ?? null, fn($q, $userId) => $q->where('dispensed_by', $userId))
            ->latest('dispensed_at')
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('Chronics/Dispensations/AllDispensations', [ // <- adjust to your actual view
            'dispensations' => $dispensations,
            'patients'      => Patient::select('id', 'first_name', 'surname')->get(),
            'users'         => User::select('id', 'name')->get(),
            'filters'       => $filters,
        ]);
    }

    /**
     * Export chronic dispensations (JSON payload).
     */
    public function export(Request $request)
    {
        try {
            $filters = $request->validate([
                'search'       => 'nullable|string',
                'start_date'   => 'nullable|date',
                'end_date'     => 'nullable|date|after_or_equal:start_date',
                'patient_id'   => 'nullable|integer|exists:patients,id',
                'dispensed_by' => 'nullable|integer|exists:users,id',
            ]);

            $query = ChronicDispensation::with([
                'patient',
                'clinicMedicationStock.medicationBatch.medication',
                'nurseMedicationStock.medicationBatch.medication',
                'dispensedBy',
            ])
                ->when($filters['search'] ?? false, function ($query, $search) {
                    $query->where(function ($q) use ($search) {
                        $q->whereHas('patient', function ($q) use ($search) {
                            $q->where('first_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%");
                        })
                            ->orWhereHas('clinicMedicationStock.medicationBatch.medication', function ($q) use ($search) {
                                $q->where('name', 'like', "%{$search}%");
                            })
                            ->orWhereHas('nurseMedicationStock.medicationBatch.medication', function ($q) use ($search) {
                                $q->where('name', 'like', "%{$search}%");
                            });
                    });
                })
                ->when($filters['start_date'] ?? false, fn($q, $date) => $q->whereDate('dispensed_at', '>=', $date))
                ->when($filters['end_date'] ?? false, fn($q, $date) => $q->whereDate('dispensed_at', '<=', $date))
                ->when($filters['patient_id'] ?? false, fn($q, $patientId) => $q->where('patient_id', $patientId))
                ->when($filters['dispensed_by'] ?? false, fn($q, $userId) => $q->where('dispensed_by', $userId))
                ->orderBy('dispensed_at', 'desc');

            $dispensations = $query->get();

            if ($dispensations->isEmpty()) {
                return response()->json([
                    'message' => 'No chronic dispensations found matching your criteria',
                ], 404);
            }

            $exportData = $dispensations->map(function ($disp) {
                $medicationName =
                $disp->clinicMedicationStock->medicationBatch->medication->name ?? $disp->nurseMedicationStock->medicationBatch->medication->name ?? 'N/A';

                return [
                    'Date'               => optional($disp->dispensed_at)->format('Y-m-d H:i:s'),
                    'Patient First Name' => $disp->patient->first_name ?? 'N/A',
                    'Patient Surname'    => $disp->patient->surname ?? 'N/A',
                    'Medication'         => $medicationName,
                    'Quantity'           => $disp->quantity,
                    'Dosage/Route'       => $disp->route,
                    'Frequency'          => $disp->frequency,
                    'Duration'           => $disp->duration,
                    'Dispensed By'       => $disp->dispensedBy->name ?? 'N/A',
                    'Notes'              => $disp->notes ?? '',
                ];
            });

            return response()->json($exportData);

        } catch (\Throwable $e) {
            return response()->json([
                'message' => 'Failed to generate export: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete one chronic dispensation.
     */
    public function destroy(ChronicDispensation $chronicDispensation)
    {
        $chronicDispensation->delete();
        return back()->with('success', 'Deleted chronic dispensation.');
    }
}
