<?php

namespace App\Http\Controllers\Chronics;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\{Patient, Company, Consultation};

class ChronicsPatientsController extends Controller
{
    public function index(Request $request)
    {
        $search        = (string) $request->input('search', '');
        $gender        = $request->input('gender');
        $companyId     = $request->input('company_id');
        $maritalStatus = $request->input('marital_status');
        $ageMin        = $request->input('age_min');
        $ageMax        = $request->input('age_max');

        // ── Normalize search forms (handled case/spacing/hyphens) ──────────────
        $searchTrim  = trim(preg_replace('/\s+/u', ' ', $search));
        $normIdEq    = preg_replace('/[\s-]+/', '', mb_strtolower($searchTrim)); // id_number normalized
        $normEmpEq   = preg_replace('/[\s-]+/', '', mb_strtoupper($searchTrim)); // employee_number normalized
        $normIdLike  = '%' . $normIdEq . '%';
        $normEmpLike = '%' . $normEmpEq . '%';
        $tokens      = array_values(array_filter(explode(' ', $searchTrim), fn ($t) => $t !== ''));

        $patients = Patient::query()
            ->with(['company:id,name'])
            ->addSelect('patients.*')
            ->withCount('dependents')
            ->addSelect([
                'latest_consultation_id' => Consultation::select('id')
                    ->whereColumn('consultations.patient_id', 'patients.id')
                    ->orderByDesc('consultation_date')
                    ->limit(1),
            ])
            ->whereNull('parent_patient_id')
            // ⬇️ Force chronic only
            ->where('is_chronic_patient', 1)

            // ── Robust search (same logic as Patients index) ───────────────────
            ->when($searchTrim !== '', function ($query) use ($searchTrim, $tokens, $normIdEq, $normEmpEq, $normIdLike, $normEmpLike) {
                $query->where(function ($q) use ($searchTrim, $tokens, $normIdEq, $normEmpEq, $normIdLike, $normEmpLike) {
                    $q->whereRaw("LOWER(REPLACE(REPLACE(id_number, ' ', ''), '-', '')) = ?", [$normIdEq])
                      ->orWhereRaw("UPPER(REPLACE(REPLACE(employee_number, ' ', ''), '-', '')) = ?", [$normEmpEq])
                      ->orWhere(function ($qq) use ($tokens) {
                          if (count($tokens) >= 2) {
                              [$a, $b, $c] = [$tokens[0] ?? null, $tokens[1] ?? null, $tokens[2] ?? null];
                              $qq->where(function ($qqq) use ($a, $b, $c) {
                                      $qqq->where('first_name', 'like', "%{$a}%")
                                          ->when($c,
                                              fn($qqqq) => $qqqq->where('surname', 'like', "%{$c}%"),
                                              fn($qqqq) => $qqqq->where('surname', 'like', "%{$b}%")
                                          );
                                  })
                                  ->orWhere(function ($qqq) use ($a, $b, $c) {
                                      $qqq->where('surname', 'like', "%{$a}%")
                                          ->when($c,
                                              fn($qqqq) => $qqqq->where('first_name', 'like', "%{$c}%"),
                                              fn($qqqq) => $qqqq->where('first_name', 'like', "%{$b}%")
                                          );
                                  })
                                  ->orWhereRaw(
                                      "CONCAT_WS(' ', first_name, middle_name, surname) LIKE ?",
                                      ['%' . implode(' ', array_filter([$a, $b, $c])) . '%']
                                  );
                          }
                      })
                      ->orWhere('first_name', 'like', "%{$searchTrim}%")
                      ->orWhere('middle_name', 'like', "%{$searchTrim}%")
                      ->orWhere('surname', 'like', "%{$searchTrim}%")
                      ->orWhere('employee_number', 'like', "%{$searchTrim}%")
                      ->orWhere('id_number', 'like', "%{$searchTrim}%")
                      ->orWhereRaw("CONCAT_WS(' ', first_name, middle_name, surname) LIKE ?", ["%{$searchTrim}%"])
                      ->orWhereRaw("LOWER(REPLACE(REPLACE(id_number, ' ', ''), '-', '')) LIKE ?", [$normIdLike])
                      ->orWhereRaw("UPPER(REPLACE(REPLACE(employee_number, ' ', ''), '-', '')) LIKE ?", [$normEmpLike]);
                });
            })

            // ── Other filters (same as Patients index) ─────────────────────────
            ->when($gender, fn($q) => $q->where('gender', $gender))
            ->when($companyId, fn($q) => $q->where('company_id', $companyId))
            ->when($maritalStatus, fn($q) => $q->where('marital_status', $maritalStatus))
            ->when(($ageMin || $ageMax), function ($q) use ($ageMin, $ageMax) {
                $now = Carbon::now();
                if ($ageMin) {
                    $q->whereDate('date_of_birth', '<=', $now->copy()->subYears($ageMin));
                }
                if ($ageMax) {
                    $q->whereDate('date_of_birth', '>=', $now->copy()->subYears($ageMax + 1)->addDay());
                }
            })

            ->orderByDesc('updated_at')
            ->orderByDesc('created_at')
            ->paginate(15)
            ->withQueryString();

        // ── Stats (scoped to chronic patients, employees-only like your original) ─
        $employeesTotalChronic  = Patient::whereNull('parent_patient_id')->where('is_chronic_patient', 1)->count();
        $dependentsTotalChronic = Patient::whereNotNull('parent_patient_id')->where('is_chronic_patient', 1)->count();
        $chronicTotal           = $employeesTotalChronic; // mirrors your original meaning

        $genderCounts = Patient::whereNull('parent_patient_id')
            ->where('is_chronic_patient', 1)
            ->select('gender', DB::raw('COUNT(*) as count'))
            ->groupBy('gender')
            ->pluck('count', 'gender');

        $male    = (int) ($genderCounts['Male'] ?? 0);
        $female  = (int) ($genderCounts['Female'] ?? 0);
        $unknown = max(0, $employeesTotalChronic - ($male + $female));

        return inertia('Chronics/Patients/Index', [
            'patients'  => $patients,
            // Include the same filters object; force chronic=true for UI consistency
            'filters'   => array_merge(
                $request->only('search', 'gender', 'company_id', 'marital_status', 'age_min', 'age_max'),
                ['is_chronic_patient' => true]
            ),
            'companies' => Company::select('id', 'name')->get(),
            'stats'     => [
                'employees_total'  => $employeesTotalChronic,
                'dependents_total' => $dependentsTotalChronic,
                'chronic_total'    => $chronicTotal,
                'gender'           => [
                    'male'    => $male,
                    'female'  => $female,
                    'unknown' => $unknown,
                ],
            ],
        ]);
    }
}
