<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Clinic;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ClinicAccessController extends Controller
{
    /**
     * Show the UI to edit a user's clinic access.
     */
    public function edit(User $user)
    {
        $clinics = Clinic::orderBy('name')->get(['id','name','city']);

        // Use the relationship instead of querying the pivot model
        $selectedClinicIds = $user->accessibleClinics()->pluck('clinics.id')->all();

        return inertia('Users/ClinicAccess', [ // <-- match your TSX component name
            'user'              => $user->only(['id','name','email','clinic_id','role']),
            'clinics'           => $clinics,
            'selectedClinicIds' => $selectedClinicIds,
            'primaryClinicId'   => $user->clinic_id,
        ]);
    }

    /**
     * Sync the set of accessible clinics for a user.
     * - Validates clinic IDs exist
     * - Ensures primary clinic (if any) stays included
     */
    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'clinic_ids'   => ['array'],
            'clinic_ids.*' => ['integer', Rule::exists('clinics','id')],
        ]);

        $ids = collect($data['clinic_ids'] ?? [])
            ->map(fn ($id) => (int) $id)
            ->unique()
            ->values();

        // Always include the user's primary clinic (if set)
        if (!empty($user->clinic_id)) {
            $ids = $ids->push((int) $user->clinic_id)->unique()->values();
        }

        $user->accessibleClinics()->sync($ids->all());

        // Inertia expects a redirect; 303 is normal. back() is fine.
        return back()->with('success', 'Clinic access updated successfully.');
    }

    /**
     * Grant a single clinic (AJAX-friendly).
     */
    public function store(Request $request, User $user)
    {
        $validated = $request->validate([
            'clinic_id' => ['required','integer', Rule::exists('clinics','id')],
        ]);

        $user->accessibleClinics()->syncWithoutDetaching([$validated['clinic_id']]);

        return back()->with('success', 'Access granted.');
    }

    /**
     * Revoke a single clinic (AJAX-friendly).
     * Won’t allow removing primary clinic.
     */
    public function destroy(User $user, Clinic $clinic)
    {
        if ($user->clinic_id && (int) $user->clinic_id === (int) $clinic->id) {
            return back()->with('error', 'Cannot revoke access to user’s primary clinic.');
        }

        $user->accessibleClinics()->detach($clinic->id);

        return back()->with('success', 'Access revoked.');
    }
}
