<?php
namespace App\Http\Controllers;

use App\Models\Clinic;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ClinicController extends Controller
{
    public function index(Request $request)
    {
        $clinics = Clinic::query()
            ->when($request->input('search'), function ($query, $search) {
                $query->where('name', 'like', "%{$search}%")
                    ->orWhere('city', 'like', "%{$search}%")
                    ->orWhere('manager', 'like', "%{$search}%");
            })
            ->paginate(10)
            ->withQueryString(); // Keep query params when paginating

        return Inertia::render('Clinics/Index', [
            'clinics' => $clinics,
            'filters' => $request->only('search'),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Clinics/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'           => 'required|string|max:255',
            'address'        => 'nullable|string|max:255',
            'city'           => 'nullable|string|max:100',
            'afhoz_number'   => 'nullable|string|max:50',
            'contact_person' => 'nullable|string|max:100',
            'manager'        => 'nullable|string|max:100',
            'country'        => 'nullable|string|max:100',
            'email_address'  => 'nullable|email|max:255',
        ]);

        Clinic::create($validated);

        return redirect()->route('clinics.index')->with('success', 'Clinic created successfully.');
    }

    /**
     * Display the specified resource.
     */
    // app/Http/Controllers/ClinicController.php

    public function show(Clinic $clinic)
    {
        // Don't loadCount('patients') since patients.clinic_id doesn't exist.
        $clinic->loadCount([
            'users',
            'triages',
            'consultations',
            'dispensations',
            'medicationRequests',
            'medicationStocks',
        ]);

        // ---- Unique patient count via triages + consultations (distinct patient_id)
        $triagePatientIds = $clinic->triages()
            ->whereNotNull('patient_id')
            ->pluck('patient_id')
            ->unique();

        $consultPatientIds = $clinic->consultations()
            ->whereNotNull('patient_id')
            ->pluck('patient_id')
            ->unique();

        $uniquePatientIds    = $triagePatientIds->merge($consultPatientIds)->unique();
        $uniquePatientsCount = $uniquePatientIds->count();

        // expose as an attribute on the model so frontend can read clinic.unique_patients_count
        $clinic->setAttribute('unique_patients_count', $uniquePatientsCount);

        // ---- Recent staff
        $recentUsers = $clinic->users()
            ->select('id', 'name', 'email', 'role', 'created_at')
            ->latest()
            ->limit(5)
            ->get();

        // ---- Recent patients: derive from latest triages, unique by patient
        $triagesForRecentPatients = $clinic->triages()
            ->with(['patient:id,first_name,surname'])
            ->whereNotNull('patient_id')
            ->latest()
            ->get();

        $recentPatients = collect();
        foreach ($triagesForRecentPatients as $t) {
            if (! $t->patient) {
                continue;
            }
            $pid = $t->patient->id;
            if (! $recentPatients->has($pid)) {
                $recentPatients->put($pid, [
                    'id'        => $pid,
                    'full_name' => trim(($t->patient->first_name ?? '') . ' ' . ($t->patient->surname ?? '')),
                    'seen_at'   => $t->created_at, // when this clinic last triaged them
                ]);
            }
            if ($recentPatients->count() >= 5) {
                break;
            }
        }
        $recentPatients = $recentPatients->values();

        // ---- Recent triages (with patient & consultor)
        $recentTriages = $clinic->triages()
            ->with([
                'patient:id,first_name,surname',
                'consultor:id,name',
            ])
            ->latest()
            ->limit(5)
            ->get()
            ->map(function ($t) {
                return [
                    'id'           => $t->id,
                    'triage_level' => $t->triage_level ?? null,
                    'is_active'    => (bool) ($t->is_active ?? false),
                    'created_at'   => $t->created_at,
                    'patient'      => $t->patient ? [
                        'id'        => $t->patient->id,
                        'full_name' => trim(($t->patient->first_name ?? '') . ' ' . ($t->patient->surname ?? '')),
                    ] : null,
                    'consultor'    => $t->consultor ? [
                        'id'   => $t->consultor->id,
                        'name' => $t->consultor->name,
                    ] : null,
                ];
            });

        // ---- Recent consultations (with patient & doctor)
        $recentConsultations = $clinic->consultations()
            ->with([
                'patient:id,first_name,surname',
                'doctor:id,name',
            ])
            ->latest('consultation_date')
            ->limit(5)
            ->get()
            ->map(function ($c) {
                return [
                    'id'                => $c->id,
                    'consultation_date' => $c->consultation_date,
                    'diagnosis'         => $c->diagnosis,
                    'patient'           => $c->patient ? [
                        'id'        => $c->patient->id,
                        'full_name' => trim(($c->patient->first_name ?? '') . ' ' . ($c->patient->surname ?? '')),
                    ] : null,
                    'doctor'            => $c->doctor ? [
                        'id'   => $c->doctor->id,
                        'name' => $c->doctor->name,
                    ] : null,
                ];
            });

        // ---- Recent medication requests
        $recentMedicationRequests = $clinic->medicationRequests()
            ->with(['user:id,name'])
            ->latest()
            ->limit(5)
            ->get()
            ->map(function ($r) {
                return [
                    'id'           => $r->id,
                    'status'       => $r->status ?? null,
                    'created_at'   => $r->created_at,
                    'requested_by' => $r->user ? [
                        'id'   => $r->user->id,
                        'name' => $r->user->name,
                    ] : null,
                ];
            });

        // ---- Recent stock updates
        $recentStocks = $clinic->medicationStocks()
            ->with(['medicationBatch.medication:id,name']) // 👈 key change
            ->latest()
            ->limit(5)
            ->get()
            ->map(function ($s) {
                $med = optional($s->medicationBatch)->medication;

                return [
                    'id'         => $s->id,
                    'medication' => $med ? [
                        'id'   => $med->id,
                        'name' => $med->name,
                    ] : null,
                    'quantity'   => $s->quantity ?? null,
                    'updated_at' => $s->updated_at,
                ];
            });

        return Inertia::render('Clinics/Show', [
            'clinic'                     => $clinic, // includes unique_patients_count attribute
            'recent_users'               => $recentUsers,
            'recent_patients'            => $recentPatients,
            'recent_triages'             => $recentTriages,
            'recent_consultations'       => $recentConsultations,
            'recent_medication_requests' => $recentMedicationRequests,
            'recent_stocks'              => $recentStocks,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Clinic $clinic)
    {
        return Inertia::render('Clinics/Edit', [
            'clinic' => $clinic,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Clinic $clinic)
    {
        $validated = $request->validate([
            'name'           => 'required|string|max:255',
            'address'        => 'nullable|string|max:255',
            'city'           => 'nullable|string|max:100',
            'afhoz_number'   => 'nullable|string|max:50',
            'contact_person' => 'nullable|string|max:100',
            'manager'        => 'nullable|string|max:100',
            'country'        => 'nullable|string|max:100',
            'email_address'  => 'nullable|email|max:255',
        ]);

        $clinic->update($validated);

        return redirect()->route('clinics.index')->with('success', 'Clinic updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Clinic $clinic)
    {
        $clinic->delete();

        return redirect()->route('clinics.index')->with('success', 'Clinic deleted successfully.');
    }
}
