<?php
namespace App\Http\Controllers;

use App\Models\Clinic;
use App\Models\ClinicMedicationStock;
use App\Models\Log;
use App\Models\StockTransfer;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ClinicMedicationStockController extends Controller
{

    public function index(Clinic $clinic)
    {
        $stocks = ClinicMedicationStock::with('medicationBatch.medication')
            ->where('clinic_id', $clinic->id)
            ->when(request('search'), function ($query, $search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
        // 👇 Highest quantity first, newest as tiebreaker
            ->orderByDesc('quantity')
            ->orderByDesc('clinic_medication_stocks.created_at')
            ->paginate(10)
            ->withQueryString();

        // Users (nurses) at this clinic
        $usersQuery = $clinic->users();
        if ($searchUser = request('search_user')) {
            $usersQuery->where(function ($q) use ($searchUser) {
                $q->where('name', 'like', "%{$searchUser}%")
                    ->orWhere('email', 'like', "%{$searchUser}%");
            });
        }
        $users = $usersQuery->get();

        return Inertia::render('Clinics/MedicationStocks/Index', [
            'clinic'  => $clinic,
            'stocks'  => $stocks,
            'users'   => $users,
            'filters' => request()->only(['search', 'search_user']),
        ]);
    }

    public function store(Request $request, Clinic $clinic)
    {
        $validated = $request->validate([
            'medication_id' => 'required|exists:medications,id',
            'quantity'      => 'required|integer|min:1',
            'expiry_date'   => 'nullable|date',
            'batch_number'  => 'nullable|string|max:255',
        ]);

        $validated['clinic_id'] = $clinic->id;

        // Wrap in DB transaction to ensure consistency with logging
        DB::transaction(function () use ($validated, $request) {
            $stock = ClinicMedicationStock::create($validated);

            // Create log entry for the new stock
            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'created',
                'loggable_type' => ClinicMedicationStock::class,
                'loggable_id'   => $stock->id,
                'description'   => 'Created new clinic medication stock',
                'old_values'    => null,
                'new_values'    => $stock->getAttributes(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->userAgent(),
            ]);
        });

        return redirect()->back()->with('success', 'Medication stock added successfully');
    }

    public function pendingStockTransfers(Request $request, Clinic $clinic)
    {
        $query = StockTransfer::where('to_clinic_id', $clinic->id)
            ->where('status', 'pending')
            ->with('medicationBatch.medication')
            ->orderBy('created_at', 'desc'); // Sort by latest first

        // Add search filter for medication name
        if ($request->has('search') && ! empty($request->search)) {
            $searchTerm = $request->search;
            $query->whereHas('medicationBatch.medication', function ($q) use ($searchTerm) {
                $q->where('name', 'like', '%' . $searchTerm . '%');
            });
        }

        $pendingTransfers = $query->paginate(10);

        return inertia('Clinics/MedicationStocks/PendingStockTransfers', [
            'clinic'  => $clinic,
            'stocks'  => $pendingTransfers,
            'filters' => $request->only(['search']),
        ]);
    }

    public function countPendingStockTransfers(Clinic $clinic)
    {
        $count = StockTransfer::where('to_clinic_id', $clinic->id)
            ->where('status', 'pending')
            ->count();

        return response()->json(['pending_count' => $count]);
    }

    public function updateClinicMedQuantity(Request $request, ClinicMedicationStock $stock)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:0',
        ]);

        // Store old values for logging
        $oldValues = $stock->getOriginal();

        // Update stock
        $stock->update($validated);

        // Create log entry
        Log::create([
            'user_id'       => Auth::id(),
            'user_name'     => Auth::user()->name ?? null,
            'action'        => 'updated',
            'loggable_type' => ClinicMedicationStock::class,
            'loggable_id'   => $stock->id,
            'description'   => 'Updated clinic medication stock quantity',
            'old_values'    => $oldValues,
            'new_values'    => $stock->getAttributes(),
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->userAgent(),
        ]);

        return response()->json([
            'message' => 'Stock updated successfully',
            'stock'   => $stock,
        ], 200);
    }

    public function destroy(ClinicMedicationStock $stock)
    {
        $stock->delete();
        return redirect()->back()->with('success', 'Stock entry deleted');
    }
}
