<?php

namespace App\Http\Controllers;

use App\Models\Clinic;
use App\Models\ClinicMedicationStock;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class ClinicReportsController extends Controller
{
    public function clinicMedicationStockTake(Request $request, Clinic $clinic)
    {

        $search      = $request->query('search');       // medication name / batch number
        $expiryFrom  = $request->query('expiry_from');  // YYYY-MM-DD
        $expiryTo    = $request->query('expiry_to');    // YYYY-MM-DD
        $includeZero = $request->boolean('include_zero', false);

        $query = ClinicMedicationStock::query()
            ->with([
                'clinic:id,name,address,city,email_address',
                'medicationBatch.medication:id,name,dosage,form,unit',
            ])
            ->where('clinic_id', $clinic->id)
            ->when(!$includeZero, fn ($q) => $q->where('quantity', '>', 0))
            ->when($search, function ($q) use ($search) {
                $q->whereHas('medicationBatch.medication', fn ($m) =>
                    $m->where('name', 'like', "%{$search}%")
                )->orWhereHas('medicationBatch', fn ($b) =>
                    $b->where('batch_number', 'like', "%{$search}%")
                );
            })
            ->when($expiryFrom || $expiryTo, function ($q) use ($expiryFrom, $expiryTo) {
                if ($expiryFrom && $expiryTo) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereBetween('expiry_date', [$expiryFrom, $expiryTo])
                    );
                } elseif ($expiryFrom) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '>=', $expiryFrom)
                    );
                } else {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '<=', $expiryTo)
                    );
                }
            });

        $stocks = $query->get()
            ->sortBy(fn ($s) => sprintf(
                '%s|%s',
                mb_strtolower(optional(optional($s->medicationBatch)->medication)->name ?? ''),
                optional($s->medicationBatch)->expiry_date ?? '9999-12-31'
            ))
            ->values();

        $generatedAt = now()->format('Y-m-d H:i:s');

        $filters = [];
        if ($search)     $filters[] = "Search: {$search}";
        if ($expiryFrom) $filters[] = "Expiry From: {$expiryFrom}";
        if ($expiryTo)   $filters[] = "Expiry To: {$expiryTo}";
        if ($includeZero)$filters[] = "Include Zero: yes";
        $filtersLine = implode(' | ', $filters);

        $clinicLine = trim($clinic->name . ($clinic->city ? " ({$clinic->city})" : ''));

        $fileName = sprintf(
            'Clinic_Stock_Take_%s_%s.xlsx',
            preg_replace('/\s+/', '_', $clinic->name ?? 'Clinic'),
            now()->format('Ymd_His')
        );

        return response()->streamDownload(function () use ($stocks, $clinicLine, $generatedAt, $filtersLine) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // Sheet name
            $sheetName = $this->safeSheetName('Clinic Stock Take');
            $writer->getCurrentSheet()->setName($sheetName);

            // Header
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['CLINIC MEDICATION STOCK TAKE'],
                ['Generated At', $generatedAt],
                ['Clinic', $clinicLine],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);

            // Table header
            $headers = [
                'Medication',
                'Dosage',
                'Form',
                'Unit',
                'Batch #',
                'Expiry Date',
                'Manufacture Date',
                'Received Date',
                'Quantity',
            ];
            $writer->addRow(Row::fromValues($headers, $headerStyle));

            // Column widths (optional)
            $opts = $writer->getOptions();
            $opts->setColumnWidth(36, 1); // Medication
            $opts->setColumnWidth(14, 5); // Batch #
            $opts->setColumnWidth(14, 6); // Expiry

            // Rows
            $totalQty = 0;
            foreach ($stocks as $s) {
                $batch = $s->medicationBatch;
                $med   = optional($batch)->medication;

                $row = [
                    $med->name ?? '',
                    $med->dosage ?? '',
                    $med->form ?? '',
                    $med->unit ?? '',
                    $batch->batch_number ?? '',
                    optional($batch->expiry_date)->format('Y-m-d') ?? (is_string($batch->expiry_date ?? null) ? $batch->expiry_date : ''),
                    optional($batch->manufacture_date)->format('Y-m-d') ?? (is_string($batch->manufacture_date ?? null) ? $batch->manufacture_date : ''),
                    optional($batch->received_date)->format('Y-m-d') ?? (is_string($batch->received_date ?? null) ? $batch->received_date : ''),
                    (int) $s->quantity,
                ];

                $totalQty += (int) $s->quantity;
                $writer->addRow(Row::fromValues($row, $wrapStyle));
            }

            // Footer totals
            $writer->addRow(Row::fromValues([]));
            $writer->addRow(Row::fromValues(['Total Lines', $stocks->count()]));
            $writer->addRow(Row::fromValues(['Total Quantity', $totalQty]));

            // Summary by medication
            if ($stocks->isNotEmpty()) {
                $writer->addRow(Row::fromValues([]));
                $writer->addRow(Row::fromValues(['Medication Summary'], $titleStyle));
                $writer->addRow(Row::fromValues(['Medication', 'Total Quantity'], $headerStyle));

                $byMed = $stocks
                    ->groupBy(fn ($s) => optional(optional($s->medicationBatch)->medication)->name ?? '(Unknown)')
                    ->map(fn ($group) => $group->sum('quantity'))
                    ->sortKeys();

                foreach ($byMed as $medName => $qty) {
                    $writer->addRow(Row::fromValues([$medName, (int) $qty]));
                }
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /* ---------- helpers ---------- */

    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    private function writeSheetHeader(XLSXWriter $writer, Style $titleStyle, Style $labelStyle, array $lines): void
    {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
