<?php

namespace App\Http\Controllers;

use App\Models\ClinicMedicationStock;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class ClinicStockTakeReportController extends Controller
{
    /**
     * Export a stock-take Excel for ALL clinics.
     * Filters:
     *  - search: medication name or batch number (LIKE)
     *  - expiry_from / expiry_to: YYYY-MM-DD
     *  - include_zero (bool): include zero-qty lines (default false)
     *
     * Sheets:
     *  - "All Clinics — Summary" (totals per clinic)
     *  - One sheet per clinic with full table and per-medication summary.
     */
    public function allClinicsStockTake(Request $request)
    {
        $search      = $request->query('search');       // medication name / batch number
        $expiryFrom  = $request->query('expiry_from');  // YYYY-MM-DD
        $expiryTo    = $request->query('expiry_to');    // YYYY-MM-DD
        $includeZero = $request->boolean('include_zero', false);

        $query = ClinicMedicationStock::query()
            ->with([
                'clinic:id,name,address,city,email_address',
                'medicationBatch.medication:id,name,dosage,form,unit',
                'medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
            ])
            ->when(!$includeZero, fn ($q) => $q->where('quantity', '>', 0))
            ->when($search, function ($q) use ($search) {
                $q->whereHas('medicationBatch.medication', fn ($m) =>
                    $m->where('name', 'like', "%{$search}%")
                )->orWhereHas('medicationBatch', fn ($b) =>
                    $b->where('batch_number', 'like', "%{$search}%")
                );
            })
            ->when($expiryFrom || $expiryTo, function ($q) use ($expiryFrom, $expiryTo) {
                if ($expiryFrom && $expiryTo) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereBetween('expiry_date', [$expiryFrom, $expiryTo])
                    );
                } elseif ($expiryFrom) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '>=', $expiryFrom)
                    );
                } else {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '<=', $expiryTo)
                    );
                }
            });

        $stocks = $query->get()
            ->sortBy(fn ($s) => sprintf(
                '%s|%s|%s',
                mb_strtolower(optional($s->clinic)->name ?? ''),
                mb_strtolower(optional(optional($s->medicationBatch)->medication)->name ?? ''),
                optional($s->medicationBatch)->expiry_date ?? '9999-12-31'
            ))
            ->values();

        // Group by clinic label "Name (City)"
        $byClinic = $stocks->groupBy(function ($s) {
            $c = $s->clinic;
            $name = (string)($c->name ?? 'Unknown Clinic');
            $city = (string)($c->city ?? '');
            return trim($name . ($city ? " ({$city})" : ''));
        })->sortKeys();

        $generatedAt = now()->format('Y-m-d H:i:s');

        $filters = [];
        if ($search)      $filters[] = "Search: {$search}";
        if ($expiryFrom)  $filters[] = "Expiry From: {$expiryFrom}";
        if ($expiryTo)    $filters[] = "Expiry To: {$expiryTo}";
        if ($includeZero) $filters[] = "Include Zero: yes";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Clinic_Stock_Take_All_Clinics_%s.xlsx', now()->format('Ymd_His'));

        return response()->streamDownload(function () use ($byClinic, $generatedAt, $filtersLine) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            /* ===== Overview sheet ===== */
            $writer->getCurrentSheet()->setName($this->safeSheetName('All Clinics — Summary'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['CLINIC MEDICATION STOCK TAKE — ALL CLINICS'],
                ['Generated At', $generatedAt],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);

            // Overview table headers
            $overviewHeaders = ['Clinic', 'Total Lines', 'Total Quantity'];
            $writer->addRow(Row::fromValues($overviewHeaders, $headerStyle));

            $totalLinesAll = 0;
            $totalQtyAll   = 0;

            foreach ($byClinic as $clinicLabel => $dataset) {
                $lines = $dataset->count();
                $qty   = (int) $dataset->sum('quantity');
                $totalLinesAll += $lines;
                $totalQtyAll   += $qty;

                $writer->addRow(Row::fromValues([
                    $clinicLabel,
                    $lines,
                    $qty,
                ]));
            }

            // Overview totals
            $writer->addRow(Row::fromValues([]));
            $writer->addRow(Row::fromValues(['ALL CLINICS LINES', $totalLinesAll]));
            $writer->addRow(Row::fromValues(['ALL CLINICS QUANTITY', $totalQtyAll]));

            /* ===== One sheet per clinic ===== */
            $used = ['All Clinics — Summary' => true];
            $first = false; // already used the first sheet

            foreach ($byClinic as $clinicLabel => $dataset) {
                $writer->addNewSheetAndMakeItCurrent();
                $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($clinicLabel));
                $writer->getCurrentSheet()->setName($sheetName);

                // Header
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['CLINIC MEDICATION STOCK TAKE'],
                    ['Generated At', $generatedAt],
                    ['Clinic', $clinicLabel],
                ]);

                // Table header (same as your single-clinic report)
                $headers = [
                    'Medication',
                    'Dosage',
                    'Form',
                    'Unit',
                    'Batch #',
                    'Expiry Date',
                    'Manufacture Date',
                    'Received Date',
                    'Quantity',
                ];
                $writer->addRow(Row::fromValues($headers, $headerStyle));

                // Column widths
                $opts = $writer->getOptions();
                $opts->setColumnWidth(36, 1); // Medication
                $opts->setColumnWidth(14, 5); // Batch #
                $opts->setColumnWidth(14, 6); // Expiry

                // Rows
                $totalQty = 0;
                foreach ($dataset as $s) {
                    $batch = $s->medicationBatch;
                    $med   = optional($batch)->medication;

                    $row = [
                        $med->name ?? '',
                        $med->dosage ?? '',
                        $med->form ?? '',
                        $med->unit ?? '',
                        $batch->batch_number ?? '',
                        optional($batch->expiry_date)->format('Y-m-d') ?? (is_string($batch->expiry_date ?? null) ? $batch->expiry_date : ''),
                        optional($batch->manufacture_date)->format('Y-m-d') ?? (is_string($batch->manufacture_date ?? null) ? $batch->manufacture_date : ''),
                        optional($batch->received_date)->format('Y-m-d') ?? (is_string($batch->received_date ?? null) ? $batch->received_date : ''),
                        (int) $s->quantity,
                    ];

                    $totalQty += (int) $s->quantity;
                    $writer->addRow(Row::fromValues($row, $wrapStyle));
                }

                // Footer totals
                $writer->addRow(Row::fromValues([]));
                $writer->addRow(Row::fromValues(['Total Lines', $dataset->count()]));
                $writer->addRow(Row::fromValues(['Total Quantity', $totalQty]));

                // Medication summary
                if ($dataset->isNotEmpty()) {
                    $writer->addRow(Row::fromValues([]));
                    $writer->addRow(Row::fromValues(['Medication Summary'], $titleStyle));
                    $writer->addRow(Row::fromValues(['Medication', 'Total Quantity'], $headerStyle));

                    $byMed = $dataset
                        ->groupBy(fn ($s) => optional(optional($s->medicationBatch)->medication)->name ?? '(Unknown)')
                        ->map(fn ($group) => $group->sum('quantity'))
                        ->sortKeys();

                    foreach ($byMed as $medName => $qty) {
                        $writer->addRow(Row::fromValues([$medName, (int) $qty]));
                    }
                }
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /* ---------- helpers ---------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /** Simple header block helper (matches your other reports). */
    private function writeSheetHeader(XLSXWriter $writer, Style $titleStyle, Style $labelStyle, array $lines): void
    {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
