<?php
namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Patient;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CompanyController extends Controller
{
    // app/Http/Controllers/CompanyController.php

    public function index(Request $request)
    {
        $companies = Company::query()
            ->when(
                $request->search,
                fn($q, $search) =>
                $q->where('name', 'like', "%$search%")
                    ->orWhere('company_email', 'like', "%$search%")
                    ->orWhere('contact_person', 'like', "%$search%")
            )
            ->addSelect([
                // employees + dependents (via parent company)
                'people_count' => \App\Models\Patient::query()
                    ->selectRaw('COUNT(*)')
                    ->where(function ($qq) {
                        $qq->whereColumn('company_id', 'companies.id')
                            ->orWhereHas('parent', function ($q2) {
                                $q2->whereColumn('company_id', 'companies.id');
                            });
                    }),
            ])
            ->orderBy('name')
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('Companies/Index', [
            'companies' => $companies,
            'filters'   => $request->only('search'),
        ]);
    }

    public function create()
    {
        return Inertia::render('Companies/Create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'           => 'required|string|max:255',
            'address'        => 'required|string|max:255',
            'site_telephone' => 'required|string|max:255',
            'company_email'  => 'required|email',
            'contact_person' => 'nullable|string|max:255',
            'province'       => 'nullable|string|max:255',
            'designation'    => 'nullable|string|max:255',
            'contact_number' => 'nullable|string|max:255',
            'is_private'     => 'nullable|boolean',
            'next'           => 'nullable|string',
        ]);

        $validated['is_private'] = (bool) ($validated['is_private'] ?? false);

        $originalEmail    = strtolower($validated['company_email']);
        [$local, $domain] = array_pad(explode('@', $originalEmail, 2), 2, '');
        $uniqueEmail      = $originalEmail;

        if ($domain !== '') {
            $suffix = 1;
            while (Company::where('company_email', $uniqueEmail)->exists()) {
                $uniqueEmail = "{$local}{$suffix}@{$domain}";
                $suffix++;
            }
        }
        $validated['company_email'] = $uniqueEmail;

        $created  = null;
        $attempts = 0;
        do {
            try {
                $payload = collect($validated)->except('next')->all();
                $created = Company::create($payload);
            } catch (\Illuminate\Database\QueryException $e) {
                if (str_contains($e->getMessage(), 'unique') || $e->getCode() === '23000') {
                    if ($domain !== '') {
                        if (preg_match('/^(.+?)(\d+)$/', $local, $m)) {
                            $local = $m[1] . ((int) $m[2] + 1);
                        } else {
                            $local = rtrim($local) . '1';
                        }
                        $uniqueEmail                = "{$local}@{$domain}";
                        $validated['company_email'] = $uniqueEmail;
                        $attempts++;
                        continue;
                    }
                }
                throw $e;
            }
            break;
        } while ($attempts < 5);

        $next  = $request->string('next')->toString();
        $route = $next === 'patients.create' ? 'patients.create' : 'companies.index';

        $msg = 'Company created successfully.';
        if ($uniqueEmail !== $originalEmail) {
            $msg .= " Note: email already existed, saved as {$uniqueEmail}.";
        }

        return redirect()->route($route)->with('success', $msg);
    }

    public function edit(Company $company)
    {
        return Inertia::render('Companies/Edit', [
            'company' => $company,
        ]);
    }

    public function update(Request $request, Company $company)
    {
        $validated = $request->validate([
            'name'           => 'required|string|max:255',
            'address'        => 'required|string|max:255',
            'site_telephone' => 'required|string|max:255',
            'company_email'  => 'required|email|unique:companies,company_email,' . $company->id,
            'contact_person' => 'nullable|string|max:255',
            'province'       => 'nullable|string|max:255',
            'designation'    => 'nullable|string|max:255',
            'contact_number' => 'nullable|string|max:255',
            'is_private'     => 'nullable|boolean',
        ]);

        $validated['is_private'] = (bool) ($validated['is_private'] ?? false);

        $company->update($validated);

        return redirect()->route('companies.index')->with('success', 'Company updated successfully.');
    }

    public function destroy(Company $company)
    {
        $company->delete();

        return redirect()->route('companies.index')->with('success', 'Company deleted successfully.');
    }

    /** NEW: Company show page with male/female summary + lists */
    public function show(Company $company)
    {
        // Employees: direct patients at this company (parent null)
        $employees = Patient::query()
            ->where('company_id', $company->id)
            ->whereNull('parent_patient_id')
            ->select('id', 'first_name', 'middle_name', 'surname', 'gender', 'id_number', 'phone', 'date_of_birth', 'employee_number', 'company_id')
            ->orderBy('surname')->orderBy('first_name')
            ->get();

        // Dependents linked *via parent* whose company_id is this company
        $dependents = Patient::query()
            ->whereNotNull('parent_patient_id')
            ->whereHas('parent', function ($q) use ($company) {
                $q->where('company_id', $company->id);
            })
            ->with(['parent:id,first_name,surname,company_id'])
            ->select('id', 'first_name', 'middle_name', 'surname', 'gender', 'relationship', 'parent_patient_id', 'id_number', 'phone', 'date_of_birth')
            ->orderBy('surname')->orderBy('first_name')
            ->get();

        // Male/Female counts across both sets
        $maleCount   = $employees->where('gender', 'Male')->count() + $dependents->where('gender', 'Male')->count();
        $femaleCount = $employees->where('gender', 'Female')->count() + $dependents->where('gender', 'Female')->count();
        $otherCount  = $employees->where('gender', 'Other')->count() + $dependents->where('gender', 'Other')->count();

        return Inertia::render('Companies/Show', [
            'company'    => $company->only(['id', 'name', 'address', 'company_email', 'site_telephone', 'contact_person', 'province', 'designation', 'contact_number', 'is_private']),
            'stats'      => [
                'employees'  => $employees->count(),
                'dependents' => $dependents->count(),
                'total'      => $employees->count() + $dependents->count(),
                'gender'     => [
                    'male'   => $maleCount,
                    'female' => $femaleCount,
                    'other'  => $otherCount,
                ],
            ],
            'employees'  => $employees,
            'dependents' => $dependents,
        ]);
    }
}
