<?php
namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Clinic;
use App\Models\ClinicMedicationStock;
use App\Models\Consultation;
use App\Models\DiagnosisPreset;
use App\Models\Dispensation;
use App\Models\Log;
use App\Models\NurseMedicationStock;
use App\Models\Patient;
use App\Models\Triage;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Inertia\Inertia;

class ConsultationController extends Controller
{
    /**
     * Helper: list of company IDs this user is allowed to see (via user_company_accesses).
     * If you created a relation $user->accessibleCompanies(), we’ll use that first.
     */
    private function allowedCompanyIdsFor(User $user): array
    {
        if (method_exists($user, 'accessibleCompanies')) {
            return $user->accessibleCompanies()
                ->pluck('companies.id')->map(fn($v) => (int) $v)->all();
        }

        // Fallback to pivot table if no relation defined
        return DB::table('user_company_accesses')
            ->where('user_id', $user->id)
            ->pluck('company_id')
            ->map(fn($v) => (int) $v)
            ->all();
    }

    /**
     * NEW view policy:
     *  - If the user is a Doctor or Nurse → can view ANY consultation.
     *  - Otherwise (Admin/SuperAdmin/etc.) → can view if:
     *      • they are the owner (consultation.doctor_id == user.id), OR
     *      • they have access to the patient's company (same logic as before).
     */
    private function canUserViewConsultation(User $user, Consultation $consultation): bool
    {
        $role = $user->role?->value ?? $user->role;

        // Doctors & Nurses may view any consultation
        if (in_array($role, [UserRole::Doctor->value, UserRole::Nurse->value], true)) {
            return true;
        }

        // Owner via doctor_id (creator of the consultation)
        if ((int) $consultation->doctor_id === (int) $user->id) {
            return true;
        }

                                             // Company-based access for other roles
        $patient   = $consultation->patient; // ensure loaded
        $companyId = optional($patient)->company_id;
        if (! $companyId) {
            return false;
        }

        $allowed = $this->allowedCompanyIdsFor($user);
        return in_array((int) $companyId, $allowed, true);
    }

    /**
     * Edit policy:
     *  - ONLY the user who created the consultation (stored in doctor_id) can edit.
     *    (Whether the creator is a doctor or a nurse.)
     */
    private function canUserEditConsultation(User $user, Consultation $consultation): bool
    {
        return (int) $consultation->doctor_id === (int) $user->id;
    }

    public function index(Request $request)
    {
        $patientId    = $request->input('patient_id');
        $search       = $request->input('search');
        $startDate    = $request->input('start_date');
        $endDate      = $request->input('end_date');
        $injuryOnDuty = $request->input('injury_on_duty');

        $referToRadiology  = $request->input('refer_for_radiology');
        $referToLab        = $request->input('refer_for_lab');
        $referToSpecialist = $request->input('refer_to_specialist');
        $referToCasualty   = $request->input('refer_to_casualty');
        $referOutPatient   = $request->input('refer_out_patient');

        $followUpRequired = $request->input('follow_up_required');
        $clinicId         = $request->input('clinic_id');
        $doctorId         = $request->input('doctor_id');

        if ($patientId && ! Patient::find($patientId)) {
            abort(404, 'Patient not found');
        }

        $user = Auth::user();

        $consultations = Consultation::with(['clinic', 'doctor', 'patient.company'])
            ->when($patientId, fn($query) => $query->where('patient_id', $patientId))
            ->when($search, function ($query, $search) {
                $query->whereHas('patient', function ($q) use ($search) {
                    $q->where('first_name', 'like', "%{$search}%")
                        ->orWhere('middle_name', 'like', "%{$search}%")
                        ->orWhere('surname', 'like', "%{$search}%");
                });
            })
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                $query->whereBetween('consultation_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($injuryOnDuty !== null, fn($q) => $q->where('injury_on_duty', (bool) $injuryOnDuty))
            ->when($referToRadiology !== null, fn($q) => $q->where('refer_for_radiology', (bool) $referToRadiology))
            ->when($referToLab !== null, fn($q) => $q->where('refer_for_lab', (bool) $referToLab))
            ->when($referToSpecialist !== null, fn($q) => $q->where('refer_to_specialist', (bool) $referToSpecialist))
            ->when($referToCasualty !== null, fn($q) => $q->where('refer_to_casualty', (bool) $referToCasualty))
            ->when($referOutPatient !== null, fn($q) => $q->where('refer_out_patient', (bool) $referOutPatient))
            ->when($followUpRequired !== null, fn($q) => $q->where('follow_up_required', (bool) $followUpRequired))
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->when($doctorId, fn($q) => $q->where('doctor_id', $doctorId))
            ->latest('created_at')
            ->paginate(10)
            ->appends($request->all());

        $clinics = Clinic::select('id', 'name')->orderBy('name')->get();

        $roles = [
            UserRole::Doctor->value,
            UserRole::Nurse->value,
            UserRole::SuperAdmin->value,
            UserRole::Admin->value,
        ];

        $doctors = User::whereIn('role', $roles)
            ->select('id', 'name')
            ->orderBy('name')
            ->get();

        // Map to lightweight payload for listing (kept as in your code)
        $consultations->getCollection()->transform(function ($c) use ($user) {
            /** @var \App\Models\Consultation $c */
            return [
                'id'                => $c->id,
                'consultation_date' => $c->consultation_date ? Carbon::parse($c->consultation_date)->format('Y-m-d') : null,
                'diagnosis'         => $c->diagnosis,
                'created_at'        => $c->created_at,
                'injury_on_duty'    => (bool) $c->injury_on_duty,
                'clinic'            => ['id' => optional($c->clinic)->id, 'name' => optional($c->clinic)->name],
                'doctor'            => ['id' => optional($c->doctor)->id, 'name' => optional($c->doctor)->name],
                'patient'           => [
                    'id'         => optional($c->patient)->id,
                    'first_name' => optional($c->patient)->first_name,
                    'surname'    => optional($c->patient)->surname,
                    'company'    => [
                        'id'   => optional(optional($c->patient)->company)->id,
                        'name' => optional(optional($c->patient)->company)->name,
                    ],
                ],
            ];
        });

        /**
         * Company breakdown analytics kept as-is (unchanged logic).
         */
        $role    = $user->role?->value ?? $user->role;
        $bypass  = in_array($role, [UserRole::Doctor->value], true); // doctors used to bypass here
        $allowed = $this->allowedCompanyIdsFor($user);

        $companyBreakdown = Consultation::query()
            ->join('patients as p', 'p.id', '=', 'consultations.patient_id')
            ->leftJoin('patients as parent', 'parent.id', '=', 'p.parent_patient_id')
            ->leftJoin('companies', 'companies.id', '=', DB::raw('COALESCE(p.company_id, parent.company_id)'))
        // filters mirrored
            ->when($patientId, fn($q) => $q->where('consultations.patient_id', $patientId))
            ->when($search, function ($q, $search) {
                $q->where(function ($q1) use ($search) {
                    $q1->where('p.first_name', 'like', "%{$search}%")
                        ->orWhere('p.middle_name', 'like', "%{$search}%")
                        ->orWhere('p.surname', 'like', "%{$search}%");
                });
            })
            ->when($startDate && $endDate, function ($q) use ($startDate, $endDate) {
                $q->whereBetween('consultations.consultation_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($injuryOnDuty !== null, fn($q) => $q->where('consultations.injury_on_duty', (bool) $injuryOnDuty))
            ->when($referToRadiology !== null, fn($q) => $q->where('consultations.refer_for_radiology', (bool) $referToRadiology))
            ->when($referToLab !== null, fn($q) => $q->where('consultations.refer_for_lab', (bool) $referToLab))
            ->when($referToSpecialist !== null, fn($q) => $q->where('consultations.refer_to_specialist', (bool) $referToSpecialist))
            ->when($referToCasualty !== null, fn($q) => $q->where('consultations.refer_to_casualty', (bool) $referToCasualty))
            ->when($referOutPatient !== null, fn($q) => $q->where('consultations.refer_out_patient', (bool) $referOutPatient))
            ->when($followUpRequired !== null, fn($q) => $q->where('consultations.follow_up_required', (bool) $followUpRequired))
            ->when($clinicId, fn($q) => $q->where('consultations.clinic_id', $clinicId))
            ->when($doctorId, fn($q) => $q->where('consultations.doctor_id', $doctorId))
        // privacy scoping for non-bypass
            ->when(! $bypass, function ($q) use ($allowed, $user) {
                $q->where(function ($q1) use ($allowed, $user) {
                    if (! empty($allowed)) {
                        $q1->whereIn(DB::raw('COALESCE(p.company_id, parent.company_id)'), $allowed);
                    }
                    // also include "owned" files via doctor_id
                    $q1->orWhere('consultations.doctor_id', $user->id);
                });
            })
            ->selectRaw('COALESCE(companies.name, "Unknown") as company_name, COUNT(*) as total')
            ->groupBy('company_name')
            ->orderByDesc('total')
            ->get()
            ->map(fn($row) => [
                'company_name' => $row->company_name,
                'total'        => (int) $row->total,
            ]);

        return Inertia::render('Consultations/ConsultationIndex', [
            'consultations'     => $consultations,
            'clinics'           => $clinics,
            'doctors'           => $doctors,
            'company_breakdown' => $companyBreakdown,
            'filters'           => $request->only([
                'search',
                'patient_id',
                'start_date',
                'end_date',
                'injury_on_duty',
                'refer_for_radiology',
                'refer_for_lab',
                'refer_to_specialist',
                'refer_to_casualty',
                'refer_out_patient',
                'follow_up_required',
                'clinic_id',
                'doctor_id',
            ]),
        ]);
    }

    public function patientConsultations(Request $request, Patient $patient)
    {
        $search    = $request->input('search');
        $startDate = $request->input('start_date');
        $endDate   = $request->input('end_date');
        $clinicId  = $request->input('clinic_id');
        $doctorId  = $request->input('doctor_id');

        $user = Auth::user();

        $consultations = Consultation::with(['clinic', 'doctor', 'patient.company'])
            ->where('patient_id', $patient->id)
            ->when($search, function ($query, $search) {
                $query->whereHas('patient', function ($q) use ($search) {
                    $q->where('first_name', 'like', "%{$search}%")
                        ->orWhere('surname', 'like', "%{$search}%");
                });
            })
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                $query->whereBetween('consultation_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->when($doctorId, fn($q) => $q->where('doctor_id', $doctorId))
            ->latest('consultation_date')
            ->paginate(10)
            ->appends($request->all());

        // Attach can_view & can_edit reflecting the new policies
        $consultations->getCollection()->transform(function ($c) use ($user) {
            /** @var Consultation $c */
            return [
                'id'                => $c->id,
                'consultation_date' => $c->consultation_date ? Carbon::parse($c->consultation_date)->format('Y-m-d') : null,
                'diagnosis'         => $c->diagnosis,
                'clinic'            => ['id' => optional($c->clinic)->id, 'name' => optional($c->clinic)->name],
                'doctor'            => ['id' => optional($c->doctor)->id, 'name' => optional($c->doctor)->name],
                'patient'           => [
                    'id'         => optional($c->patient)->id,
                    'first_name' => optional($c->patient)->first_name,
                    'surname'    => optional($c->patient)->surname,
                    'company'    => [
                        'id'   => optional(optional($c->patient)->company)->id,
                        'name' => optional(optional($c->patient)->company)->name,
                    ],
                ],
                'can_view'          => $this->canUserViewConsultation($user, $c),
                'can_edit'          => $this->canUserEditConsultation($user, $c),
            ];
        });

        $clinics = Clinic::select('id', 'name')->orderBy('name')->get();
        $doctors = User::where('role', UserRole::Doctor)
            ->select('id', 'name')
            ->orderBy('name')
            ->get();

        return Inertia::render('Consultations/PatientConsultations', [
            'consultations' => $consultations,
            'patient'       => $patient->only(['id', 'first_name', 'surname']),
            'clinics'       => $clinics,
            'doctors'       => $doctors,
            'filters'       => $request->only([
                'search',
                'start_date',
                'end_date',
                'clinic_id',
                'doctor_id',
            ]),
        ]);
    }

    public function showDispenseForm(Consultation $consultation, Request $request)
    {
        if (! $consultation) {
            abort(404, 'Consultation not found');
        }

        $user = Auth::user();
        if (! $user || ! $user->clinic_id) {
            abort(403, 'Your account is not assigned to a clinic.');
        }

        $search = $request->get('search');

        $clinicMedicationStocks = ClinicMedicationStock::with(['medicationBatch.medication'])
            ->where('clinic_id', $user->clinic_id)
            ->where('quantity', '>', 0)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->get();

        $dispensations = Dispensation::with([
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy',
        ])
            ->where('consultation_id', $consultation->id)
            ->latest()
            ->get();

        $clinicId = $user->clinic_id;

        $nurseMedicationStocks = NurseMedicationStock::with(['nurse', 'clinic', 'medicationBatch.medication'])
            ->where('nurse_id', Auth::id())
            ->where('clinic_id', $clinicId)
            ->where('quantity', '>', 0)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->get();

        $frequencyOptions = [
            'Stat'               => 'Stat',
            'As needed'          => 'As needed',
            'At bedtime'         => 'At bedtime',
            'Every morning'      => 'Every morning',
            'Every evening'      => 'Every evening',
            'Once daily'         => 'Once daily',
            'Twice daily'        => 'Twice daily',
            'Three times daily'  => 'Three times daily',
            'Four times daily'   => 'Four times daily',
            'Every 15 minutes'   => 'Every 15 minutes',
            'Every 30 minutes'   => 'Every 30 minutes',
            'Every hour'         => 'Every hour',
            'Every 2 hours'      => 'Every 2 hours',
            'Every 3 hours'      => 'Every 3 hours',
            'Every 4 hours'      => 'Every 4 hours',
            'Every 6 hours'      => 'Every 6 hours',
            'Every 8 hours'      => 'Every 8 hours',
            'Every 12 hours'     => 'Every 12 hours',
            'Every other day'    => 'Every other day',
            'Every 48 hours'     => 'Every 48 hours',
            'Every 72 hours'     => 'Every 72 hours',
            'Every 96 hours'     => 'Every 96 hours',
            'Twice weekly'       => 'Twice weekly',
            'Three times weekly' => 'Three times weekly',
            'Weekly'             => 'Weekly',
            'Monthly'            => 'Monthly',
        ];

        $routeOptions = [
            'Oral'          => 'Oral',
            'Sublingual'    => 'Sublingual',
            'Topical'       => 'Topical',
            'Inhalation'    => 'Inhalation',
            'Intramuscular' => 'Intramuscular',
            'Intravenous'   => 'Intravenous',
            'Subcutaneous'  => 'Subcutaneous',
            'Rectal'        => 'Rectal',
            'Vaginal'       => 'Vaginal',
            'Ophthalmic'    => 'Ophthalmic',
            'Otic'          => 'Otic',
            'Nasal'         => 'Nasal',
        ];

        return Inertia::render('Medications/MedicationDispense', [
            'consultation'           => $consultation->load('patient', 'clinic', 'doctor'),
            'clinicMedicationStocks' => $clinicMedicationStocks,
            'dispensations'          => $dispensations,
            'nurseMedicationStocks'  => $nurseMedicationStocks,
            'frequencyOptions'       => $frequencyOptions,
            'routeOptions'           => $routeOptions,
            'filters'                => ['search' => $search],
        ]);
    }

    public function transferForm(Request $request, Consultation $consultation)
    {
        $q = trim((string) $request->input('q', ''));

        $patients = Patient::query()
            ->select('id', 'first_name', 'surname', 'parent_patient_id')
            ->when($q !== '', function ($query) use ($q) {
                $query->where(function ($q1) use ($q) {
                    $q1->where('first_name', 'like', "%{$q}%")
                        ->orWhere('surname', 'like', "%{$q}%")
                        ->orWhere('id', $q)
                        ->orWhere('id_number', 'like', "%{$q}%");
                });
            })
            ->orderBy('first_name')
            ->limit(25)
            ->get()
            ->map(fn($p) => [
                'id'           => $p->id,
                'full_name'    => trim($p->first_name . ' ' . $p->surname),
                'is_dependent' => (bool) $p->parent_patient_id,
            ]);

        return Inertia::render('Consultations/ConsultationTransfer', [
            'consultation' => [
                'id'      => $consultation->id,
                'patient' => [
                    'id'        => $consultation->patient->id,
                    'full_name' => trim($consultation->patient->first_name . ' ' . $consultation->patient->surname),
                ],
            ],
            'patients'     => $patients,
            'filters'      => ['q' => $q !== '' ? $q : null],
        ]);
    }

    public function transferPatient(Request $request, Consultation $consultation)
    {
        $data = $request->validate([
            'to_patient_id' => ['required', 'exists:patients,id'],
            'note'          => ['nullable', 'string', 'max:1000'],
        ]);

        $fromPatientId = $consultation->patient_id;
        $toPatientId   = (int) $data['to_patient_id'];

        if ($fromPatientId === $toPatientId) {
            return back()->withErrors('The target patient is the same as the current patient.');
        }

        $fromPatient = Patient::find($fromPatientId);
        $toPatient   = Patient::find($toPatientId);

        DB::transaction(function () use ($consultation, $fromPatientId, $toPatientId) {
            $consultation->update(['patient_id' => $toPatientId]);

            Dispensation::where('consultation_id', $consultation->id)
                ->update(['patient_id' => $toPatientId]);

            Triage::where('consultation_id', $consultation->id)
                ->update(['patient_id' => $toPatientId]);
        });

        try {
            Log::create([
                'user_id'     => Auth::id(),
                'user_name'   => Auth::user()->name ?? null,
                'action'      => 'transferred',
                'description' => "Transferred consultation #{$consultation->id} from patient #{$fromPatientId} to patient #{$toPatientId}",
                'loggable_type' => Consultation::class,
                'loggable_id'   => $consultation->id,
                'old_values'    => ['patient_id' => $fromPatientId],
                'new_values'    => ['patient_id' => $toPatientId],
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);
        } catch (\Throwable $e) {
            // ignore logging failures
        }

        return redirect()
            ->route('consultations.show', $consultation->id)
            ->with('success', "Consultation transferred to {$toPatient->full_name} successfully.");
    }

    public function editReferrals(Consultation $consultation)
    {
        $consultation->load([
            'patient.company',
            'doctor',
            'clinic',
            'doctorsnote.doctor',
        ]);

        return Inertia::render('Consultations/ConsultationReferralsForm', [
            'consultation' => [
                'id'                      => $consultation->id,
                'patient_id'              => $consultation->patient_id,
                'doctor_id'               => $consultation->doctor_id,
                'clinic_id'               => $consultation->clinic_id,
                'consultation_date'       => $consultation->consultation_date
                    ? Carbon::parse($consultation->consultation_date)->format('Y-m-d')
                    : null,
                'injury_on_duty'          => (bool) $consultation->injury_on_duty,

                // Vitals
                'systolic'                => $consultation->systolic,
                'diastolic'               => $consultation->diastolic,
                'temperature'             => $consultation->temperature,
                'heart_rate'              => $consultation->heart_rate,
                'respiratory_rate'        => $consultation->respiratory_rate,
                'oxygen_saturation'       => $consultation->oxygen_saturation,
                'weight'                  => $consultation->weight,
                'height'                  => $consultation->height,
                'bmi'                     => $consultation->bmi,

                // Labs / Tests
                'blood_sugar'             => $consultation->blood_sugar,
                'blood_sugar_unit'        => $consultation->blood_sugar_unit,
                'blood_sugar_context'     => $consultation->blood_sugar_context,
                'blood_sugar_measured_at' => $consultation->blood_sugar_measured_at,
                'urine_test_result'       => $consultation->urine_test_result,
                'hiv_status'              => $consultation->hiv_status,
                'pregnancy_test_result'   => $consultation->pregnancy_test_result,

                // Notes
                'presenting_complaint'    => $consultation->presenting_complaint,
                'clinical_notes'          => $consultation->clinical_notes,
                'diagnosis'               => $consultation->diagnosis,
                'treatment_plan'          => $consultation->treatment_plan,
                'prescription'            => $consultation->prescription,

                // Follow-up / Referrals (focus of this screen)
                'follow_up_required'      => (bool) $consultation->follow_up_required,
                'follow_up_date'          => $consultation->follow_up_date
                    ? Carbon::parse($consultation->follow_up_date)->format('Y-m-d')
                    : null,
                'referral_notes'          => $consultation->referral_notes,

                'refer_for_radiology'     => (bool) $consultation->refer_for_radiology,
                'refer_for_lab'           => (bool) $consultation->refer_for_lab,
                'refer_to_specialist'     => (bool) $consultation->refer_to_specialist,
                'specialist_name'         => $consultation->specialist_name,
                'refer_to_casualty'       => (bool) $consultation->refer_to_casualty,
                'refer_out_patient'       => (bool) $consultation->refer_out_patient,

                // NEW fields
                'restricted_duty'         => (bool) $consultation->restricted_duty,
                'referred_hospital'       => $consultation->referred_hospital,

                // Sick leave
                'sick_leave'              => (bool) $consultation->sick_leave,
                'reason_for_leave'        => $consultation->reason_for_leave,
                'sick_leave_notes'        => $consultation->sick_leave_notes,
                'number_of_days'          => $consultation->number_of_days,

                // Stamps
                'created_at'              => $consultation->created_at,
                'updated_at'              => $consultation->updated_at,

                'patient'                 => [
                    'id'              => $consultation->patient->id,
                    'full_name'       => trim($consultation->patient->first_name . ' ' . $consultation->patient->surname),
                    'first_name'      => $consultation->patient->first_name,
                    'surname'         => $consultation->patient->surname,
                    'employee_number' => $consultation->patient->employee_number,
                    'home_address'    => $consultation->patient->home_address,
                    'date_of_birth'   => $consultation->patient->date_of_birth,
                    'email'           => $consultation->patient->email,
                    'gender'          => $consultation->patient->gender,
                    'phone'           => $consultation->patient->phone,
                    'id_number'       => $consultation->patient->id_number,
                ],
                'doctor'                  => [
                    'name'           => optional($consultation->doctor)->name,
                    'email'          => optional($consultation->doctor)->email,
                    'signature'      => optional($consultation->doctor)->signature,
                    'qualifications' => optional($consultation->doctor)->qualifications,
                ],
                'clinic'                  => [
                    'name'           => optional($consultation->clinic)->name,
                    'address'        => optional($consultation->clinic)->address,
                    'contact_person' => optional($consultation->clinic)->contact_person,
                    'email_address'  => optional($consultation->clinic)->email_address,
                    'city'           => optional($consultation->clinic)->city,
                ],

                'doctors_note'            => $consultation->doctorsnote ? [
                    'id'         => $consultation->doctorsnote->id,
                    'title'      => $consultation->doctorsnote->title,
                    'body'       => $consultation->doctorsnote->body,
                    'note_type'  => $consultation->doctorsnote->note_type,
                    'private'    => (bool) $consultation->doctorsnote->private,
                    'created_at' => optional($consultation->doctorsnote->created_at)->toIso8601String(),
                    'doctor'     => [
                        'id'   => $consultation->doctorsnote->doctor_id,
                        'name' => optional($consultation->doctorsnote->doctor)->name,
                    ],
                ] : null,
            ],
        ]);
    }

    public function updateReferrals(Request $request, Consultation $consultation)
    {
        $data = $request->validate([
            // Follow-up
            'follow_up_required'  => 'required|boolean',
            'follow_up_date'      => 'nullable|date',
            'referral_notes'      => 'nullable|string',

            // Referrals
            'refer_for_radiology' => 'required|boolean',
            'refer_for_lab'       => 'required|boolean',
            'refer_to_specialist' => 'required|boolean',
            'specialist_name'     => 'nullable|string',
            'refer_to_casualty'   => 'required|boolean',
            'refer_out_patient'   => 'required|boolean',

            // NEW fields in referrals screen
            'restricted_duty'     => 'required|boolean',
            'referred_hospital'   => 'nullable|string',

            // Sick leave
            'sick_leave'          => 'required|boolean',
            'number_of_days'      => 'nullable|numeric',
            'reason_for_leave'    => 'nullable|string',
            'sick_leave_notes'    => 'nullable|string',
        ]);

        if (empty($data['refer_to_specialist'])) {
            $data['specialist_name'] = null;
        }

        if (empty($data['sick_leave'])) {
            $data['number_of_days']   = null;
            $data['reason_for_leave'] = null;
            $data['sick_leave_notes'] = null;
        }

        $consultation->update($data);

        return redirect()
            ->route('consultations.referrals.edit', $consultation->id)
            ->with('success', 'Referrals updated successfully.');
    }

    public function create(Request $request)
    {
        $user      = $request->user();
        $patientId = $request->input('patient_id');

        $patients = Patient::select('id', 'first_name', 'surname')
            ->when($patientId, fn($query) => $query->where('id', $patientId))
            ->get()
            ->map(fn($p) => [
                'id'        => $p->id,
                'full_name' => "{$p->first_name} {$p->surname}",
            ]);

        $consultationHistory = collect();
        if ($patientId) {
            $consultationHistory = Consultation::with(['clinic', 'doctor'])
                ->where('patient_id', $patientId)
                ->latest('consultation_date')
                ->limit(5)
                ->get()
                ->map(function ($consultation) {
                    return [
                        'id'                => $consultation->id,
                        'consultation_date' => $consultation->consultation_date
                            ? Carbon::parse($consultation->consultation_date)->format('Y-m-d')
                            : null,
                        'diagnosis'         => $consultation->diagnosis,
                        'clinic'            => $consultation->clinic?->name,
                        'doctor'            => $consultation->doctor?->name,
                    ];
                });
        }

        $diagnosisPresets = DiagnosisPreset::select('id', 'name', 'icd_code', 'description')->get();

        return Inertia::render('Consultations/ConsultationCreate', [
            'patients'             => $patients,
            'clinics'              => Clinic::select('id', 'name')->get(),
            'doctors'              => User::whereIn('role', [
                UserRole::Doctor->value,
                UserRole::Nurse->value,
                UserRole::SuperAdmin->value,
            ])->select('id', 'name')->get(),
            'patient_id'           => $patientId,
            'current_user'         => [
                'id'        => $user->id,
                'clinic_id' => $user->clinic_id,
            ],
            'consultation_history' => $consultationHistory,
            'diagnosis_presets'    => $diagnosisPresets,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'patient_id'              => 'required|exists:patients,id',
            'consultation_date'       => 'required|date',
            'injury_on_duty'          => 'boolean',

            // Vitals
            'systolic'                => 'nullable|integer',
            'diastolic'               => 'nullable|integer',
            'temperature'             => 'nullable|numeric',
            'heart_rate'              => 'nullable|numeric',
            'respiratory_rate'        => 'nullable|numeric',
            'oxygen_saturation'       => 'nullable|numeric',
            'weight'                  => 'nullable|numeric',
            'height'                  => 'nullable|numeric',

            // Labs / Tests
            'blood_sugar'             => 'nullable|numeric',
            'blood_sugar_unit'        => 'nullable|in:mmol/L,mg/dL',
            'blood_sugar_context'     => 'nullable|in:fasting,random,postprandial,ogtt,other',
            'blood_sugar_measured_at' => 'nullable|date',

            'urine_test_result'       => 'nullable|string',
            'hiv_status'              => 'nullable|string',
            'pregnancy_test_result'   => 'nullable|string',

            // Notes
            'presenting_complaint'    => 'nullable|string',
            'clinical_notes'          => 'nullable|string',
            'diagnosis'               => 'nullable|string',
            'treatment_plan'          => 'nullable|string',
            'prescription'            => 'nullable|string',

            // Follow-up
            'follow_up_required'      => 'boolean',

            // Referrals
            'refer_for_radiology'     => 'boolean',
            'refer_for_lab'           => 'boolean',
            'refer_to_specialist'     => 'boolean',
            'specialist_name'         => 'nullable|string',
            'refer_to_casualty'       => 'boolean',
            'refer_out_patient'       => 'boolean',

            // NEW fields
            'restricted_duty'         => 'boolean',
            'referred_hospital'       => 'nullable|string',

            'sick_leave'              => 'boolean',
            'reason_for_leave'        => 'nullable|string',
            'sick_leave_notes'        => 'nullable|string',
            'number_of_days'          => 'nullable|numeric',

            'follow_up_date'          => 'nullable|date',

            'referral_notes'          => 'nullable|string',

            // Next step
            'next'                    => 'nullable|in:dispense,imaging,lab,notes',
        ]);

        // Creator/Owner = current user id (whether doctor or nurse)
        $validated['doctor_id'] = Auth::id();

        if (! Auth::user()->clinic_id) {
            return back()->withErrors('Clinic ID is required to create a consultation.')->withInput();
        }

        // Normalize blood sugar to mmol/L
        if (! empty($validated['blood_sugar'])) {
            $unit = $validated['blood_sugar_unit'] ?? 'mmol/L';
            if ($unit === 'mg/dL') {
                $validated['blood_sugar'] = round(((float) $validated['blood_sugar']) / 18, 2);
            } else {
                $validated['blood_sugar'] = round(((float) $validated['blood_sugar']), 2);
            }
            $validated['blood_sugar_unit'] = 'mmol/L';
        }

        $weight = $validated['weight'] ?? null;
        $height = $validated['height'] ?? null;

        $bmi = null;
        if ($weight && $height) {
            $heightInMeters = $height / 100;
            $bmi            = round($weight / ($heightInMeters * $heightInMeters), 2);
        }

        $consultation = Consultation::create(array_merge($validated, [
            'clinic_id' => Auth::user()->clinic_id,
            'bmi'       => $bmi,
        ]));

        // Link triage if an active one exists
        $triage = Triage::where('patient_id', $validated['patient_id'])
            ->where('is_active', true)
            ->latest()
            ->first();

        if ($triage) {
            $triage->update([
                'consultation_id' => $consultation->id,
                'is_active'       => false,
            ]);
        }

        $patient     = Patient::find($validated['patient_id']);
        $patientName = $patient ? $patient->full_name : 'Unknown Patient';

        Log::create([
            'user_id'     => Auth::id(),
            'user_name'   => Auth::user()->name ?? null,
            'action'      => 'created',
            'description' => "Created consultation ID {$consultation->id} for patient {$patientName}",
            'loggable_type' => Consultation::class,
            'loggable_id'   => $consultation->id,
            'old_values'    => null,
            'new_values'    => $consultation->toArray(),
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        // Redirect based on next step
        $next = $request->input('next');

        if ($next === 'imaging') {
            return redirect()
                ->route('referrals.imaging.create', [$consultation->id, $validated['patient_id']])
                ->with('success', 'Consultation recorded. Proceed to imaging referral.');
        }

        if ($next === 'lab') {
            return redirect()
                ->route('referrals.lab.create', [$consultation->id, $validated['patient_id']])
                ->with('success', 'Consultation recorded. Proceed to lab referral.');
        }

        if ($next === 'dispense') {
            return redirect()
                ->route('consultations.dispense', $consultation->id)
                ->with('success', 'Consultation recorded. Proceed to dispensing.');
        }

        if ($next === 'notes') {
            return redirect()
                ->route('consultations.doctor-notes.create', ['consultation' => $consultation->id])
                ->with('success', 'Consultation recorded. Proceed to doctor notes.');
        }

        if (Auth::user()->role === UserRole::Doctor) {
            return redirect()
                ->route('consultations.doctor-notes.create', ['consultation' => $consultation->id])
                ->with('success', 'Consultation recorded successfully. Proceed to add doctor notes.');
        }

        return redirect()
            ->route('consultations.dispense', $consultation->id)
            ->with('success', 'Consultation recorded successfully.');
    }

    public function show(Consultation $consultation)
    {
        $user = Auth::user();

        // Load everything we need
        $consultation->load([
            'patient.company',
            'doctor', // must include the consulting user
            'clinic',
            'doctorsnote.doctor',
            'dispensations.clinicMedicationStock.medicationBatch.medication',
            'dispensations.nurseMedicationStock.medicationBatch.medication',
            'dispensations.dispensedBy',
        ]);

        // ENFORCE view auth:
        // Doctors & Nurses: allowed; others: fall back to company/owner rule
        if (! $this->canUserViewConsultation($user, $consultation)) {
            return Inertia::render('Consultations/PrivateFile', [
                'title'   => 'Access Restricted',
                'message' => 'You cannot access this file. If you believe this is an error, please contact your administrator.',
            ]);
        }

        // Map dispensations payload
        $dispensations = $consultation->dispensations->map(function ($d) {
            $clinicStock = $d->clinicMedicationStock;
            $nurseStock  = $d->nurseMedicationStock;

            $clinicMed = optional(optional($clinicStock)->medicationBatch)->medication;
            $nurseMed  = optional(optional($nurseStock)->medicationBatch)->medication;

            $medication  = $nurseMed ?: $clinicMed;
            $batchNumber = optional(optional($nurseStock)->medicationBatch)->batch_number ?? optional(optional($clinicStock)->medicationBatch)->batch_number;

            return [
                'id'                         => $d->id,
                'consultation_id'            => $d->consultation_id,
                'patient_id'                 => $d->patient_id,
                'origin'                     => $nurseStock ? 'nurse' : ($clinicStock ? 'clinic' : null),
                'clinic_medication_stock_id' => $d->clinic_medication_stock_id,
                'nurse_medication_stock_id'  => $d->nurse_medication_stock_id,
                'medication'                 => $medication ? [
                    'name'   => $medication->name,
                    'dosage' => $medication->dosage,
                    'form'   => $medication->form,
                    'unit'   => $medication->unit,
                ] : null,
                'batch_number'               => $batchNumber,
                'quantity'                   => $d->quantity,
                'frequency'                  => $d->frequency,
                'route'                      => $d->route,
                'duration'                   => $d->duration,
                'notes'                      => $d->notes,
                'dispensed_at'               => optional($d->dispensed_at)->toIso8601String(),
                'dispensed_by'               => $d->dispensedBy ? [
                    'id'    => $d->dispensedBy->id,
                    'name'  => $d->dispensedBy->name,
                    'email' => $d->dispensedBy->email ?? null,
                ] : null,
            ];
        })->values();

        // Whether current user can edit (used by UI to show/hide buttons)
        $canEdit = $this->canUserEditConsultation($user, $consultation);

        /**
         * Determine if the consulting user truly has the Doctor role.
         * Supports either a single 'role' (enum|string) or an optional roles() relation.
         */
        $doctorUser    = $consultation->doctor;
        $hasDoctorRole = false;

        if ($doctorUser) {
            $roleVal = $doctorUser->role?->value ?? $doctorUser->role; // enum|string|null
            if (is_string($roleVal)) {
                $hasDoctorRole = strcasecmp(trim($roleVal), 'doctor') === 0;
            }

            // Fallback: if you also have a many-to-many roles() relation
            if (! $hasDoctorRole && method_exists($doctorUser, 'roles')) {
                try {
                    $hasDoctorRole = $doctorUser->roles()
                        ->whereRaw('LOWER(name) = ?', ['doctor'])
                        ->exists();
                } catch (\Throwable $e) {
                    // ignore if roles relation/table doesn't exist
                }
            }
        }

        return Inertia::render('Consultations/ConsultationShow', [
            'consultation' => [
                'id'                      => $consultation->id,
                'patient_id'              => $consultation->patient_id,
                'doctor_id'               => $consultation->doctor_id,
                'clinic_id'               => $consultation->clinic_id,
                'consultation_date'       => $consultation->consultation_date
                    ? Carbon::parse($consultation->consultation_date)->format('Y-m-d')
                    : null,
                'injury_on_duty'          => $consultation->injury_on_duty,

                // Vitals
                'systolic'                => $consultation->systolic,
                'diastolic'               => $consultation->diastolic,
                'temperature'             => $consultation->temperature,
                'heart_rate'              => $consultation->heart_rate,
                'respiratory_rate'        => $consultation->respiratory_rate,
                'oxygen_saturation'       => $consultation->oxygen_saturation,
                'weight'                  => $consultation->weight,
                'height'                  => $consultation->height,
                'bmi'                     => $consultation->bmi,

                // Labs / Tests
                'blood_sugar'             => $consultation->blood_sugar,
                'blood_sugar_unit'        => $consultation->blood_sugar_unit,
                'blood_sugar_context'     => $consultation->blood_sugar_context,
                'blood_sugar_measured_at' => $consultation->blood_sugar_measured_at,
                'urine_test_result'       => $consultation->urine_test_result,
                'hiv_status'              => $consultation->hiv_status,
                'pregnancy_test_result'   => $consultation->pregnancy_test_result,

                // Notes
                'presenting_complaint'    => $consultation->presenting_complaint,
                'clinical_notes'          => $consultation->clinical_notes,
                'diagnosis'               => $consultation->diagnosis,
                'treatment_plan'          => $consultation->treatment_plan,
                'prescription'            => $consultation->prescription,
                'follow_up_required'      => $consultation->follow_up_required,

                // Leave / referral
                'sick_leave'              => $consultation->sick_leave,
                'reason_for_leave'        => $consultation->reason_for_leave,
                'sick_leave_notes'        => $consultation->sick_leave_notes,
                'number_of_days'          => $consultation->number_of_days,

                'refer_for_radiology'     => $consultation->refer_for_radiology,
                'refer_for_lab'           => $consultation->refer_for_lab,
                'refer_to_specialist'     => $consultation->refer_to_specialist,
                'specialist_name'         => $consultation->specialist_name,
                'refer_to_casualty'       => $consultation->refer_to_casualty,
                'refer_out_patient'       => $consultation->refer_out_patient,

                // NEW fields
                'restricted_duty'         => (bool) $consultation->restricted_duty,
                'referred_hospital'       => $consultation->referred_hospital,

                'follow_up_date'          => $consultation->follow_up_date
                    ? Carbon::parse($consultation->follow_up_date)->format('Y-m-d')
                    : null,

                'referral_notes'          => $consultation->referral_notes,
                'created_at'              => $consultation->created_at,
                'updated_at'              => $consultation->updated_at,

                'patient'                 => [
                    'id'              => $consultation->patient->id,
                    'full_name'       => $consultation->patient->first_name . ' ' . $consultation->patient->surname,
                    'first_name'      => $consultation->patient->first_name,
                    'surname'         => $consultation->patient->surname,
                    'employee_number' => $consultation->patient->employee_number,
                    'home_address'    => $consultation->patient->home_address,
                    'date_of_birth'   => $consultation->patient->date_of_birth,
                    'email'           => $consultation->patient->email,
                    'gender'          => $consultation->patient->gender,
                    'id_number'       => $consultation->patient->id_number,
                ],

                'doctor'                  => [
                    'name'           => $consultation->doctor->name,
                    'email'          => $consultation->doctor->email,
                    'signature'      => $consultation->doctor->signature,
                    'qualifications' => $consultation->doctor->qualifications,
                ],

                // ✅ Key flag used by the UI to allow nurse dispensing only if consulted by an actual Doctor
                'consulted_by_doctor'     => (bool) $hasDoctorRole,

                'clinic'                  => [
                    'name'           => $consultation->clinic->name,
                    'address'        => $consultation->clinic->address,
                    'contact_person' => $consultation->clinic->contact_person,
                    'email_address'  => $consultation->clinic->email_address,
                    'city'           => $consultation->clinic->city,
                ],

                'doctors_note'            => $consultation->doctorsnote ? [
                    'id'         => $consultation->doctorsnote->id,
                    'title'      => $consultation->doctorsnote->title,
                    'body'       => $consultation->doctorsnote->body,
                    'note_type'  => $consultation->doctorsnote->note_type,
                    'private'    => (bool) $consultation->doctorsnote->private,
                    'created_at' => optional($consultation->doctorsnote->created_at)->toIso8601String(),
                    'doctor'     => [
                        'id'   => $consultation->doctorsnote->doctor_id,
                        'name' => optional($consultation->doctorsnote->doctor)->name,
                    ],
                ] : null,

                'dispensations'           => $dispensations,

                // Expose edit guard to the UI
                'can_edit'                => $canEdit,
            ],
        ]);
    }

    public function edit(Consultation $consultation)
    {
        $user = request()->user();

        // ENFORCE edit policy: only creator (doctor_id) can edit
        if (! $this->canUserEditConsultation($user, $consultation)) {
            return Inertia::render('Consultations/PrivateFile', [
                'title'   => 'Edit Restricted',
                'message' => 'Only the creator of this consultation can edit it.',
            ]);
        }

        // relations for header bits
        $consultation->loadMissing(['patient', 'doctor', 'clinic']);

        // ALSO load dispensations & medication relations for the sidebar
        $consultation->loadMissing([
            'dispensations.clinicMedicationStock.medicationBatch.medication',
            'dispensations.nurseMedicationStock.medicationBatch.medication',
            'dispensations.dispensedBy',
        ]);

        $fmtDate = static fn(?string $date) => $date ? \Carbon\Carbon::parse($date)->format('Y-m-d') : null;

        // map dispensations to a clean payload
        $dispensations = $consultation->dispensations->map(function ($d) {
            $clinicStock = $d->clinicMedicationStock;
            $nurseStock  = $d->nurseMedicationStock;

            $clinicMed = optional(optional($clinicStock)->medicationBatch)->medication;
            $nurseMed  = optional(optional($nurseStock)->medicationBatch)->medication;

            $medication  = $nurseMed ?: $clinicMed;
            $batchNumber = optional(optional($nurseStock)->medicationBatch)->batch_number ?? optional(optional($clinicStock)->medicationBatch)->batch_number;

            return [
                'id'           => $d->id,
                'origin'       => $nurseStock ? 'nurse' : ($clinicStock ? 'clinic' : null),
                'medication'   => $medication ? [
                    'name'   => $medication->name,
                    'dosage' => $medication->dosage,
                    'form'   => $medication->form,
                    'unit'   => $medication->unit,
                ] : null,
                'batch_number' => $batchNumber,
                'quantity'     => $d->quantity,
                'frequency'    => $d->frequency,
                'route'        => $d->route,
                'duration'     => $d->duration,
                'notes'        => $d->notes,
                'dispensed_at' => optional($d->dispensed_at)?->toIso8601String(),
                'dispensed_by' => $d->dispensedBy ? [
                    'id'   => $d->dispensedBy->id,
                    'name' => $d->dispensedBy->name,
                ] : null,
            ];
        })->values();

        $patientFullName = trim(sprintf(
            '%s %s',
            $consultation->patient?->first_name ?? '',
            $consultation->patient?->surname ?? ''
        ));

        return Inertia::render('Consultations/ConsultationEdit', [
            'consultation'  => [
                'id'                      => $consultation->id,
                'patient_id'              => $consultation->patient_id,
                'doctor_id'               => $consultation->doctor_id,
                'clinic_id'               => $consultation->clinic_id,
                'consultation_date'       => $fmtDate($consultation->consultation_date),
                'injury_on_duty'          => (bool) $consultation->injury_on_duty,

                // Vitals
                'systolic'                => $consultation->systolic,
                'diastolic'               => $consultation->diastolic,
                'temperature'             => $consultation->temperature,
                'heart_rate'              => $consultation->heart_rate,
                'respiratory_rate'        => $consultation->respiratory_rate,
                'oxygen_saturation'       => $consultation->oxygen_saturation,
                'weight'                  => $consultation->weight,
                'height'                  => $consultation->height,
                'bmi'                     => $consultation->bmi,

                // Labs / Tests
                'blood_sugar'             => $consultation->blood_sugar,
                'blood_sugar_unit'        => $consultation->blood_sugar_unit,
                'blood_sugar_context'     => $consultation->blood_sugar_context,
                'blood_sugar_measured_at' => $consultation->blood_sugar_measured_at,
                'urine_test_result'       => $consultation->urine_test_result,
                'hiv_status'              => $consultation->hiv_status,
                'pregnancy_test_result'   => $consultation->pregnancy_test_result,

                // Notes
                'presenting_complaint'    => $consultation->presenting_complaint,
                'clinical_notes'          => $consultation->clinical_notes,
                'diagnosis'               => $consultation->diagnosis,
                'treatment_plan'          => $consultation->treatment_plan,
                'prescription'            => $consultation->prescription,
                'follow_up_required'      => (bool) $consultation->follow_up_required,

                // Leave / referrals
                'sick_leave'              => (bool) $consultation->sick_leave,
                'reason_for_leave'        => $consultation->reason_for_leave,
                'sick_leave_notes'        => $consultation->sick_leave_notes,
                'number_of_days'          => $consultation->number_of_days,

                // Referral flags
                'refer_for_radiology'     => (bool) $consultation->refer_for_radiology,
                'refer_for_lab'           => (bool) $consultation->refer_for_lab,
                'refer_to_specialist'     => (bool) $consultation->refer_to_specialist,
                'refer_to_casualty'       => (bool) $consultation->refer_to_casualty,
                'refer_out_patient'       => (bool) $consultation->refer_out_patient,
                'specialist_name'         => $consultation->specialist_name,

                // NEW fields
                'restricted_duty'         => (bool) $consultation->restricted_duty,
                'referred_hospital'       => $consultation->referred_hospital,

                'follow_up_date'          => $fmtDate($consultation->follow_up_date),
                'referral_notes'          => $consultation->referral_notes,

                'created_at'              => $consultation->created_at,
                'updated_at'              => $consultation->updated_at,

                'patient'                 => ['full_name' => $patientFullName],
                'doctor'                  => ['name' => $consultation->doctor?->name],
                'clinic'                  => ['name' => $consultation->clinic?->name],
            ],

            // Dispensations for sidebar
            'dispensations' => $dispensations,

            'patients'      => Patient::select('id', 'first_name', 'surname')
                ->orderBy('first_name')->orderBy('surname')->get()
                ->map(fn($p) => [
                    'id'        => $p->id,
                    'full_name' => trim(($p->first_name ?? '') . ' ' . ($p->surname ?? ''))],
                ),
            'clinics'       => Clinic::select('id', 'name')->orderBy('name')->get(),
            'doctors'       => User::whereIn('role', [
                \App\Enums\UserRole::Doctor->value,
                \App\Enums\UserRole::Nurse->value,
                \App\Enums\UserRole::SuperAdmin->value,
            ])->select('id', 'name')->orderBy('name')->get(),
            'current_user'  => [
                'id'        => $user->id,
                'clinic_id' => $user->clinic_id,
            ],
        ]);
    }

    public function update(Request $request, Consultation $consultation)
    {
        $user = $request->user();

        // ENFORCE edit policy (owner only)
        if (! $this->canUserEditConsultation($user, $consultation)) {
            return Inertia::render('Consultations/PrivateFile', [
                'title'   => 'Update Restricted',
                'message' => 'Only the creator of this consultation can update it.',
            ]);
        }

        $validated = $request->validate([
            'patient_id'              => 'required|exists:patients,id',
            'doctor_id'               => 'nullable|exists:users,id',
            'clinic_id'               => 'nullable|exists:clinics,id',
            'consultation_date'       => 'required|date',
            'injury_on_duty'          => 'boolean',

            // Vitals
            'systolic'                => 'nullable|integer',
            'diastolic'               => 'nullable|integer',
            'temperature'             => 'nullable|numeric',
            'heart_rate'              => 'nullable|numeric',
            'respiratory_rate'        => 'nullable|numeric',
            'oxygen_saturation'       => 'nullable|numeric',
            'weight'                  => 'nullable|numeric',
            'height'                  => 'nullable|numeric',

            // Labs / Tests
            'blood_sugar'             => 'nullable|numeric',
            'blood_sugar_unit'        => 'nullable|in:mmol/L,mg/dL',
            'blood_sugar_context'     => 'nullable|in:fasting,random,postprandial,ogtt,other',
            'blood_sugar_measured_at' => 'nullable|date',

            'urine_test_result'       => 'nullable|string',
            'hiv_status'              => 'nullable|string',
            'pregnancy_test_result'   => 'nullable|string',

            // Notes
            'presenting_complaint'    => 'nullable|string',
            'clinical_notes'          => 'nullable|string',
            'diagnosis'               => 'nullable|string',
            'treatment_plan'          => 'nullable|string',
            'prescription'            => 'nullable|string',

            // Follow-up
            'follow_up_required'      => 'boolean',
            'sick_leave'              => 'boolean',
            'reason_for_leave'        => 'nullable|string',
            'sick_leave_notes'        => 'nullable|string',
            'number_of_days'          => 'nullable|integer',

            // Referrals
            'refer_for_radiology'     => 'boolean',
            'refer_for_lab'           => 'boolean',
            'refer_to_specialist'     => 'boolean',
            'specialist_name'         => 'nullable|string',
            'refer_to_casualty'       => 'boolean',
            'refer_out_patient'       => 'boolean',

            // NEW fields
            'restricted_duty'         => 'boolean',
            'referred_hospital'       => 'nullable|string',

            'follow_up_date'          => 'nullable|date',

            'referral_notes'          => 'nullable|string',
        ]);

        // Normalize blood sugar value: always store mmol/L
        if (! empty($validated['blood_sugar'])) {
            $unit = $validated['blood_sugar_unit'] ?? 'mmol/L';
            if ($unit === 'mg/dL') {
                $validated['blood_sugar'] = round(((float) $validated['blood_sugar']) / 18, 2);
            } else {
                $validated['blood_sugar'] = round(((float) $validated['blood_sugar']), 2);
            }
            $validated['blood_sugar_unit'] = 'mmol/L';
        }

        $weight = $validated['weight'] ?? null;
        $height = $validated['height'] ?? null;

        $bmi = null;
        if ($weight && $height) {
            $heightInMeters = $height / 100;
            $bmi            = round($weight / ($heightInMeters * $heightInMeters), 2);
        }

        $oldValues = $consultation->getOriginal();

        $consultation->update(array_merge($validated, [
            'bmi' => $bmi,
        ]));

        $patient     = Patient::find($validated['patient_id']);
        $patientName = $patient ? $patient->full_name : 'Unknown Patient';

        Log::create([
            'user_id'     => Auth::id(),
            'user_name'   => Auth::user()->name ?? null,
            'action'      => 'updated',
            'description' => "Updated consultation ID {$consultation->id} for patient {$patientName}",
            'loggable_type' => get_class($consultation),
            'loggable_id'   => $consultation->id,
            'old_values'    => $oldValues,
            'new_values'    => $consultation->getChanges(),
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        return redirect()->route('consultations.show', $consultation->id)
            ->with('success', 'Consultation updated successfully.');
    }

    public function closeConsultation(Consultation $consultation)
    {
        $consultation->update(['is_active' => false]);

        $triage = Triage::where('consultation_id', $consultation->id)->first();
        if ($triage) {
            $triage->update(['is_active' => false]);
        }

        return Redirect::route('dashboard')
            ->with('success', 'Consultation closed successfully.');
    }
}
