<?php

namespace App\Http\Controllers;

use App\Models\Clinic;
use App\Models\Consultation;
use App\Models\Triage;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardReportsController extends Controller
{
    /**
     * Helper: resolve the clinic IDs the current user can access (primary + pivot).
     */
    private function accessibleClinicIds(): array
    {
        $user = Auth::user();

        if (!$user) {
            return [];
        }

        // Prefer the helper if your User model has it
        if (method_exists($user, 'accessibleClinicIds')) {
            return $user->accessibleClinicIds();
        }

        // Safe fallback
        $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
        if (!empty($user->clinic_id)) {
            $ids[] = (int) $user->clinic_id;
        }
        return array_values(array_unique($ids));
    }

    /**
     * JSON: 7-day triage counts (check-ins) + today's consultations per clinic
     * Route name: reports.weekly
     */
    public function weeklySummary(Request $request)
    {
        $clinicIds = $this->accessibleClinicIds();
        if (empty($clinicIds)) {
            return response()->json([
                'week' => [], 'todayConsultations' => [], 'totalToday' => 0,
            ]);
        }

        // 7-day window: today and previous 6 days
        $end   = Carbon::today()->endOfDay();
        $start = Carbon::today()->subDays(6)->startOfDay();

        // Aggregate triages by DATE(created_at)
        $raw = Triage::query()
            ->select(DB::raw('DATE(created_at) as d'), DB::raw('COUNT(*) as c'))
            ->whereIn('clinic_id', $clinicIds)
            ->whereBetween('created_at', [$start, $end])
            ->groupBy('d')
            ->pluck('c', 'd'); // [ 'YYYY-MM-DD' => count ]

        // Build a normalized 7-day sequence with zero-fills
        $cursor = $start->copy();
        $week = [];
        while ($cursor->lte($end)) {
            $key = $cursor->toDateString();
            $week[] = [
                'date'  => $key,
                'label' => $cursor->format('D'),
                'count' => (int) ($raw[$key] ?? 0),
            ];
            $cursor->addDay();
        }

        // Today's consultations per clinic (using consultation_date)
        $todayStart = Carbon::today()->startOfDay();
        $todayEnd   = Carbon::today()->endOfDay();

        $todayRows = Consultation::query()
            ->join('clinics', 'clinics.id', '=', 'consultations.clinic_id')
            ->whereIn('consultations.clinic_id', $clinicIds)
            ->whereBetween('consultations.consultation_date', [$todayStart, $todayEnd])
            ->groupBy('consultations.clinic_id', 'clinics.name')
            ->orderBy('clinics.name')
            ->get([
                'consultations.clinic_id as clinic_id',
                'clinics.name as clinic_name',
                DB::raw('COUNT(*) as count'),
            ]);

        $todayList = $todayRows->map(function ($r) {
            return [
                'clinic_id'   => (int) $r->clinic_id,
                'clinic_name' => (string) $r->clinic_name,
                'count'       => (int) $r->count,
            ];
        })->values()->all();

        $totalToday = array_sum(array_column($todayList, 'count'));

        return response()->json([
            'week'             => $week,
            'todayConsultations' => $todayList,
            'totalToday'       => $totalToday,
        ]);
    }

    /**
     * JSON: today's open triages list (the card on the right column in your page)
     * Route name: daily.triages
     */
    public function todayTriages()
    {
        $clinicIds = $this->accessibleClinicIds();
        if (empty($clinicIds)) {
            return response()->json(['triages' => []]);
        }

        $start = Carbon::today()->startOfDay();
        $end   = Carbon::today()->endOfDay();

        $triages = Triage::query()
            ->with([
                'patient:id,first_name,surname',
                'nurse:id,name',
                'consultor:id,name',
            ])
            ->whereIn('clinic_id', $clinicIds)
            ->whereBetween('created_at', [$start, $end])
            ->orderByDesc('created_at')
            ->get(['id', 'triage_level', 'created_at', 'patient_id', 'nurse_id', 'consultor_id']);

        return response()->json(['triages' => $triages]);
    }
}
