<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use Carbon\Carbon;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class DiagnosisReportController extends Controller
{
    /**
     * Monthly Diagnosis report:
     * - Required: year, month
     * - One sheet per clinic (Clinic Name (City)).
     * - Also includes an "All Clinics" sheet.
     * - Columns: Diagnosis | Total Cases | Employees | Dependents
     * - Uses consultation_date in-range, falling back to created_at.
     */
    public function monthly(Request $request)
    {
        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month, 422, 'Year and month are required.');

        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end   = (clone $start)->endOfMonth();
        $periodLabel = $start->isoFormat('MMMM YYYY');

        // Pull consultations in the month with patient & clinic
        $rows = Consultation::query()
            ->with([
                'patient:id,first_name,surname,parent_patient_id',
                'clinic:id,name,city',
            ])
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('consultation_date', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->whereNull('consultation_date')
                         ->whereBetween('created_at', [$start, $end]);
                  });
            })
            ->orderBy('consultation_date')
            ->orderBy('created_at')
            ->get();

        // Group by clinic label: "Name (City)"
        $byClinic = $rows->groupBy(function ($c) {
            $clinic = $c->clinic;
            if (!$clinic) return 'Unknown Clinic';
            $name = (string) ($clinic->name ?? '');
            $city = (string) ($clinic->city ?? '');
            return trim($name . ($city ? " ({$city})" : ''));
        })->sortKeys();

        // Header meta
        $generatedAt = now()->format('Y-m-d H:i:s');
        $filtersLine = "Period: {$periodLabel}";
        $fileName = sprintf('Diagnosis_%d_%02d.xlsx', $year, $month);

        return response()->streamDownload(function () use ($rows, $byClinic, $generatedAt, $filtersLine) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // ===== All Clinics (Overview) =====
            $writer->getCurrentSheet()->setName($this->safeSheetName('All Clinics'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['DIAGNOSIS REPORT — ALL CLINICS'],
                ['Generated At', $generatedAt],
                ['Filters', $filtersLine],
            ]);
            $this->writeDiagnosisTable($writer, $rows, $headerStyle, $wrapStyle);
            $this->writeDiagnosisTotals($writer, $rows);

            // ===== One sheet per clinic =====
            $used = ['All Clinics' => true];
            foreach ($byClinic as $clinicLabel => $dataset) {
                $writer->addNewSheetAndMakeItCurrent();
                $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($clinicLabel));
                $writer->getCurrentSheet()->setName($sheetName);

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ["DIAGNOSIS REPORT — {$clinicLabel}"],
                    ['Generated At', $generatedAt],
                    ['Filters', $filtersLine],
                ]);
                $this->writeDiagnosisTable($writer, $dataset, $headerStyle, $wrapStyle);
                $this->writeDiagnosisTotals($writer, $dataset);
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /** Build the diagnosis table for a dataset: Diagnosis | Total | Employees | Dependents */
    private function writeDiagnosisTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle
    ): void {
        $headers = ['Diagnosis', 'Total Cases', 'Employees', 'Dependents'];
        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Widen columns
        $opts = $writer->getOptions();
        $opts->setColumnWidth(50, 1); // Diagnosis
        $opts->setColumnWidth(14, 2); // Total
        $opts->setColumnWidth(14, 3); // Employees
        $opts->setColumnWidth(14, 4); // Dependents

        // Normalize diagnosis text
        $normalize = function ($val): string {
            $dx = trim((string) ($val ?? ''));
            return $dx !== '' ? $dx : '(No diagnosis)';
        };

        // Totals per diagnosis
        $byDxCounts = $dataset
            ->groupBy(fn($c) => $normalize($c->diagnosis))
            ->map->count();

        // Employee vs Dependent splits
        $empCounts = $dataset
            ->filter(fn($c) => empty(optional($c->patient)->parent_patient_id))
            ->groupBy(fn($c) => $normalize($c->diagnosis))
            ->map->count();

        $depCounts = $dataset
            ->filter(fn($c) => !empty(optional($c->patient)->parent_patient_id))
            ->groupBy(fn($c) => $normalize($c->diagnosis))
            ->map->count();

        // Sort by total desc, then diagnosis asc
        $sorted = $byDxCounts
            ->sortDesc()
            ->map(function ($total, $dx) use ($empCounts, $depCounts) {
                return [
                    'dx'   => $dx,
                    'tot'  => (int) $total,
                    'emp'  => (int) ($empCounts[$dx] ?? 0),
                    'dep'  => (int) ($depCounts[$dx] ?? 0),
                ];
            })
            ->values();

        foreach ($sorted as $row) {
            $writer->addRow(Row::fromValues([
                $row['dx'],
                $row['tot'],
                $row['emp'],
                $row['dep'],
            ], $wrapStyle));
        }
    }

    private function writeDiagnosisTotals(XLSXWriter $writer, $dataset): void
    {
        $writer->addRow(Row::fromValues([]));
        $uniqueDiagnoses = $dataset->pluck('diagnosis')
            ->map(fn($v) => trim((string) ($v ?? '')) ?: '(No diagnosis)')
            ->unique()
            ->count();

        $writer->addRow(Row::fromValues(['Total Consultations', $dataset->count()]));
        $writer->addRow(Row::fromValues(['Unique Diagnoses', $uniqueDiagnoses]));
    }

    /* ---------------- helpers ---------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
