<?php
namespace App\Http\Controllers;

use App\Models\ClinicMedicationStock;
use App\Models\Consultation;
use App\Models\Dispensation;
use App\Models\NurseMedicationStock;
use App\Models\Patient;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class DispensationController extends Controller
{

    

    public function index(Request $request, Consultation $consultation)
    {
        if (! $consultation) {
            abort(404, 'Consultation not found');
        }

        $user = Auth::user();
        if (! $user || ! $user->clinic_id) {
            abort(403, 'Your account is not assigned to a clinic.');
        }

        $search = $request->input('search');

        dd($user->clinic_id);

        // Clinic medication stocks for the AUTHENTICATED USER'S clinic (primary/home clinic)
        $clinicMedicationStocks = ClinicMedicationStock::with(['medicationBatch.medication'])
            ->where('clinic_id', $user->clinic_id) // ← changed from $consultation->clinic_id
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->paginate(10)
            ->withQueryString();

        // Nurse medication stocks for the authenticated nurse in their clinic
        $nurseMedicationStocks = NurseMedicationStock::with(['nurse', 'clinic', 'medicationBatch.medication'])
            ->where('nurse_id', Auth::id())
            ->where('clinic_id', $user->clinic_id)
            ->where('quantity', '>', 0)
            ->when($search, function ($query) use ($search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->paginate(10)
            ->withQueryString();

        // Dispensations for this consultation (both clinic & nurse stock)
        $dispensations = Dispensation::with([
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy',
        ])
            ->where('consultation_id', $consultation->id)
            ->latest()
            ->get();

        return inertia('Medications/MedicationDispense', [
            'consultation'           => $consultation->load('patient'),
            'clinicMedicationStocks' => $clinicMedicationStocks,
            'nurseMedicationStocks'  => $nurseMedicationStocks,
            'dispensations'          => $dispensations,
            'filters'                => $request->only('search'),
        ]);
    }

    

    public function export(Request $request)
    {
        try {
            $filters = $request->validate([
                'search'       => 'nullable|string',
                'start_date'   => 'nullable|date',
                'end_date'     => 'nullable|date|after_or_equal:start_date',
                'patient_id'   => 'nullable|integer|exists:patients,id',
                'dispensed_by' => 'nullable|integer|exists:users,id',
            ]);

            $query = Dispensation::with([
                'patient',
                'clinicMedicationStock.medicationBatch.medication',
                'nurseMedicationStock.medicationBatch.medication',
                'dispensedBy',
                'consultation',
            ])
                ->when($filters['search'] ?? false, function ($query, $search) {
                    $query->where(function ($q) use ($search) {
                        $q->whereHas('patient', function ($q) use ($search) {
                            $q->where('first_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%");
                        })
                            ->orWhereHas('clinicMedicationStock.medicationBatch.medication', function ($q) use ($search) {
                                $q->where('name', 'like', "%{$search}%");
                            })
                            ->orWhereHas('nurseMedicationStock.medicationBatch.medication', function ($q) use ($search) {
                                $q->where('name', 'like', "%{$search}%");
                            });
                    });
                })
                ->when($filters['start_date'] ?? false, function ($query, $date) {
                    $query->whereDate('dispensed_at', '>=', $date);
                })
                ->when($filters['end_date'] ?? false, function ($query, $date) {
                    $query->whereDate('dispensed_at', '<=', $date);
                })
                ->when($filters['patient_id'] ?? false, function ($query, $patientId) {
                    $query->where('patient_id', $patientId);
                })
                ->when($filters['dispensed_by'] ?? false, function ($query, $userId) {
                    $query->where('dispensed_by', $userId);
                })
                ->orderBy('dispensed_at', 'desc');

            $dispensations = $query->get();

            if ($dispensations->isEmpty()) {
                return response()->json([
                    'message' => 'No dispensations found matching your criteria',
                ], 404);
            }

            $exportData = $dispensations->map(function ($dispensation) {
                // Determine medication name from clinic or nurse stock
                $medicationName = $dispensation->clinicMedicationStock->medicationBatch->medication->name ?? $dispensation->nurseMedicationStock->medicationBatch->medication->name ?? 'N/A';

                return [
                    'Date'               => \Carbon\Carbon::parse($dispensation->dispensed_at)->format('Y-m-d H:i:s'),
                    'Patient First Name' => $dispensation->patient->first_name ?? 'N/A',
                    'Patient Surname'    => $dispensation->patient->surname ?? 'N/A',
                    'Medication'         => $medicationName,
                    'Quantity'           => $dispensation->quantity,
                    'Dosage'             => $dispensation->route,
                    'Dispensed By'       => $dispensation->dispensedBy->name ?? 'N/A',
                    'Diagnosis'          => $dispensation->consultation->diagnosis ?? 'N/A',
                    'Dispense Notes'     => $dispensation->notes ?? '',
                ];
            });

            return response()->json($exportData);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to generate export: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function allDispensations(Request $request)
    {
        $filters = $request->only('search', 'start_date', 'end_date', 'patient_id', 'dispensed_by');

        $dispensations = Dispensation::with([
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy:id,name',
            'patient:id,first_name,surname',
        ])
            ->when($filters['search'] ?? null, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    // Search clinic stock medication
                    $q->whereHas('clinicMedicationStock.medicationBatch.medication', function ($q2) use ($search) {
                        $q2->where('name', 'like', "%{$search}%");
                    })
                    // Search nurse stock medication
                        ->orWhereHas('nurseMedicationStock.medicationBatch.medication', function ($q2) use ($search) {
                            $q2->where('name', 'like', "%{$search}%");
                        })
                    // Search patient name
                        ->orWhereHas('patient', function ($q2) use ($search) {
                            $q2->where('first_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%");
                        });
                });
            })
            ->when($filters['start_date'] ?? null, function ($query, $date) {
                $query->whereDate('dispensed_at', '>=', $date);
            })
            ->when($filters['end_date'] ?? null, function ($query, $date) {
                $query->whereDate('dispensed_at', '<=', $date);
            })
            ->when($filters['patient_id'] ?? null, function ($query, $patientId) {
                $query->where('patient_id', $patientId);
            })
            ->when($filters['dispensed_by'] ?? null, function ($query, $userId) {
                $query->where('dispensed_by', $userId);
            })
            ->latest('dispensed_at')
            ->paginate(10)
            ->withQueryString();

        return inertia('Dispensations/AllDispensations', [
            'dispensations' => $dispensations,
            'patients'      => Patient::select('id', 'first_name', 'surname')->get(),
            'users'         => User::select('id', 'name')->get(),
            'filters'       => $filters,
        ]);
    }

    public function storeBatch(Request $request, Consultation $consultation)
    {
        $data = $request->validate([
            'patient_id'         => 'required|exists:patients,id',
            'items'              => 'required|array|min:1',
            'items.*.stock_type' => 'required|string|in:clinic,nurse',
            'items.*.stock_id'   => 'required|integer',
            'items.*.quantity'   => 'required|integer|min:1',
            'items.*.frequency'  => 'nullable|string|max:255',
            'items.*.route'      => 'nullable|string|max:255',
            'items.*.duration'   => 'nullable|integer|min:1',
            'items.*.notes'      => 'nullable|string',
            'items.*.request_id' => 'nullable|string|max:64', // client idempotency per row
        ]);

        DB::beginTransaction();
        try {
            foreach ($data['items'] as $item) {
                // Reuse your single-item logic inline (lock stock, upsert dispensation, decrement stock)
                if ($item['stock_type'] === 'clinic') {
                    $stock = \App\Models\ClinicMedicationStock::whereKey($item['stock_id'])
                        ->lockForUpdate()
                        ->firstOrFail();
                    $stockRelationKey = 'clinic_medication_stock_id';
                } else {
                    $stock = \App\Models\NurseMedicationStock::whereKey($item['stock_id'])
                        ->lockForUpdate()
                        ->firstOrFail();
                    $stockRelationKey = 'nurse_medication_stock_id';
                }

                if ($stock->quantity < $item['quantity']) {
                    throw \Illuminate\Validation\ValidationException::withMessages([
                        'items' => ["Only {$stock->quantity} items available for stock ID {$stock->id}"],
                    ]);
                }

                // Idempotency per row
                if (! empty($item['request_id'])) {
                    $already = \App\Models\Dispensation::where('request_id', $item['request_id'])->first();
                    if ($already) {
                        continue; // skip duplicate row
                    }
                }

                $where = [
                    'consultation_id' => $consultation->id,
                    'patient_id'      => $data['patient_id'],
                    $stockRelationKey => $stock->id,
                ];

                $disp      = \App\Models\Dispensation::where($where)->lockForUpdate()->first();
                $oldValues = $disp?->getOriginal();

                if ($disp) {
                    $disp->quantity     = (int) $disp->quantity + (int) $item['quantity'];
                    $disp->frequency    = $item['frequency'] ?? $disp->frequency;
                    $disp->route        = $item['route'] ?? $disp->route;
                    $disp->duration     = $item['duration'] ?? $disp->duration;
                    $disp->notes        = $item['notes'] ?? $disp->notes;
                    $disp->dispensed_by = Auth::id();
                    $disp->dispensed_at = now();
                    if (! empty($item['request_id'])) {
                        $disp->request_id = $item['request_id'];
                    }
                    $disp->save();
                } else {
                    $disp = new \App\Models\Dispensation(array_merge($where, [
                        'quantity'     => $item['quantity'],
                        'frequency'    => $item['frequency'] ?? null,
                        'route'        => $item['route'] ?? null,
                        'duration'     => $item['duration'] ?? null,
                        'notes'        => $item['notes'] ?? null,
                        'dispensed_by' => Auth::id(),
                        'dispensed_at' => now(),
                        'request_id'   => $item['request_id'] ?? null,
                    ]));

                    try {
                        $disp->save();
                    } catch (\Illuminate\Database\QueryException $e) {
                        if (isset($e->errorInfo[1]) && (int) $e->errorInfo[1] === 1062) {
                            $disp               = \App\Models\Dispensation::where($where)->lockForUpdate()->first();
                            $oldValues          = $disp?->getOriginal();
                            $disp->quantity     = (int) $disp->quantity + (int) $item['quantity'];
                            $disp->frequency    = $item['frequency'] ?? $disp->frequency;
                            $disp->route        = $item['route'] ?? $disp->route;
                            $disp->duration     = $item['duration'] ?? $disp->duration;
                            $disp->notes        = $item['notes'] ?? $disp->notes;
                            $disp->dispensed_by = Auth::id();
                            $disp->dispensed_at = now();
                            if (! empty($item['request_id'])) {
                                $disp->request_id = $item['request_id'];
                            }
                            $disp->save();
                        } else {
                            throw $e;
                        }
                    }
                }

                // Decrement stock
                $stock->decrement('quantity', $item['quantity']);

                // Optional: logging per row
                $patient = \App\Models\Patient::find($data['patient_id']);
                \App\Models\Log::create([
                    'user_id'     => Auth::id(),
                    'user_name'   => Auth::user()->name ?? null,
                    'action'      => $oldValues ? 'updated' : 'created',
                    'description' => "Medication dispensation ID {$disp->id} (batch) for patient " . ($patient?->full_name ?? 'Unknown'),
                    'loggable_type' => \App\Models\Dispensation::class,
                    'loggable_id'   => $disp->id,
                    'old_values'    => $oldValues,
                    'new_values'    => $disp->toArray(),
                    'ip_address'    => $request->ip(),
                    'user_agent'    => $request->header('User-Agent'),
                ]);
            }

            DB::commit();
            return back()->with('success', 'Medications dispensed successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return back()->withErrors(['error' => 'Failed to dispense medications. Please try again.']);
        }
    }

    public function store(Request $request, Consultation $consultation)
    {
        $validated = $request->validate([
            'stock_type' => 'required|string|in:clinic,nurse',
            'stock_id'   => 'required|integer',
            'quantity'   => 'required|integer|min:1',
            'patient_id' => 'required|exists:patients,id',
            'frequency'  => 'nullable|string|max:255',
            'route'      => 'nullable|string|max:255',
            'duration'   => 'nullable|integer|min:1',
            'notes'      => 'nullable|string',
            // Optional: provide a client-generated UUID to make retries idempotent
            'request_id' => 'nullable|string|size:36',
        ]);

        try {
            DB::beginTransaction();

            // 1) Lock the stock row so only one request decrements at a time
            if ($validated['stock_type'] === 'clinic') {
                $stock = ClinicMedicationStock::whereKey($validated['stock_id'])
                    ->lockForUpdate()
                    ->firstOrFail();
                $stockRelationKey = 'clinic_medication_stock_id';
            } else {
                $stock = NurseMedicationStock::whereKey($validated['stock_id'])
                    ->lockForUpdate()
                    ->firstOrFail();
                $stockRelationKey = 'nurse_medication_stock_id';
            }

            if ($stock->quantity < $validated['quantity']) {
                throw ValidationException::withMessages([
                    'quantity' => "Only {$stock->quantity} items available",
                ]);
            }

            // 2) Optional idempotency: if this exact request_id was processed, short-circuit
            if (! empty($validated['request_id'])) {
                $already = Dispensation::where('request_id', $validated['request_id'])->first();
                if ($already) {
                    DB::rollBack(); // Nothing to do; don’t change stock twice
                    return back()->with('success', 'This dispensation was already processed.');
                }
            }

            // 3) Lock or create the dispensation row
            $where = [
                'consultation_id' => $consultation->id,
                'patient_id'      => $validated['patient_id'],
                $stockRelationKey => $stock->id,
            ];

            // Try to lock an existing row first
            $disp      = Dispensation::where($where)->lockForUpdate()->first();
            $oldValues = $disp?->getOriginal();

            if ($disp) {
                // Update + increment in one locked scope
                $disp->quantity     = (int) $disp->quantity + (int) $validated['quantity'];
                $disp->frequency    = $validated['frequency'] ?? $disp->frequency;
                $disp->route        = $validated['route'] ?? $disp->route;
                $disp->duration     = $validated['duration'] ?? $disp->duration;
                $disp->notes        = $validated['notes'] ?? $disp->notes;
                $disp->dispensed_by = Auth::id();
                $disp->dispensed_at = now();
                if (! empty($validated['request_id'])) {
                    $disp->request_id = $validated['request_id'];
                }
                $disp->save();
            } else {
                // Create a fresh one; unique index will protect us against a rare race
                $disp = new Dispensation(array_merge($where, [
                    'quantity'     => $validated['quantity'],
                    'frequency'    => $validated['frequency'] ?? null,
                    'route'        => $validated['route'] ?? null,
                    'duration'     => $validated['duration'] ?? null,
                    'notes'        => $validated['notes'] ?? null,
                    'dispensed_by' => Auth::id(),
                    'dispensed_at' => now(),
                    'request_id'   => $validated['request_id'] ?? null,
                ]));

                try {
                    $disp->save();
                } catch (\Illuminate\Database\QueryException $e) {
                    // If another request inserted the same row first, fall back to update
                    // MySQL duplicate key error code is 1062
                    if (isset($e->errorInfo[1]) && (int) $e->errorInfo[1] === 1062) {
                        $disp               = Dispensation::where($where)->lockForUpdate()->first();
                        $oldValues          = $disp?->getOriginal();
                        $disp->quantity     = (int) $disp->quantity + (int) $validated['quantity'];
                        $disp->frequency    = $validated['frequency'] ?? $disp->frequency;
                        $disp->route        = $validated['route'] ?? $disp->route;
                        $disp->duration     = $validated['duration'] ?? $disp->duration;
                        $disp->notes        = $validated['notes'] ?? $disp->notes;
                        $disp->dispensed_by = Auth::id();
                        $disp->dispensed_at = now();
                        if (! empty($validated['request_id'])) {
                            $disp->request_id = $validated['request_id'];
                        }
                        $disp->save();
                    } else {
                        throw $e;
                    }
                }
            }

            // 4) Finally, decrement stock (still under the stock row lock)
            $stock->decrement('quantity', $validated['quantity']);

            // 5) Logging (unchanged)
            $patient     = \App\Models\Patient::find($validated['patient_id']);
            $patientName = $patient ? $patient->full_name : 'Unknown Patient';

            \App\Models\Log::create([
                'user_id'     => Auth::id(),
                'user_name'   => Auth::user()->name ?? null,
                'action'      => $oldValues ? 'updated' : 'created',
                'description' => "Medication dispensation ID {$disp->id} for patient {$patientName}",
                'loggable_type' => \App\Models\Dispensation::class,
                'loggable_id'   => $disp->id,
                'old_values'    => $oldValues,
                'new_values'    => $disp->toArray(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            DB::commit();
            return back()->with('success', 'Medication dispensed successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return back()->withErrors(['error' => 'Failed to dispense medication. Please try again.']);
        }
    }
}
