<?php
namespace App\Http\Controllers;

use App\Models\ClinicMedicationStock;
use App\Models\Consultation;
use App\Models\Dispensation;
use App\Models\DispensationCorrection;
use App\Models\NurseMedicationStock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Inertia\Inertia;

class DispensationCorrectionController extends Controller
{

    public function editDispensed(Consultation $consultation)
    {
        // Same data the modal needed, but as a page
        $consultation->load('patient', 'clinic', 'doctor');

        $dispensations = Dispensation::with([
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
            'dispensedBy:id,name',
        ])
            ->where('consultation_id', $consultation->id)
            ->latest()
            ->get();

        // stocks used as “target” options
        $user                   = request()->user();
        $clinicId               = $user?->clinic_id;
        $clinicMedicationStocks = \App\Models\ClinicMedicationStock::with(['medicationBatch.medication'])
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->where('quantity', '>', 0)
            ->orderByDesc('id')
            ->get();

        $nurseMedicationStocks = \App\Models\NurseMedicationStock::with(['medicationBatch.medication'])
            ->where('nurse_id', $user?->id)
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->where('quantity', '>', 0)
            ->orderByDesc('id')
            ->get();

        return Inertia::render('Dispensations/EditDispensed', [
            'consultation'           => $consultation,
            'dispensations'          => $dispensations,
            'clinicMedicationStocks' => $clinicMedicationStocks,
            'nurseMedicationStocks'  => $nurseMedicationStocks,
        ]);
    }

    public function create(Dispensation $dispensation)
    {
        $dispensation->load([
            'consultation.patient',
            'clinicMedicationStock.medicationBatch.medication',
            'nurseMedicationStock.medicationBatch.medication',
        ]);

        $user     = request()->user();
        $clinicId = $user?->clinic_id;

        $clinicMedicationStocks = \App\Models\ClinicMedicationStock::with(['medicationBatch.medication'])
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->where('quantity', '>', 0)
            ->orderByDesc('id')
            ->get();

        $nurseMedicationStocks = \App\Models\NurseMedicationStock::with(['medicationBatch.medication'])
            ->where('nurse_id', $user?->id)
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->where('quantity', '>', 0)
            ->orderByDesc('id')
            ->get();

        return Inertia::render('Dispensations/RequestCorrection', [
            'dispensation'           => $dispensation,
            'consultation'           => $dispensation->consultation,
            'clinicMedicationStocks' => $clinicMedicationStocks,
            'nurseMedicationStocks'  => $nurseMedicationStocks,
        ]);
    }

    public function index(Request $request)
    {
        $status  = $request->query('status', 'pending'); // pending | approved | rejected | cancelled | all
        $search  = $request->query('search');
        $start   = $request->query('start_date');
        $end     = $request->query('end_date');
        $perPage = max(5, (int) $request->query('per_page', 15));

        $corrections = DispensationCorrection::query()
            ->with([
                'dispensation:id,consultation_id,patient_id,clinic_medication_stock_id,nurse_medication_stock_id,quantity,notes,dispensed_at',
                'patient:id,first_name,surname,employee_number,company_id,parent_patient_id',
                'patient.company:id,name',
                'requester:id,name',
                'approver:id,name',
            ])
            ->when($status !== 'all', fn($q) => $q->where('status', $status))
            ->when($start, fn($q, $d) => $q->whereDate('requested_at', '>=', $d))
            ->when($end, fn($q, $d) => $q->whereDate('requested_at', '<=', $d))
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->whereHas('patient', function ($p) use ($search) {
                        $p->where('first_name', 'like', "%{$search}%")
                            ->orWhere('surname', 'like', "%{$search}%")
                            ->orWhere('employee_number', 'like', "%{$search}%");
                    })
                        ->orWhere('reason', 'like', "%{$search}%");
                });
            })
            ->latest('requested_at')
            ->paginate($perPage)
            ->withQueryString();

        // Build maps for stocks...
        $fromClinicIds = $corrections->getCollection()
            ->filter(fn($c) => $c->from_stock_type === 'clinic')
            ->pluck('from_stock_id')->unique()->values();

        $fromNurseIds = $corrections->getCollection()
            ->filter(fn($c) => $c->from_stock_type === 'nurse')
            ->pluck('from_stock_id')->unique()->values();

        $toClinicIds = $corrections->getCollection()
            ->filter(fn($c) => $c->to_stock_type === 'clinic')
            ->pluck('to_stock_id')->unique()->values();

        $toNurseIds = $corrections->getCollection()
            ->filter(fn($c) => $c->to_stock_type === 'nurse')
            ->pluck('to_stock_id')->unique()->values();

        $clinicStocksMap = $fromClinicIds->merge($toClinicIds)->unique()->isNotEmpty()
            ? ClinicMedicationStock::with(['medicationBatch.medication'])
            ->whereIn('id', $fromClinicIds->merge($toClinicIds)->unique()->values())
            ->get()->keyBy('id')->map(function ($s) {
            return [
                'id'              => $s->id,
                'batch_number'    => $s->medicationBatch->batch_number ?? null,
                'quantity'        => $s->quantity,
                'medication_name' => $s->medicationBatch->medication->name ?? null,
            ];
        })
            : collect();

        $nurseStocksMap = $fromNurseIds->merge($toNurseIds)->unique()->isNotEmpty()
            ? NurseMedicationStock::with(['medicationBatch.medication'])
            ->whereIn('id', $fromNurseIds->merge($toNurseIds)->unique()->values())
            ->get()->keyBy('id')->map(function ($s) {
            return [
                'id'              => $s->id,
                'batch_number'    => $s->medicationBatch->batch_number ?? null,
                'quantity'        => $s->quantity,
                'medication_name' => $s->medicationBatch->medication->name ?? null,
            ];
        })
            : collect();

        // ===== Only SUPERADMIN can approve/reject =====
        $user       = $request->user();
        $canApprove = false;
        if ($user) {
            if (method_exists($user, 'hasRole')) {
                // Spatie\Permission support
                $canApprove = $user->hasRole('superadmin');
            } elseif (! empty($user->role)) {
                // Single "role" column on users table
                $canApprove = $user->role === 'superadmin';
            } elseif (isset($user->roles)) {
                // Array/collection of role names or role models
                $roles = $user->roles;
                if ($roles instanceof \Illuminate\Support\Collection) {
                    $canApprove = $roles->pluck('name')->contains('superadmin') || $roles->contains('superadmin');
                } elseif (is_array($roles)) {
                    $canApprove = in_array('superadmin', $roles, true)
                    || in_array(['name' => 'superadmin'], $roles, true);
                }
            }
        }

        return inertia('Dispensations/CorrectionsIndex', [
            'corrections'     => $corrections,
            'filters'         => [
                'status'     => $status,
                'search'     => $search,
                'start_date' => $start,
                'end_date'   => $end,
                'per_page'   => $perPage,
            ],
            'clinicStocksMap' => $clinicStocksMap,
            'nurseStocksMap'  => $nurseStocksMap,
            'canApprove'      => $canApprove,
        ]);
    }

    /**
     * Nurse requests a correction for a wrong dispensation.
     */
    public function requestCorrection(Request $request, Dispensation $dispensation)
    {
        $validated = $request->validate([
            'to_stock_type' => 'required|in:clinic,nurse',
            'to_stock_id'   => 'required|integer',
            'quantity'      => 'required|integer|min:1',
            'reason'        => 'nullable|string|max:2000',
            'request_uuid'  => 'nullable|uuid',
        ]);

        // Identify the "from" stock from the existing wrong dispensation
        if ($dispensation->clinic_medication_stock_id) {
            $fromType = 'clinic';
            $fromId   = $dispensation->clinic_medication_stock_id;
        } elseif ($dispensation->nurse_medication_stock_id) {
            $fromType = 'nurse';
            $fromId   = $dispensation->nurse_medication_stock_id;
        } else {
            abort(422, 'This dispensation has no stock reference.');
        }

        // Cannot correct more than the dispensed quantity on this row
        if ((int) $validated['quantity'] > (int) $dispensation->quantity) {
            abort(422, 'Requested correction quantity exceeds the dispensed quantity for this record.');
        }

        // Prevent duplicate requests (idempotency via request_uuid)
        if (! empty($validated['request_uuid'])) {
            $exists = DispensationCorrection::where('request_uuid', $validated['request_uuid'])->first();
            if ($exists) {
                return back()->with('success', 'This correction request already exists.');
            }
        }

        $corr = DispensationCorrection::create([
            'dispensation_id' => $dispensation->id,
            'consultation_id' => $dispensation->consultation_id,
            'patient_id'      => $dispensation->patient_id,

            'from_stock_type' => $fromType,
            'from_stock_id'   => $fromId,

            'to_stock_type'   => $validated['to_stock_type'],
            'to_stock_id'     => $validated['to_stock_id'],

            'quantity'        => (int) $validated['quantity'],
            'reason'          => $validated['reason'] ?? null,
            'status'          => 'pending',

            'requested_by'    => Auth::id(),
            'requested_at'    => now(),
            'request_uuid'    => $validated['request_uuid'] ?? null,
        ]);

        // Audit log
        \App\Models\Log::create([
            'user_id'     => Auth::id(),
            'user_name'   => Auth::user()->name ?? null,
            'action'      => 'created',
            'description' => "Correction request #{$corr->id} for dispensation #{$dispensation->id}",
            'loggable_type' => DispensationCorrection::class,
            'loggable_id'   => $corr->id,
            'old_values'    => null,
            'new_values'    => $corr->toArray(),
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        return back()->with('success', 'Correction request submitted. Awaiting approval.');
    }

    /**
     * Super admin approves & executes the correction atomically.
     */
    public function approveCorrection(Request $request, DispensationCorrection $correction)
    {
        // $this->authorize('approve-correction'); // Gate/Policy must allow only super admin

        abort_if($correction->status !== 'pending', 422, 'Only pending corrections can be approved.');

        $qty = (int) $correction->quantity;

        DB::beginTransaction();
        try {
            // Lock the wrong dispensation and verify quantities
            /** @var Dispensation $wrong */
            $wrong = Dispensation::whereKey($correction->dispensation_id)
                ->lockForUpdate()->firstOrFail();

            if ($qty > (int) $wrong->quantity) {
                throw ValidationException::withMessages([
                    'quantity' => 'Correction quantity exceeds current dispensed quantity.',
                ]);
            }

            // Lock "from" (wrong) stock row
            if ($correction->from_stock_type === 'clinic') {
                $fromStock = ClinicMedicationStock::whereKey($correction->from_stock_id)
                    ->lockForUpdate()->firstOrFail();
            } else {
                $fromStock = NurseMedicationStock::whereKey($correction->from_stock_id)
                    ->lockForUpdate()->firstOrFail();
            }

            // Lock "to" (correct) stock row
            if ($correction->to_stock_type === 'clinic') {
                $toStock = ClinicMedicationStock::whereKey($correction->to_stock_id)
                    ->lockForUpdate()->firstOrFail();
            } else {
                $toStock = NurseMedicationStock::whereKey($correction->to_stock_id)
                    ->lockForUpdate()->firstOrFail();
            }

            // 1) Return wrong medication to stock
            $fromStock->increment('quantity', $qty);

            // 2) Reduce quantity on the wrong dispensation row
            $oldWrong        = $wrong->getOriginal();
            $wrong->quantity = (int) $wrong->quantity - $qty;
            $wrong->notes    = trim(($wrong->notes ?? '') . " | Correction approved (#{$correction->id}), {$qty} returned.");
            $wrong->save();

            // 3) Dispense the correct medication (must have stock available)
            if ((int) $toStock->quantity < $qty) {
                throw ValidationException::withMessages([
                    'stock' => "Not enough quantity in the target stock (have {$toStock->quantity}, need {$qty}).",
                ]);
            }
            $toStock->decrement('quantity', $qty);

            // Create or merge a new dispensation for the correct stock (same consultation & patient)
            $where = [
                'consultation_id' => $wrong->consultation_id,
                'patient_id'      => $wrong->patient_id,
            ];
            if ($correction->to_stock_type === 'clinic') {
                $where['clinic_medication_stock_id'] = $toStock->id;
            } else {
                $where['nurse_medication_stock_id'] = $toStock->id;
            }

            $oldCorrect  = null;
            $correctDisp = Dispensation::where($where)->lockForUpdate()->first();
            if ($correctDisp) {
                $oldCorrect                = $correctDisp->getOriginal();
                $correctDisp->quantity     = (int) $correctDisp->quantity + $qty;
                $correctDisp->frequency    = $wrong->frequency; // carry over if appropriate
                $correctDisp->route        = $wrong->route;
                $correctDisp->duration     = $wrong->duration;
                $correctDisp->dispensed_by = Auth::id(); // approver
                $correctDisp->dispensed_at = now();
                $correctDisp->notes        = trim(($correctDisp->notes ?? '') . " | Correction from #{$wrong->id}, req #{$correction->id}");
                $correctDisp->save();
            } else {
                $correctDispData = array_merge($where, [
                    'quantity'  => $qty,
                    'frequency' => $wrong->frequency,
                    'route'     => $wrong->route,
                    'duration'  => $wrong->duration,
                    'notes'     => "Correction from dispensation #{$wrong->id}, request #{$correction->id}",
                    'dispensed_by' => Auth::id(),
                    'dispensed_at' => now(),
                ]);
                $correctDisp = Dispensation::create($correctDispData);
            }

            // 4) Mark correction approved
            $oldCorr                 = $correction->getOriginal();
            $correction->status      = 'approved';
            $correction->approved_by = Auth::id();
            $correction->approved_at = now();
            $correction->save();

            // 5) Logs
            \App\Models\Log::create([
                'user_id'     => Auth::id(),
                'user_name'   => Auth::user()->name ?? null,
                'action'      => 'approved',
                'description' => "Approved correction #{$correction->id}: returned {$qty} to {$correction->from_stock_type} stock #{$fromStock->id}, dispensed {$qty} from {$correction->to_stock_type} stock #{$toStock->id}",
                'loggable_type' => DispensationCorrection::class,
                'loggable_id' => $correction->id,
                'old_values' => $oldCorr,
                'new_values' => $correction->toArray(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
            ]);

            \App\Models\Log::create([
                'user_id'     => Auth::id(),
                'user_name'   => Auth::user()->name ?? null,
                'action'      => 'updated',
                'description' => "Wrong dispensation adjusted (#{$wrong->id}) due to correction #{$correction->id}",
                'loggable_type' => Dispensation::class,
                'loggable_id' => $wrong->id,
                'old_values' => $oldWrong,
                'new_values' => $wrong->toArray(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
            ]);

            \App\Models\Log::create([
                'user_id'     => Auth::id(),
                'user_name'   => Auth::user()->name ?? null,
                'action'      => $oldCorrect ? 'updated' : 'created',
                'description' => "Correct dispensation " . ($oldCorrect ? "updated" : "created") . " (#{$correctDisp->id}) from correction #{$correction->id}",
                'loggable_type' => Dispensation::class,
                'loggable_id' => $correctDisp->id,
                'old_values' => $oldCorrect,
                'new_values' => $correctDisp->toArray(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
            ]);

            DB::commit();

            return back()->with('success', "Correction #{$correction->id} approved and applied.");
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return back()->withErrors(['error' => 'Failed to approve correction: ' . $e->getMessage()]);
        }
    }

    /**
     * Super admin rejects a correction.
     */
    public function rejectCorrection(Request $request, DispensationCorrection $correction)
    {
        // $this->authorize('approve-correction');

        abort_if($correction->status !== 'pending', 422, 'Only pending corrections can be rejected.');

        $data = $request->validate([
            'rejection_notes' => 'nullable|string|max:2000',
        ]);

        $old = $correction->getOriginal();

        $correction->update([
            'status'          => 'rejected',
            'rejected_by'     => Auth::id(),
            'rejected_at'     => now(),
            'rejection_notes' => $data['rejection_notes'] ?? null,
        ]);

        \App\Models\Log::create([
            'user_id'     => Auth::id(),
            'user_name'   => Auth::user()->name ?? null,
            'action'      => 'rejected',
            'description' => "Rejected correction #{$correction->id}",
            'loggable_type' => DispensationCorrection::class,
            'loggable_id'   => $correction->id,
            'old_values'    => $old,
            'new_values'    => $correction->toArray(),
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->header('User-Agent'),
        ]);

        return back()->with('success', "Correction #{$correction->id} rejected.");
    }

    // ...

/**
 * Simple, robust check for superadmin across different role setups.
 */
    protected function userIsSuperadmin($user): bool
    {
        if (! $user) {
            return false;
        }

        // 1) Spatie\Permission
        if (method_exists($user, 'hasRole')) {
            if ($user->hasRole('superadmin')) {
                return true;
            }

        }

        // 2) Single "role" column
        if (! empty($user->role) && $user->role === 'superadmin') {
            return true;
        }

        // 3) Array / collection of roles
        if (isset($user->roles)) {
            $roles = $user->roles;
            if ($roles instanceof \Illuminate\Support\Collection) {
                if ($roles->pluck('name')->contains('superadmin') || $roles->contains('superadmin')) {
                    return true;
                }

            } elseif (is_array($roles)) {
                if (in_array('superadmin', $roles, true) || in_array(['name' => 'superadmin'], $roles, true)) {
                    return true;
                }

            }
        }

        return false;
    }

/**
 * Delete a dispensation and return its full quantity back to the original stock.
 * Only the user who dispensed it OR a superadmin may perform this.
 */
    public function deleteAndReturn(Request $request, Dispensation $dispensation)
    {
        $user = $request->user();
        abort_if(! $user, 403, 'Unauthorized.');

        // Authorization: dispenser or superadmin
        $isDispenser  = (int) $dispensation->dispensed_by === (int) $user->id;
        $isSuperadmin = $this->userIsSuperadmin($user);

        abort_if(! ($isDispenser || $isSuperadmin), 403, 'Only the dispenser or a superadmin may delete and return this dispensation.');

        // Must have a stock reference
        if (! $dispensation->clinic_medication_stock_id && ! $dispensation->nurse_medication_stock_id) {
            abort(422, 'This dispensation has no stock reference to return to.');
        }

        // Nothing to return?
        $qty = (int) $dispensation->quantity;
        if ($qty <= 0) {
            abort(422, 'This dispensation has zero quantity; nothing to return.');
        }

        DB::beginTransaction();
        try {
            // Re-fetch and lock the dispensation row for update
            /** @var Dispensation $lockedDisp */
            $lockedDisp = Dispensation::whereKey($dispensation->id)
                ->lockForUpdate()
                ->firstOrFail();

            // Re-check quantity after lock
            $qty = (int) $lockedDisp->quantity;
            if ($qty <= 0) {
                throw ValidationException::withMessages([
                    'quantity' => 'This dispensation already has zero quantity; nothing to return.',
                ]);
            }

            $oldDisp = $lockedDisp->getOriginal();

            // Identify and lock the stock row we need to return to
            $stockType = null;
            if ($lockedDisp->clinic_medication_stock_id) {
                $stockType = 'clinic';
                /** @var ClinicMedicationStock $stock */
                $stock = ClinicMedicationStock::whereKey($lockedDisp->clinic_medication_stock_id)
                    ->lockForUpdate()
                    ->firstOrFail();
            } else {
                $stockType = 'nurse';
                /** @var NurseMedicationStock $stock */
                $stock = NurseMedicationStock::whereKey($lockedDisp->nurse_medication_stock_id)
                    ->lockForUpdate()
                    ->firstOrFail();
            }

            $oldStock = $stock->getOriginal();

            // Return quantity back to the stock
            $stock->increment('quantity', $qty);

            // Delete the dispensation row
            $lockedDisp->delete();

            // Logs: stock adjustment
            \App\Models\Log::create([
                'user_id'     => $user->id,
                'user_name'   => $user->name ?? null,
                'action'      => 'updated',
                'description' => "Returned {$qty} to {$stockType} stock #{$stock->id} after deleting dispensation #{$dispensation->id}",
                'loggable_type' => get_class($stock),
                'loggable_id' => $stock->id,
                'old_values' => $oldStock,
                'new_values' => $stock->toArray(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
            ]);

            // Logs: dispensation deletion
            \App\Models\Log::create([
                'user_id'     => $user->id,
                'user_name'   => $user->name ?? null,
                'action'      => 'deleted',
                'description' => "Deleted dispensation #{$dispensation->id} and returned {$qty} to {$stockType} stock #{$stock->id}",
                'loggable_type' => Dispensation::class,
                'loggable_id' => $dispensation->id,
                'old_values' => $oldDisp,
                'new_values' => null,
                'ip_address' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
            ]);

            DB::commit();

            return back()->with('success', "Dispensation #{$dispensation->id} deleted and {$qty} returned to {$stockType} stock.");
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            return back()->withErrors(['error' => 'Failed to delete and return to stock: ' . $e->getMessage()]);
        }
    }
}
