<?php
namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\DoctorNote;
use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class DoctorNoteController extends Controller
{
    public function index(Request $request)
    {
        $user    = Auth::user();
        $roleVal = method_exists($user, 'role') ? ($user->role?->value ?? $user->role) : $user->role;
        $isSuper = $roleVal === 'superadmin';
        $userId  = $user->id;

        $search   = $request->input('search');
        $noteType = $request->input('note_type');   // 'Progress' | 'Discharge' | 'Prescription' | 'Other'
        $private  = $request->input('private');     // 'yes' | 'no' | null
        $onlyMine = $request->boolean('only_mine'); // optional

        $notes = \App\Models\DoctorNote::query()
        // relations already define slim selects + withTrashed when applicable
            ->with(['doctor', 'patient', 'consultation'])

        // Visibility: non-superadmins see public notes or their own private notes
            ->when(! $isSuper && ! $onlyMine, function ($q) use ($userId) {
                $q->where(function ($qq) use ($userId) {
                    $qq->where('private', false)
                        ->orWhere('doctor_id', $userId);
                });
            })

        // Only my notes
            ->when($onlyMine, fn($q) => $q->where('doctor_id', $userId))

        // Filters
            ->when($noteType, fn($q) => $q->where('note_type', $noteType))
            ->when($private === 'yes', fn($q) => $q->where('private', true))
            ->when($private === 'no', fn($q) => $q->where('private', false))
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('title', 'like', "%{$search}%")
                        ->orWhere('body', 'like', "%{$search}%")
                        ->orWhereHas('doctor', fn($d) => $d->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('patient', function ($p) use ($search) {
                            $p->where('first_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%")
                                ->orWhere('employee_number', 'like', "%{$search}%");
                        });
                });
            })

            ->latest() // created_at desc
            ->paginate(15)
            ->appends($request->all());

        return Inertia::render('DoctorNotes/Index', [
            'notes'     => $notes,
            'filters'   => [
                'search'    => $search,
                'note_type' => $noteType,
                'private'   => $private,
                'only_mine' => $onlyMine,
            ],
            'noteTypes' => ['Progress', 'Discharge', 'Prescription', 'Other'],
            'canCreate' => in_array($roleVal, ['doctor', 'superadmin'], true),
        ]);
    }

    public function show(DoctorNote $doctorNote)
    {
        $user    = Auth::user();
        $roleVal = method_exists($user, 'role') ? ($user->role?->value ?? $user->role) : $user->role;
        $isSuper = $roleVal === 'superadmin';

        if ($doctorNote->private && ! $isSuper && $doctorNote->doctor_id !== $user->id) {
            abort(403, 'You are not allowed to view this note.');
        }

        // Keep it simple: eager-load essentials
        $doctorNote->load([
            'doctor:id,name,email',
            'patient:id,first_name,surname,employee_number,gender,id_number',
            'consultation' => function ($q) {
                $q->select([
                    'id', 'patient_id', 'doctor_id', 'clinic_id',
                    'injury_on_duty', 'consultation_date',
                    'systolic', 'diastolic', 'temperature', 'heart_rate',
                    'respiratory_rate', 'oxygen_saturation', 'weight', 'height', 'bmi',
                    'blood_sugar', 'blood_sugar_unit', 'blood_sugar_context', 'blood_sugar_measured_at',
                    'urine_test_result', 'hiv_status', 'pregnancy_test_result',
                    'presenting_complaint', 'clinical_notes', 'diagnosis', 'treatment_plan', 'prescription',
                    'sick_leave', 'reason_for_leave', 'sick_leave_notes', 'number_of_days',
                    'follow_up_date', 'follow_up_required',
                    'referral_notes', 'refer_for_radiology', 'refer_for_lab', 'refer_to_specialist', 'specialist_name',
                    'refer_to_casualty', 'refer_out_patient',
                    'is_active', 'status',
                    'created_at',
                ]);
            },
            'consultation.clinic:id,name,address,city,email_address',
        ]);

        $c = $doctorNote->consultation;

        return Inertia::render('DoctorNotes/Show', [
            'note' => [
                'id'           => $doctorNote->id,
                'title'        => $doctorNote->title,
                'body'         => $doctorNote->body,
                'note_type'    => $doctorNote->note_type,
                'private'      => (bool) $doctorNote->private,
                'meta'         => is_string($doctorNote->meta) ? (json_decode($doctorNote->meta, true) ?? []) : ($doctorNote->meta ?? []),
                'created_at'   => optional($doctorNote->created_at)->toIso8601String(),
                'updated_at'   => optional($doctorNote->updated_at)->toIso8601String(),

                'doctor'       => [
                    'id'    => $doctorNote->doctor?->id,
                    'name'  => $doctorNote->doctor?->name,
                    'email' => $doctorNote->doctor?->email,
                ],

                'patient'      => $doctorNote->patient ? [
                    'id'              => $doctorNote->patient->id,
                    'first_name'      => $doctorNote->patient->first_name,
                    'surname'         => $doctorNote->patient->surname,
                    'employee_number' => $doctorNote->patient->employee_number,
                    'gender'          => $doctorNote->patient->gender,
                    'id_number'       => $doctorNote->patient->id_number,
                ] : null,

                // Expanded consultation block
                'consultation' => $c ? [
                    'id'                      => $c->id,
                    'consultation_date'       => optional($c->consultation_date)->toDateString(),
                    'created_at'              => optional($c->created_at)->toIso8601String(),
                    'injury_on_duty'          => (bool) $c->injury_on_duty,

                    // Vitals
                    'systolic'                => $c->systolic,
                    'diastolic'               => $c->diastolic,
                    'temperature'             => $c->temperature,
                    'heart_rate'              => $c->heart_rate,
                    'respiratory_rate'        => $c->respiratory_rate,
                    'oxygen_saturation'       => $c->oxygen_saturation,
                    'weight'                  => $c->weight,
                    'height'                  => $c->height,
                    'bmi'                     => $c->bmi,

                    // Glucose
                    'blood_sugar'             => $c->blood_sugar,
                    'blood_sugar_unit'        => $c->blood_sugar_unit,
                    'blood_sugar_context'     => $c->blood_sugar_context,
                    'blood_sugar_measured_at' => optional($c->blood_sugar_measured_at)->toIso8601String(),

                    // Other tests
                    'urine_test_result'       => $c->urine_test_result,
                    'hiv_status'              => $c->hiv_status,
                    'pregnancy_test_result'   => $c->pregnancy_test_result,

                    // Notes
                    'presenting_complaint'    => $c->presenting_complaint,
                    'clinical_notes'          => $c->clinical_notes,
                    'diagnosis'               => $c->diagnosis,
                    'treatment_plan'          => $c->treatment_plan,
                    'prescription'            => $c->prescription,

                    // Leave / follow-up
                    'sick_leave'              => (bool) $c->sick_leave,
                    'reason_for_leave'        => $c->reason_for_leave,
                    'sick_leave_notes'        => $c->sick_leave_notes,
                    'number_of_days'          => $c->number_of_days,
                    'follow_up_date'          => optional($c->follow_up_date)->toDateString(),
                    'follow_up_required'      => (bool) $c->follow_up_required,

                    // Referrals
                    'referral_notes'          => $c->referral_notes,
                    'refer_for_radiology'     => (bool) $c->refer_for_radiology,
                    'refer_for_lab'           => (bool) $c->refer_for_lab,
                    'refer_to_specialist'     => (bool) $c->refer_to_specialist,
                    'specialist_name'         => $c->specialist_name,
                    'refer_to_casualty'       => (bool) $c->refer_to_casualty,
                    'refer_out_patient'       => (bool) $c->refer_out_patient,

                    // Status flags
                    'is_active'               => (bool) $c->is_active,
                    'status'                  => (bool) $c->status,

                    // Clinic
                    'clinic'                  => $c->clinic ? [
                        'id'            => $c->clinic->id ?? null,
                        'name'          => $c->clinic->name,
                        'address'       => $c->clinic->address,
                        'city'          => $c->clinic->city,
                        'email_address' => $c->clinic->email_address,
                    ] : null,
                ] : null,
            ],
        ]);
    }

    public function createForConsultation($consultationId)
    {
        $consultation = Consultation::with([
            'patient',
            'doctor',
            'clinic',
            'medicationUsages',
            'dispensations',
            'triage',
        ])->findOrFail($consultationId);

        return Inertia::render('DoctorNotes/Create', [
            'consultation' => $consultation->toArray(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'consultation_id' => ['required', 'exists:consultations,id'],
            'patient_id'      => ['nullable', 'exists:patients,id'],
            'title'           => ['nullable', 'string', 'max:255'],
            'body'            => ['required', 'string'],
            'note_type'       => ['required', Rule::in(['Progress', 'Discharge', 'Prescription', 'Other'])],
            'private'         => ['required', 'boolean'],
        ]);

        $user         = Auth::user();
        $consultation = Consultation::findOrFail($validated['consultation_id']);

        $metaData = [
            'weight'      => $consultation->weight,
            'height'      => $consultation->height,
            'bmi'         => $consultation->bmi,
            'systolic'    => $consultation->systolic,
            'diastolic'   => $consultation->diastolic,
            'temperature' => $consultation->temperature,
            'heart_rate'  => $consultation->heart_rate,
        ];

        DoctorNote::create([
            'consultation_id' => $validated['consultation_id'],
            'doctor_id'       => $user->id,
            'patient_id'      => $validated['patient_id'] ?? null,
            'title'           => $validated['title'] ?? null,
            'body'            => $validated['body'],
            'note_type'       => $validated['note_type'],
            'private'         => $validated['private'],
            'meta'            => $metaData, // let cast handle JSON
            'created_by'      => $user->id,
            'updated_by'      => $user->id,
        ]);

        return redirect()
            ->route('consultations.prescriptions.create', $validated['consultation_id'])
            ->with('success', 'Doctor note created successfully.');
    }
}
