<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Consultation;
use App\Models\DoctorNote;
use App\Models\Diagnosis;
use App\Models\Dispensation;
use App\Models\DispensationCorrection;
use App\Models\ImagingReferral;
use App\Models\LabReferral;
use App\Models\MedicationRequest;
use App\Models\Prescription;
use App\Models\Triage;
use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class DuplicatePatientController extends Controller
{
    /**
     * Heuristics to identify likely duplicates:
     *  A. Exact id_number match (high confidence)
     *  B. employee_number + company_id match (high confidence)
     *  C. normalized full name + date_of_birth (medium)
     *  D. normalized full name + phone (medium)
     *
     *  Special rule:
     *  - If id_number is all zeros (e.g. "000000"), only surface the id_number
     *    group if there is a duplicate by normalized full name within that set.
     */
    public function index(Request $request)
    {
        // Optional filters
        $companyId = $request->integer('company_id');
        $limit     = max(10, (int) $request->integer('limit', 100));

        // Base query (employees and dependents)
        $base = Patient::query()
            ->when($companyId, fn ($q) => $q->where('company_id', $companyId));

        // A) id_number duplicates
        $dupeIdNumbers = (clone $base)
            ->whereNotNull('id_number')
            ->where('id_number', '<>', '')
            ->select('id_number', DB::raw('COUNT(*) as c'))
            ->groupBy('id_number')
            ->having('c', '>', 1)
            ->limit($limit)
            ->pluck('id_number')
            ->all();

        // B) employee_number + company_id duplicates
        $dupeEmpCompany = (clone $base)
            ->whereNotNull('employee_number')
            ->where('employee_number', '<>', '')
            ->select('employee_number', 'company_id', DB::raw('COUNT(*) as c'))
            ->groupBy('employee_number', 'company_id')
            ->having('c', '>', 1)
            ->limit($limit)
            ->get()
            ->map(fn ($r) => ['employee_number' => $r->employee_number, 'company_id' => $r->company_id])
            ->all();

        // Normalized name SQL for DB-side grouping/filters
        $normNameSql = "LOWER(TRIM(CONCAT_WS(' ', first_name, middle_name, surname)))";

        // C) full name + date_of_birth duplicates
        $dupeNameDobKeys = (clone $base)
            ->whereNotNull('date_of_birth')
            ->select([
                DB::raw("$normNameSql as norm_name"),
                DB::raw('DATE(date_of_birth) as dob'),
                DB::raw('COUNT(*) as c'),
            ])
            ->groupBy('norm_name', 'dob')
            ->having('c', '>', 1)
            ->limit($limit)
            ->get()
            ->map(fn ($r) => ['norm_name' => $r->norm_name, 'dob' => $r->dob])
            ->all();

        // D) full name + phone duplicates
        $dupeNamePhoneKeys = (clone $base)
            ->whereNotNull('phone')
            ->where('phone', '<>', '')
            ->select([
                DB::raw("$normNameSql as norm_name"),
                'phone',
                DB::raw('COUNT(*) as c'),
            ])
            ->groupBy('norm_name', 'phone')
            ->having('c', '>', 1)
            ->limit($limit)
            ->get()
            ->map(fn ($r) => ['norm_name' => $r->norm_name, 'phone' => $r->phone])
            ->all();

        $groups = [];

        // ---------- Build groups for A) id_number ----------
        if (!empty($dupeIdNumbers)) {
            $rowsById = (clone $base)
                ->with(['company:id,name', 'parent:id,first_name,surname'])
                ->whereIn('id_number', $dupeIdNumbers)
                ->orderBy('surname')->orderBy('first_name')
                ->get()
                ->groupBy('id_number');

            foreach ($rowsById as $key => $items) {
                $isAllZeros = preg_match('/^0+$/', (string) $key) === 1;

                if ($isAllZeros) {
                    // Only include this id_number group if there's a *name duplicate* inside it.
                    $hasNameDuplicate = collect($items)
                        ->map(function (Patient $p) {
                            $first  = trim((string) $p->first_name);
                            $middle = trim((string) ($p->middle_name ?? ''));
                            $last   = trim((string) $p->surname);
                            $name   = preg_replace('/\s+/u', ' ', trim("$first $middle $last"));
                            return mb_strtolower($name);
                        })
                        ->countBy()
                        ->filter(fn ($c) => $c > 1)
                        ->isNotEmpty();

                    if (!$hasNameDuplicate) {
                        continue;
                    }
                }

                $groups[] = [
                    'type'  => 'id_number',
                    'label' => "National ID: {$key}",
                    'items' => $items->values(),
                ];
            }
        }

        // ---------- Build groups for B) employee_number + company_id ----------
        foreach ($dupeEmpCompany as $pair) {
            $rows = (clone $base)
                ->with(['company:id,name', 'parent:id,first_name,surname'])
                ->where('employee_number', $pair['employee_number'])
                ->where(function ($q) use ($pair) {
                    if ($pair['company_id']) {
                        $q->where('company_id', $pair['company_id']);
                    } else {
                        $q->whereNull('company_id');
                    }
                })
                ->orderBy('surname')->orderBy('first_name')
                ->get();

            if ($rows->count() > 1) {
                $companyName = optional($rows->first()->company)->name ?? 'No Company';
                $groups[] = [
                    'type'  => 'employee_company',
                    'label' => "Employee#: {$pair['employee_number']} @ {$companyName}",
                    'items' => $rows->values(),
                ];
            }
        }

        // ---------- Build groups for C) name + dob ----------
        if (!empty($dupeNameDobKeys)) {
            foreach ($dupeNameDobKeys as $k) {
                $rows = (clone $base)
                    ->with(['company:id,name', 'parent:id,first_name,surname'])
                    ->whereRaw("$normNameSql = ?", [$k['norm_name']])
                    ->whereDate('date_of_birth', $k['dob'])
                    ->orderBy('surname')->orderBy('first_name')
                    ->get();

                if ($rows->count() > 1) {
                    $groups[] = [
                        'type'  => 'name_dob',
                        'label' => "Name + DoB: " . $rows->first()->first_name . ' ' . $rows->first()->surname . " / " . $k['dob'],
                        'items' => $rows->values(),
                    ];
                }
            }
        }

        // ---------- Build groups for D) name + phone ----------
        if (!empty($dupeNamePhoneKeys)) {
            foreach ($dupeNamePhoneKeys as $k) {
                $rows = (clone $base)
                    ->with(['company:id,name', 'parent:id,first_name,surname'])
                    ->whereRaw("$normNameSql = ?", [$k['norm_name']])
                    ->where('phone', $k['phone'])
                    ->orderBy('surname')->orderBy('first_name')
                    ->get();

                if ($rows->count() > 1) {
                    $groups[] = [
                        'type'  => 'name_phone',
                        'label' => "Name + Phone: " . $rows->first()->first_name . ' ' . $rows->first()->surname . " / " . $k['phone'],
                        'items' => $rows->values(),
                    ];
                }
            }
        }

        // Deduplicate groups by label
        $groups = collect($groups)->unique('label')->values()->all();

        return inertia('Duplicates/Index', [
            'groups'     => $groups,
            'companies'  => Company::select('id', 'name')->orderBy('name')->get(),
            'filters'    => ['company_id' => $companyId, 'limit' => $limit],
        ]);
    }

    /**
     * Merge patients: target_id (master) + source_ids[] (to merge into master)
     */
    public function merge(Request $request)
    {
        $data = $request->validate([
            'target_id'    => ['required', 'integer', Rule::exists('patients', 'id')],
            'source_ids'   => ['required', 'array', 'min:1'],
            'source_ids.*' => ['integer', Rule::exists('patients', 'id')],
        ]);

        $targetId  = (int) $data['target_id'];
        $sourceIds = array_values(array_unique(array_map('intval', $data['source_ids'])));
        $sourceIds = array_values(array_filter($sourceIds, fn ($id) => $id !== $targetId));
        if (empty($sourceIds)) {
            return back()->with('error', 'No valid source records to merge.');
        }

        DB::transaction(function () use ($targetId, $sourceIds) {
            /** @var Patient $target */
            $target = Patient::lockForUpdate()->findOrFail($targetId);

            foreach ($sourceIds as $sid) {
                /** @var Patient $source */
                $source = Patient::findOrFail($sid);

                // 1) Reassign dependents
                Patient::where('parent_patient_id', $source->id)->update(['parent_patient_id' => $target->id]);

                // 2) Reassign all directly-linked models.
                //    We automatically detect the correct FK column if it exists; otherwise we skip.
                $this->repointFlexible(Consultation::class, $source->id, $target->id);          // patient_id?
                $this->repointFlexible(Triage::class, $source->id, $target->id);                // patient_id?
                $this->repointFlexible(DoctorNote::class, $source->id, $target->id);            // patient_id?
                $this->repointFlexible(Diagnosis::class, $source->id, $target->id);             // may NOT have patient_id; will auto-skip
                $this->repointFlexible(Prescription::class, $source->id, $target->id);          // patient_id?
                $this->repointFlexible(LabReferral::class, $source->id, $target->id);           // patient_id?
                $this->repointFlexible(ImagingReferral::class, $source->id, $target->id);       // patient_id?
                $this->repointFlexible(Dispensation::class, $source->id, $target->id);          // patient_id?
                $this->repointFlexible(DispensationCorrection::class, $source->id, $target->id);// patient_id?
                $this->repointFlexible(MedicationRequest::class, $source->id, $target->id);     // patient_id?

                // 3) Promote scalar fields
                $this->promoteScalarFields($target, $source);

                // 4) Delete the source record
                $source->delete();
            }

            $target->save();
        });

        return back()->with('success', 'Merge completed successfully. All selected records now belong to the chosen patient.');
    }

    /**
     * Re-point any model/table that (likely) has a FK to patients.
     * - Detects the table name from the Eloquent model if available.
     * - Picks the first matching FK from a candidate list that actually exists on the table.
     * - If no FK candidate exists, it logs and skips silently (so it won't throw a 1054 error).
     *
     * You can pass custom FK candidates; if omitted we use a sensible default list.
     */
    protected function repointFlexible(string $modelClassOrTable, int $fromId, int $toId, array $fkCandidates = null): void
    {
        // Determine table name
        if (class_exists($modelClassOrTable)) {
            $model = app($modelClassOrTable);
            $table = $model->getTable();
        } else {
            $table = $modelClassOrTable; // raw table name
        }

        if (!Schema::hasTable($table)) {
            Log::warning("Duplicate merge: table '{$table}' does not exist; skipping.");
            return;
        }

        // Default candidates: try the most common patterns first
        $fkCandidates = $fkCandidates ?: [
            'patient_id',
            'patientId',
            'member_id',
            'person_id',
            'beneficiary_id',
            'subject_id',
            // Add any project-specific variants here if you know them
        ];

        // Pick the first candidate that exists on this table
        $columns = Schema::getColumnListing($table);
        $fk = null;
        foreach ($fkCandidates as $cand) {
            if (in_array($cand, $columns, true)) {
                $fk = $cand;
                break;
            }
        }

        if (!$fk) {
            // No direct FK to patients found; safely skip
            Log::info("Duplicate merge: no patient FK found on '{$table}'; skipping.");
            return;
        }

        // Perform update; use query builder to avoid model events
        DB::table($table)
            ->where($fk, $fromId)
            ->update([$fk => $toId]);
    }

    /**
     * Promote non-empty scalar fields from $source into $target when target fields are empty.
     * Only touches fillable attributes to keep behavior predictable.
     */
    protected function promoteScalarFields(Patient $target, Patient $source): void
    {
        $fillable = $target->getFillable();

        foreach ($fillable as $key) {
            if (in_array($key, ['id', 'parent_patient_id'], true)) {
                continue;
            }

            $t = $target->getAttribute($key);
            $s = $source->getAttribute($key);

            $tEmpty = $this->isEmptyScalar($t);
            $sHas   = !$this->isEmptyScalar($s);

            if ($tEmpty && $sHas) {
                $target->setAttribute($key, $s);
            }
        }
    }

    protected function isEmptyScalar($v): bool
    {
        if ($v === null) return true;
        if (is_string($v) && trim($v) === '') return true;
        return false;
    }
}
