<?php

namespace App\Http\Controllers;

use App\Mail\DailyNurseConsultationsMail;
use App\Models\Consultation;
use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class EmployeeReportsController extends Controller
{
    /**
     * Stream an Excel workbook of consultations per nurse for a given day.
     * Query params (optional):
     *   - date=YYYY-MM-DD (defaults to today)
     *   - clinic_id
     *   - nurse_id (mapped to doctor_id in Consultation)
     *   - search (patient/diagnosis/clinic/company)
     */
    public function nurseDailyConsultationsExcel(Request $request)
    {
        $this->guardZip();

        [$date, $dayLabel, $clinicId, $nurseId, $search] = $this->params($request);

        $rows = $this->fetchConsultations($date, $clinicId, $nurseId, $search);

        // Parents for dependents
        $parentsById = $this->fetchAccountHolders($rows);

        $filters = ["Date: {$dayLabel}"];
        if ($clinicId) $filters[] = "Clinic ID: {$clinicId}";
        if ($nurseId)  $filters[] = "Nurse(User) ID: {$nurseId}";
        if ($search)   $filters[] = "Search: {$search}";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Consultations_By_Nurse_%s.xlsx', $date->format('Y_m_d'));

        return response()->streamDownload(function () use ($rows, $filtersLine, $parentsById) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            $byNurse = $rows->groupBy(fn ($c) => optional($c->doctor)->name ?? 'Unknown');

            if ($byNurse->isEmpty()) {
                $writer->getCurrentSheet()->setName($this->safeSheetName('No Data'));
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['DAILY CONSULTATIONS — PER NURSE'],
                    ['Generated At', now()->format('Y-m-d H:i:s')],
                    $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                ]);
                $writer->addRow(Row::fromValues(['No consultations found for this day.']));
                $writer->close();
                return;
            }

            $usedSheets = [];

            foreach ($byNurse as $nurseName => $dataset) {
                if (count($usedSheets) === 0) {
                    $writer->getCurrentSheet()->setName($this->safeSheetName($nurseName));
                    $usedSheets[$nurseName] = true;
                } else {
                    $writer->addNewSheetAndMakeItCurrent();
                    $sheet = $this->uniqueSheetName($usedSheets, $this->safeSheetName($nurseName));
                    $writer->getCurrentSheet()->setName($sheet);
                }

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ["DAILY CONSULTATIONS — {$nurseName}"],
                    ['Generated At', now()->format('Y-m-d H:i:s')],
                    $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                ]);

                $this->writeConsultationsTable($writer, $dataset, $headerStyle, $wrapStyle, [
                    'parentsById' => $parentsById
                ]);

                $this->writeConsultationsSummary($writer, $dataset);
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /**
     * Generate the per-nurse workbook for TODAY and email it to fixed recipients.
     * (Used by the daily scheduler at 17:30 Africa/Harare.)
     */
    public function emailNurseDailyConsultations(Request $request)
{
    $this->guardZip();

    // --- Single TO + BCC ---
    $to  = 'jane@providencehumancapital.com';
    $bcc = ['gashiratafadzwa@gmail.com'];

    [$date, $dayLabel, $clinicId, $nurseId, $search] = $this->params($request);

    $rows        = $this->fetchConsultations($date, $clinicId, $nurseId, $search);
    $parentsById = $this->fetchAccountHolders($rows);

    $filters = ["Date: {$dayLabel}"];
    if ($clinicId) $filters[] = "Clinic ID: {$clinicId}";
    if ($nurseId)  $filters[] = "Nurse(User) ID: {$nurseId}";
    if ($search)   $filters[] = "Search: {$search}";
    $filtersLine = implode(' | ', $filters);

    // Build workbook on disk
    $fileName = sprintf('Consultations_By_Nurse_%s.xlsx', $date->format('Y_m_d'));
    $dir      = 'reports/consultations/daily';
    Storage::makeDirectory($dir);
    $path = Storage::path($dir . '/' . $fileName);

    $writer = new XLSXWriter();
    $writer->openToFile($path);

    $titleStyle  = (new Style())->setFontBold();
    $labelStyle  = (new Style())->setFontBold();
    $headerStyle = (new Style())->setFontBold();
    $wrapStyle   = (new Style())->setShouldWrapText(true);

    $byNurse = $rows->groupBy(fn ($c) => optional($c->doctor)->name ?? 'Unknown');

    if ($byNurse->isEmpty()) {
        $writer->getCurrentSheet()->setName($this->safeSheetName('No Data'));
        $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
            ['DAILY CONSULTATIONS — PER NURSE'],
            ['Generated At', now()->format('Y-m-d H:i:s')],
            $filtersLine !== '' ? ['Filters', $filtersLine] : null,
        ]);
        $writer->addRow(Row::fromValues(['No consultations found for this day.']));
        $writer->close();
    } else {
        $used = [];
        foreach ($byNurse as $nurseName => $dataset) {
            if (count($used) === 0) {
                $writer->getCurrentSheet()->setName($this->safeSheetName($nurseName));
                $used[$nurseName] = true;
            } else {
                $writer->addNewSheetAndMakeItCurrent();
                $sheet = $this->uniqueSheetName($used, $this->safeSheetName($nurseName));
                $writer->getCurrentSheet()->setName($sheet);
            }

            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ["DAILY CONSULTATIONS — {$nurseName}"],
                ['Generated At', now()->format('Y-m-d H:i:s')],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);

            $this->writeConsultationsTable($writer, $dataset, $headerStyle, $wrapStyle, [
                'parentsById' => $parentsById
            ]);

            $this->writeConsultationsSummary($writer, $dataset);
        }
        $writer->close();
    }

    // Email the workbook (TO + BCC)
    $mailable = new DailyNurseConsultationsMail($date, $path, $fileName, $filtersLine);
    Mail::to($to)->bcc($bcc)->send($mailable);

    // Clean up (keep in local for inspection if desired)
    try {
        if (! app()->environment('local')) {
            @unlink($path);
        }
    } catch (\Throwable $e) {
        // ignore
    }

    return response()->json([
        'ok'       => true,
        'date'     => $date->toDateString(),
        'sent_to'  => $to,
        'bcc'      => $bcc,
    ]);
}

    /* ========================= Internals / Helpers ========================= */

    /**
     * Common param handling. Returns array: [$date, $dayLabel, $clinicId, $nurseId, $search]
     */
    private function params(Request $request): array
    {
        $dateStr  = $request->query('date'); // optional
        $date     = $dateStr ? Carbon::parse($dateStr) : now();
        $dayLabel = $date->isoFormat('ddd, D MMM YYYY');

        $clinicId = $request->query('clinic_id');
        $nurseId  = $request->query('nurse_id'); // mapped to doctor_id in Consultation
        $search   = $request->query('search');

        return [$date, $dayLabel, $clinicId, $nurseId, $search];
        }

    /**
     * Fetches consultations for a given day with necessary relations & filters.
     */
    private function fetchConsultations(Carbon $date, $clinicId = null, $nurseId = null, $search = null)
    {
        return Consultation::query()
            ->with([
                'patient:id,first_name,middle_name,surname,gender,phone,employee_number,company_id,parent_patient_id,relationship',
                'patient.company:id,name',
                'clinic:id,name,city',
                'doctor:id,name',
            ])
            ->whereDate('consultation_date', $date->toDateString())
            ->when($clinicId, fn ($q) => $q->where('clinic_id', $clinicId))
            ->when($nurseId,  fn ($q) => $q->where('doctor_id', $nurseId))
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('diagnosis', 'like', "%{$search}%")
                       ->orWhere('presenting_complaint', 'like', "%{$search}%")
                       ->orWhere('clinical_notes', 'like', "%{$search}%")
                       ->orWhereHas('patient', function ($p) use ($search) {
                           $p->where('first_name', 'like', "%{$search}%")
                             ->orWhere('middle_name', 'like', "%{$search}%")
                             ->orWhere('surname', 'like', "%{$search}%")
                             ->orWhere('employee_number', 'like', "%{$search}%");
                       })
                       ->orWhereHas('patient.company', fn ($c) => $c->where('name', 'like', "%{$search}%"))
                       ->orWhereHas('clinic', fn ($c) => $c->where('name', 'like', "%{$search}%"));
                });
            })
            ->orderBy('consultation_date')
            ->get();
    }

    /**
     * Preload Account Holders for dependents to avoid N+1.
     */
    private function fetchAccountHolders($rows)
    {
        $parentIds = $rows->pluck('patient.parent_patient_id')->filter()->unique()->values();

        return $parentIds->isNotEmpty()
            ? Patient::whereIn('id', $parentIds)
                ->get(['id', 'first_name', 'surname', 'employee_number'])
                ->keyBy('id')
            : collect();
    }

    /**
     * Table writer: Patient + Consultation details.
     * Includes dependent flag and account holder info where applicable.
     */
    private function writeConsultationsTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle,
        array $options = []
    ): void {
        $parentsById = $options['parentsById'] ?? collect();

        $headers = [
            'Date/Time',
            'Clinic',
            // Patient core
            'First Name',
            'Middle',
            'Surname',
            'Gender',
            'Phone',
            'Employee #',
            'Company',
            'Dependent?',
            'Account Holder',
            'Account Holder Emp#',
            // Consultation
            'IOD',
            'Diagnosis',
            'Sick Leave',
            'Sick Days',
            'Sick Reason',
            'Sick Notes',
            'Ref Radiology',
            'Ref Lab',
            'Ref Specialist',
            'Specialist Name',
            'Ref Casualty',
            'Ref Out-Patient',
            'Referral Notes',
        ];

        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Column widths (optional, improves readability)
        $opts = $writer->getOptions();
        $setW = function (string $label, int $width) use ($headers, $opts) {
            $idx = array_search($label, $headers, true);
            if ($idx !== false) $opts->setColumnWidth($width, $idx + 1);
        };
        $setW('Date/Time', 20);
        $setW('Clinic', 22);
        $setW('First Name', 18);
        $setW('Surname', 18);
        $setW('Phone', 18);
        $setW('Company', 24);
        $setW('Diagnosis', 36);
        $setW('Sick Notes', 36);
        $setW('Referral Notes', 36);
        $setW('Specialist Name', 20);

        foreach ($dataset as $c) {
            $patient   = $c->patient;
            $clinic    = $c->clinic;
            $company   = optional($patient?->company)->name ?? '';

            $isDependent = !empty($patient?->parent_patient_id);
            $holder      = $isDependent ? $parentsById->get($patient->parent_patient_id) : null;
            $holderName  = $holder ? trim(($holder->first_name ?? '') . ' ' . ($holder->surname ?? '')) : '';
            $holderEmp   = $holder->employee_number ?? '';

            $row = [
                $this->fmtDateTime($c->consultation_date),
                $this->clinicLabel($clinic),
                // Patient
                $patient->first_name ?? '',
                $patient->middle_name ?? '',
                $patient->surname ?? '',
                (string)($patient->gender ?? ''),
                (string)($patient->phone ?? ''),
                (string)($patient->employee_number ?? ''),
                $company,
                $isDependent ? 'Yes' : 'No',
                $holderName,
                $holderEmp,
                // Consultation
                $c->injury_on_duty ? '✓' : '',
                (string)($c->diagnosis ?? ''),
                $c->sick_leave ? '✓' : '',
                (int)($c->number_of_days ?? 0),
                (string)($c->reason_for_leave ?? ''),
                (string)($c->sick_leave_notes ?? ''),
                $c->refer_for_radiology ? '✓' : '',
                $c->refer_for_lab ? '✓' : '',
                $c->refer_to_specialist ? '✓' : '',
                (string)($c->specialist_name ?? ''),
                $c->refer_to_casualty ? '✓' : '',
                $c->refer_out_patient ? '✓' : '',
                (string)($c->referral_notes ?? ''),
            ];

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    /**
     * Per-nurse summary footer: totals for sick leave, days, IODs, and referrals.
     */
    private function writeConsultationsSummary(XLSXWriter $writer, $dataset): void
    {
        $total        = $dataset->count();
        $sickCount    = $dataset->where('sick_leave', true)->count();
        $sickDays     = (int) $dataset->sum('number_of_days');
        $iodCount     = $dataset->where('injury_on_duty', true)->count();

        $refRadiology = $dataset->where('refer_for_radiology', true)->count();
        $refLab       = $dataset->where('refer_for_lab', true)->count();
        $refSpec      = $dataset->where('refer_to_specialist', true)->count();
        $refCasualty  = $dataset->where('refer_to_casualty', true)->count();
        $refOut       = $dataset->where('refer_out_patient', true)->count();

        $writer->addRow(Row::fromValues([]));
        $writer->addRow(Row::fromValues(['Summary']));
        $writer->addRow(Row::fromValues(['Total Consultations', $total]));
        $writer->addRow(Row::fromValues(['Sick Leave Count', $sickCount]));
        $writer->addRow(Row::fromValues(['Total Sick Leave Days', $sickDays]));
        $writer->addRow(Row::fromValues(['Injury On Duty (IOD) Count', $iodCount]));
        $writer->addRow(Row::fromValues(['Referrals — Radiology', $refRadiology]));
        $writer->addRow(Row::fromValues(['Referrals — Lab', $refLab]));
        $writer->addRow(Row::fromValues(['Referrals — Specialist', $refSpec]));
        $writer->addRow(Row::fromValues(['Referrals — Casualty', $refCasualty]));
        $writer->addRow(Row::fromValues(['Referrals — Out-Patient', $refOut]));
    }

    private function clinicLabel($clinic): string
    {
        if (!$clinic) return '';
        $name = (string)($clinic->name ?? '');
        $city = (string)($clinic->city ?? '');
        return trim($name . ($city ? " ({$city})" : ''));
    }

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /** Header block helper. */
    private function writeSheetHeader(XLSXWriter $writer, Style $titleStyle, Style $labelStyle, array $lines): void
    {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }

    /** Normalize formatting of date/datetime to Y-m-d H:i */
    private function fmtDateTime($value): string
    {
        if (!$value) return '';
        if ($value instanceof \DateTimeInterface) {
            return $value->format('Y-m-d H:i');
        }
        try {
            return Carbon::parse($value)->format('Y-m-d H:i');
        } catch (\Throwable $e) {
            return (string) $value;
        }
    }

    /** Guard: OpenSpout requires ZipArchive */
    private function guardZip(): void
    {
        if (!class_exists(\ZipArchive::class)) {
            abort(500, 'PHP zip extension is not enabled. Enable "extension=zip" in your php.ini and restart the web server.');
        }
    }
}
