<?php

namespace App\Http\Controllers\External;

use App\Http\Controllers\Controller;
use App\Models\Patient;
use Illuminate\Http\Request;

class PatientLookupController extends Controller
{
    /**
     * GET /api/external/v1/patients/lookup
     * Query params (one of):
     *  - employee_number
     *  - id_number
     *
     * 200: { found: true, patient: {...} }
     * 404: { found: false, message: "Patient not found" }
     */
    public function lookup(Request $request)
    {
        $request->validate([
            'employee_number' => ['nullable','string','max:100'],
            'id_number'       => ['nullable','string','max:100'],
        ]);

        $emp = trim((string)$request->query('employee_number', ''));
        $nid = trim((string)$request->query('id_number', ''));

        if ($emp === '' && $nid === '') {
            return response()->json([
                'error'   => 'BadRequest',
                'message' => 'Provide employee_number or id_number.',
            ], 400);
        }

        $query = Patient::query()->with([
            'company:id,name,company_email,contact_number',
            'dependents:id,first_name,middle_name,surname,parent_patient_id,date_of_birth,gender,id_number',
        ]);

        if ($emp !== '') {
            $query->where('employee_number', $emp);
        } else {
            $query->where('id_number', $nid);
        }

        $patient = $query->first();

        if (!$patient) {
            return response()->json([
                'found'   => false,
                'message' => 'Patient not found',
            ], 404);
        }

        return response()->json([
            'found'   => true,
            'patient' => $this->serializePatient($patient),
        ]);
    }

    /**
     * NEW:
     * GET /api/external/v1/patients
     * Optional query params:
     *  - page (default 1)
     *  - per_page (default 50, max 200)
     *  - search (optional, matches first_name/surname/id_number/employee_number)
     *
     * 200:
     * {
     *   "data": [ {...patient...}, ... ],
     *   "meta": { "current_page":1, "per_page":50, "total": 123, ... }
     * }
     */
    public function list(Request $request)
    {
        $perPage = (int) $request->query('per_page', 50);
        $perPage = max(1, min(200, $perPage)); // clamp 1..200

        $search = trim((string)$request->query('search', ''));

        $query = Patient::query()
            ->with([
                'company:id,name,company_email,contact_number',
                'dependents:id,first_name,middle_name,surname,parent_patient_id,date_of_birth,gender,id_number',
            ])
            ->orderByDesc('updated_at')
            ->orderByDesc('created_at');

        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('surname', 'like', "%{$search}%")
                  ->orWhere('id_number', 'like', "%{$search}%")
                  ->orWhere('employee_number', 'like', "%{$search}%");
            });
        }

        $paginator = $query->paginate($perPage)->appends($request->query());

        // map each patient to the same shape as lookup()
        $paginator->getCollection()->transform(function (Patient $p) {
            return $this->serializePatient($p);
        });

        return response()->json([
            'data' => $paginator->items(),
            'meta' => [
                'current_page' => $paginator->currentPage(),
                'per_page'     => $paginator->perPage(),
                'from'         => $paginator->firstItem(),
                'to'           => $paginator->lastItem(),
                'total'        => $paginator->total(),
                'last_page'    => $paginator->lastPage(),
                'has_more'     => $paginator->hasMorePages(),
            ],
        ]);
    }

    /** Reuse the same shape everywhere */
    private function serializePatient(Patient $patient): array
    {
        return [
            'id'                 => $patient->id,
            'first_name'         => $patient->first_name,
            'middle_name'        => $patient->middle_name,
            'surname'            => $patient->surname,
            'full_name'          => $patient->full_name,
            'employee_number'    => $patient->employee_number,
            'id_number'          => $patient->id_number,
            'date_of_birth'      => optional($patient->date_of_birth)->format('Y-m-d'),
            'gender'             => $patient->gender,
            'email'              => $patient->email,
            'phone'              => $patient->phone,
            'home_address'       => $patient->home_address,
            'suburb'             => $patient->suburb,
            'work_area'          => $patient->work_area,
            'marital_status'     => $patient->marital_status,
            'occupation'         => $patient->occupation,
            'is_chronic_patient' => (bool)$patient->is_chronic_patient,
            'emp_start_date'     => $patient->emp_start_date ?? null,
            'emp_end_date'       => $patient->emp_end_date ?? null,

            'company' => $patient->company ? [
                'id'    => $patient->company->id,
                'name'  => $patient->company->name ?? $patient->company->company_name ?? null,
                'email' => $patient->company->company_email ?? null,
                'phone' => $patient->company->contact_number ?? null,
            ] : null,

            'dependents' => $patient->dependents->map(fn($d) => [
                'id'            => $d->id,
                'first_name'    => $d->first_name,
                'middle_name'   => $d->middle_name,
                'surname'       => $d->surname,
                'id_number'     => $d->id_number,
                'date_of_birth' => optional($d->date_of_birth)->format('Y-m-d'),
                'gender'        => $d->gender,
            ])->values(),
        ];
    }
}
