<?php
namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\ClinicMedicationStock;
use App\Models\NurseMedicationStock;
use App\Models\MedicationBatch;
use Carbon\Carbon;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class FileReportController extends Controller
{
    public function consultationsMonthly(Request $request)
    {
        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month, 422, 'Both year and month are required.');

        $user       = auth()->user();
        $isNurse    = $user && ($user->role ?? null) === 'nurse';

        $search      = $request->query('search');
        $startDate   = $request->query('start_date');
        $endDate     = $request->query('end_date');
        $clinicId    = $request->query('clinic_id');
        $doctorId    = $request->query('doctor_id');
        $patientType = $request->query('patient_type');
        $period      = $request->query('period');

        // Base query with relevant relations
        $query = Consultation::query()
            ->with([
                'patient.company:id,name',
                'doctor:id,name',
                'clinic:id,name',

                // Dispensations + who dispensed + nested med batch->medication
                'dispensations.dispensedBy:id,name',
                'dispensations.clinicMedicationStock.medicationBatch.medication:id,name',
                'dispensations.nurseMedicationStock.medicationBatch.medication:id,name',
                'dispensations.stock.medicationBatch.medication:id,name',
            ])
            ->whereYear('consultation_date', $year)
            ->whereMonth('consultation_date', $month)
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('diagnosis', 'like', "%{$search}%")
                        ->orWhere('treatment_plan', 'like', "%{$search}%")
                        ->orWhereHas('clinic', fn($c) => $c->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('doctor', fn($d) => $d->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('patient', function ($p) use ($search) {
                            $p->where('first_name', 'like', "%{$search}%")
                              ->orWhere('surname', 'like', "%{$search}%");
                        });
                });
            })
            ->when($startDate && $endDate, function ($q) use ($startDate, $endDate) {
                $q->whereBetween('consultation_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            ->when($doctorId, fn($q) => $q->where('doctor_id', $doctorId))
            ->when($patientType, function ($q) use ($patientType) {
                if ($patientType === 'dependents') {
                    $q->whereHas('patient', fn($p) => $p->whereNotNull('parent_patient_id'));
                } elseif ($patientType === 'non-dependents') {
                    $q->whereHas('patient', fn($p) => $p->whereNull('parent_patient_id'));
                }
            });

        // If nurse: only consultations where THIS nurse dispensed
        if ($isNurse) {
            $query->whereHas('dispensations', fn($d) => $d->where('dispensed_by', $user->id));
        }

        // Optional additional 'period' filter
        if ($period === 'daily') {
            $query->whereDate('consultation_date', Carbon::create($year, $month, 1)->toDateString());
        } elseif ($period === 'weekly') {
            $query->whereBetween('consultation_date', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
        } elseif ($period === 'quarterly') {
            $quarter = (int) ceil($month / 3);
            $query->whereBetween('consultation_date', [
                Carbon::create($year, ($quarter * 3) - 2, 1)->startOfDay(),
                Carbon::create($year, $quarter * 3, 1)->endOfMonth()->endOfDay(),
            ]);
        }

        $consultations = $query->orderBy('consultation_date')->get();

        // Group by company name (or "No Company")
        $groups = $consultations->groupBy(fn($c) => optional($c->patient->company)->name ?? 'No Company')
                                ->sortKeys();

        $periodLabel   = Carbon::create($year, $month, 1)->isoFormat('MMMM YYYY');
        $generatedAt   = now()->format('Y-m-d H:i:s');
        $filters       = [];
        if ($search)       $filters[] = "Search: {$search}";
        if ($clinicId)     $filters[] = "Clinic ID: {$clinicId}";
        if ($doctorId)     $filters[] = "Doctor ID: {$doctorId}";
        if ($patientType)  $filters[] = "Patient Type: {$patientType}";
        if ($startDate || $endDate) $filters[] = "Custom Range: {$startDate} → {$endDate}";
        if ($period)       $filters[] = "Period filter: {$period}";
        if ($isNurse)      $filters[] = "Scope: This nurse's dispensations";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Consultations_%d_%02d.xlsx', $year, $month);

        return response()->streamDownload(function () use (
            $consultations, $groups, $periodLabel, $generatedAt, $filtersLine, $isNurse, $user
        ) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Set some sensible defaults once for the workbook (OpenSpout Options)
            $opts = $writer->getOptions();
            // (Optional) overall default width so text isn’t too cramped
            // $opts->setColumnWidthForRange(18, 1, 50); // example if you want all first 50 columns at 18

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // ===== Overall sheet =====
            $overallSheetName = $this->safeSheetName('All Consultations');
            $writer->getCurrentSheet()->setName($overallSheetName);

            $includeDispensedBy = !$isNurse;

            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['Consultations Report (All Companies)'],
                ['Period', $periodLabel],
                ['Generated At', $generatedAt],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                $isNurse ? ['Nurse', $user->name ?? ''] : null,
            ]);

            $this->writeTableForDataset(
                $writer,
                $consultations,
                $includeDispensedBy,
                $headerStyle,
                $wrapStyle
            );

            $this->writeFooterCounts($writer, $consultations);

            // ===== One sheet per company =====
            $used = [$overallSheetName => true];
            foreach ($groups as $companyName => $dataset) {
                $writer->addNewSheetAndMakeItCurrent();
                $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($companyName));
                $writer->getCurrentSheet()->setName($sheetName);

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ["Consultations Report — {$companyName}"],
                    ['Period', $periodLabel],
                    ['Generated At', $generatedAt],
                    $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                    $isNurse ? ['Nurse', $user->name ?? ''] : null,
                ]);

                $this->writeTableForDataset(
                    $writer,
                    $dataset,
                    $includeDispensedBy,
                    $headerStyle,
                    $wrapStyle
                );

                $this->writeFooterCounts($writer, $dataset);
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    // ---------- helpers ----------

    private function writeSheetHeader(XLSXWriter $writer, Style $titleStyle, Style $labelStyle, array $lines): void
    {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }

    private function writeTableForDataset(
        XLSXWriter $writer,
        $dataset,
        bool $includeDispensedBy,
        Style $headerStyle,
        Style $wrapStyle
    ): void {
        // Dynamic med columns for THIS sheet
        $maxPairs = $dataset->map(fn($c) => $c->dispensations->count())->max() ?? 0;

        // Header (Consultation Date/Time from created_at)
        $headers = [
            'Consultation Date/Time',
            'First Name',
            'Surname',
            'Gender',
            'Employee Number',
            'Company',
            'Diagnosis',
            'Clinical Notes',
        ];
        if ($includeDispensedBy) {
            $headers[] = 'Dispensed By';
        }
        $headers[] = 'Injury On Duty';
        $headers[] = 'Sick Leave';
        $headers[] = 'Sick Leave Days';

        for ($i = 1; $i <= $maxPairs; $i++) {
            $headers[] = "Medication {$i}";
            $headers[] = "Quantity {$i}";
        }

        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Widen Diagnosis & Clinical Notes columns (1-based column indices)
        // Cols: 1=DateTime, 2=First, 3=Surname, 4=Gender, 5=Emp#, 6=Company, 7=Diagnosis, 8=Clinical Notes
        $opts = $writer->getOptions();
        $opts->setColumnWidth(40, 7); // Diagnosis wider
        $opts->setColumnWidth(60, 8); // Clinical Notes wider

        // Rows
        foreach ($dataset as $c) {
            $p = $c->patient;
            $created = optional($c->created_at)->format('Y-m-d H:i') ?? '';

            $companyName = optional($p->company)->name ?? '';
            $employeeNumber = $p->employee_number
                ?? $p->staff_number
                ?? $p->employee_no
                ?? '';

            $diagnosis     = (string) ($c->diagnosis ?? '');
            $clinicalNotes = (string) ($c->clinical_notes ?? '');

            $injuryOnDuty = (bool) $c->injury_on_duty;
            $sickLeave    = (bool) $c->sick_leave;
            $sickDays     = $sickLeave ? (int) ($c->number_of_days ?? 0) : null;

            // Collect meds & qtys + dispensers
            $meds = [];
            $qtys = [];
            $by   = [];

            foreach ($c->dispensations as $d) {
                $medName = $this->resolveMedicationName($d);
                $qty     = (int) ($d->quantity ?? 0);
                $byName  = optional($d->dispensedBy)->name ?? '';

                $meds[] = $medName;
                $qtys[] = $qty;
                if ($byName !== '') $by[] = $byName;
            }

            // Dispensed By: single if exactly one unique, else distinct list
            $dispensedByCell = '';
            if ($includeDispensedBy) {
                $uniqueBy = [];
                foreach ($by as $name) {
                    if ($name !== '' && !in_array($name, $uniqueBy, true)) $uniqueBy[] = $name;
                }
                $dispensedByCell = count($uniqueBy) === 1 ? ($uniqueBy[0] ?? '') : implode(' | ', $uniqueBy);
            }

            // Pad medication pairs to $maxPairs
            $dynamic = [];
            for ($i = 0; $i < $maxPairs; $i++) {
                $dynamic[] = $meds[$i] ?? '';
                $dynamic[] = $qtys[$i] ?? null;
            }

            $row = [
                $created,
                $p->first_name ?? '',
                $p->surname ?? '',
                $p->gender ?? '',
                $employeeNumber,
                $companyName,
                $diagnosis,
                $clinicalNotes,
            ];
            if ($includeDispensedBy) $row[] = $dispensedByCell;
            $row[] = $injuryOnDuty;
            $row[] = $sickLeave;
            $row[] = $sickDays;
            $row   = array_merge($row, $dynamic);

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    private function writeFooterCounts(XLSXWriter $writer, $dataset): void
    {
        $uniquePeopleCount   = $dataset->pluck('patient_id')->filter()->unique()->count();
        $totalConsultations  = $dataset->count();

        $writer->addRow(Row::fromValues([])); // spacer
        $writer->addRow(Row::fromValues(['Total People Shown', $uniquePeopleCount]));
        $writer->addRow(Row::fromValues(['Total Consultations', $totalConsultations]));
    }

    /** Try eager paths; if empty, perform a single fallback lookup using present foreign keys. */
    private function resolveMedicationName($dispensation): string
    {
        // Eager-loaded paths
        $paths = [
            'clinicMedicationStock.medicationBatch.medication.name',
            'nurseMedicationStock.medicationBatch.medication.name',
            'stock.medicationBatch.medication.name',
        ];
        foreach ($paths as $path) {
            $val = data_get($dispensation, $path);
            if (!empty($val)) return (string) $val;
        }

        // Fallback: direct lookup if IDs exist (handles edge records that missed eager hydration)
        try {
            if ($dispensation->clinic_medication_stock_id) {
                $cms = ClinicMedicationStock::with('medicationBatch.medication')
                    ->find($dispensation->clinic_medication_stock_id);
                $name = optional(optional($cms)->medicationBatch)->medication->name ?? null;
                if ($name) return (string) $name;
            }
            if ($dispensation->nurse_medication_stock_id) {
                $nms = NurseMedicationStock::with('medicationBatch.medication')
                    ->find($dispensation->nurse_medication_stock_id);
                $name = optional(optional($nms)->medicationBatch)->medication->name ?? null;
                if ($name) return (string) $name;
            }
        } catch (\Throwable $e) {
            // swallow and return empty
        }

        return '';
    }

    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }
}
