<?php

namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Consultation;
use App\Models\ImagingReferral;
use App\Models\LabReferral;
use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class ImagingReferralController extends Controller
{
    public function index(Request $request)
    {
        $q       = $request->string('q');
        $status  = $request->string('status');
        $perPage = (int) $request->input('per_page', 20);

        $items = ImagingReferral::query()
            ->with([
                'patient:id,first_name,surname,gender,date_of_birth,employee_number',
                'consultation:id,consultation_date',
                'referrer:id,name'
            ])
            ->when($q, fn ($qq) =>
                $qq->whereHas('patient', fn ($p) =>
                    $p->where('first_name','like',"%{$q}%")
                      ->orWhere('surname','like',"%{$q}%")
                )->orWhere('notes','like',"%{$q}%")
            )
            ->when($status, fn ($qq) => $qq->where('status', $status))
            ->latest()
            ->paginate($perPage)
            ->withQueryString();

        return Inertia::render('Referrals/Imaging/Index', [
            'filters' => compact('q','status','perPage'),
            'items'   => $items,
        ]);
    }

    /** CREATE — consultation & patient via route params */
    // App/Http/Controllers/ImagingReferralController.php

public function create(Consultation $consultation, Patient $patient)
{
    abort_if((int) $consultation->patient_id !== (int) $patient->id, 404);

    // Load patient->company for display
    $patient->loadMissing(['company:id,name']);

    // Latest Lab (unchanged from before)
    $labLatest = \App\Models\LabReferral::query()
        ->where('consultation_id', $consultation->id)
        ->where('patient_id', $patient->id)
        ->latest('id')
        ->first();

    // Build a compact consultation payload with useful clinical fields
    $consultationPayload = [
        'id'                   => $consultation->id,
        'consultation_date'    => optional($consultation->consultation_date)->toIso8601String(),
        // Vitals
        'systolic'             => $consultation->systolic,
        'diastolic'            => $consultation->diastolic,
        'temperature'          => $consultation->temperature,
        'heart_rate'           => $consultation->heart_rate,
        'respiratory_rate'     => $consultation->respiratory_rate,
        'oxygen_saturation'    => $consultation->oxygen_saturation,
        'weight'               => $consultation->weight,
        'height'               => $consultation->height,
        'bmi'                  => $consultation->bmi,
        // Glucose
        'blood_sugar'          => $consultation->blood_sugar,
        'blood_sugar_unit'     => $consultation->blood_sugar_unit,
        'blood_sugar_context'  => $consultation->blood_sugar_context,
        'blood_sugar_measured_at' => optional($consultation->blood_sugar_measured_at)->toIso8601String(),
        // Notes
        'presenting_complaint' => $consultation->presenting_complaint,
        'clinical_notes'       => $consultation->clinical_notes,
        'diagnosis'            => $consultation->diagnosis,
        'treatment_plan'       => $consultation->treatment_plan,
        // Follow-up
        'follow_up_required'   => (bool) $consultation->follow_up_required,
        'follow_up_date'       => optional($consultation->follow_up_date)->toIso8601String(),
        // Flags
        'injury_on_duty'       => (bool) $consultation->injury_on_duty,
        'refer_for_lab'        => (bool) $consultation->refer_for_lab,
        'refer_for_radiology'  => (bool) $consultation->refer_for_radiology,
    ];

    return \Inertia\Inertia::render('Referrals/Imaging/Create', [
        'consultation' => $consultationPayload,
        'patient'      => [
            'id'              => $patient->id,
            'first_name'      => $patient->first_name,
            'surname'         => $patient->surname,
            'gender'          => $patient->gender,
            'date_of_birth'   => $patient->date_of_birth,
            'employee_number' => $patient->employee_number,
            'company'         => $patient->company ? [
                'id'   => $patient->company->id,
                'name' => $patient->company->name,
            ] : null,
        ],
        'defaults'     => [
            'provider'     => 'Baines Imaging Group',
            'requested_at' => now()->format('Y-m-d\TH:i'),
        ],
        'lab_latest'   => $labLatest ? [
            'id'              => $labLatest->id,
            'provider'        => $labLatest->provider,
            'lab_ref_no'      => $labLatest->lab_ref_no,
            'requested_at'    => optional($labLatest->requested_at)->toIso8601String(),
            'status'          => $labLatest->status,
            'tests_requested' => $labLatest->tests_requested,
            'attachment_path' => $labLatest->attachment_path,
        ] : null,
    ]);
}


    public function store(Request $request)
    {
        $data = $request->validate([
            'consultation_id'       => ['required', 'exists:consultations,id'],
            'patient_id'            => ['required', 'exists:patients,id'],
            'provider'              => ['nullable', 'string', 'max:255'],
            'requested_at'          => ['nullable', 'date'],
            'exams'                 => ['nullable', 'array'],
            'exams.*'               => ['string', Rule::in(['ultrasound', 'x_rays', 'ct_scan', 'mri_scanning'])],
            'region_of_interest'    => ['nullable', 'string', 'max:500'],
            'history'               => ['nullable', 'string'],
            'lmp'                   => ['nullable', 'date'],
            'is_pregnant'           => ['nullable', 'boolean'],
            'referring_doctor_name' => ['nullable', 'string', 'max:255'],
            'notes'                 => ['nullable', 'string'],
            'status'                => ['nullable', 'in:draft,sent,completed,cancelled'],
            'attachment'            => ['nullable', 'file', 'max:8192'],

            // Optional “what next”
            'next'                  => ['nullable', Rule::in(['notes','dispense','lab_edit','show'])],
        ]);

        $consultation = Consultation::findOrFail($data['consultation_id']);
        abort_if((int) $consultation->patient_id !== (int) $data['patient_id'], 422, 'Patient does not match the consultation.');

        $path = $request->hasFile('attachment')
            ? $request->file('attachment')->store('referrals/attachments', 'public')
            : null;

        $referral = ImagingReferral::create([
            'consultation_id'       => $data['consultation_id'],
            'patient_id'            => $data['patient_id'],
            'referrer_id'           => auth()->id(),
            'provider'              => $data['provider'] ?? 'Baines Imaging Group',
            'requested_at'          => $data['requested_at'] ?? now(),
            'exams'                 => $data['exams'] ?? [],
            'region_of_interest'    => $data['region_of_interest'] ?? null,
            'history'               => $data['history'] ?? null,
            'lmp'                   => $data['lmp'] ?? null,
            'is_pregnant'           => $data['is_pregnant'] ?? null,
            'referring_doctor_name' => $data['referring_doctor_name'] ?? null,
            'notes'                 => $data['notes'] ?? null,
            'status'                => $data['status'] ?? 'sent',
            'attachment_path'       => $path,
        ]);

        $next     = $request->string('next')->toString(); // 'notes' | 'dispense' | 'lab_edit' | 'show' | ''
        $isDoctor = auth()->user()?->role === UserRole::Doctor;

        if ($next === 'notes' && $isDoctor) {
            return redirect()
                ->route('consultations.doctor-notes.create', ['consultation' => $consultation->id])
                ->with('success', 'Imaging referral created. Proceed to doctor notes.');
        }

        if ($next === 'dispense') {
            $url = route('consultations.dispense', ['consultation' => $consultation->id]);
            if ($request->header('X-Inertia')) {
                return Inertia::location($url);
            }
            return redirect()->to($url);
        }

        if ($next === 'lab_edit') {
            $lab = LabReferral::query()
                ->where('consultation_id', $consultation->id)
                ->where('patient_id', $referral->patient_id)
                ->latest('id')
                ->first();

            if ($lab) {
                return redirect()
                    ->route('referrals.lab.edit', $lab->id)
                    ->with('success', 'Imaging referral created. Proceed to edit lab referral.');
            }

            return redirect()
                ->route('referrals.lab.create', [
                    'consultation' => $consultation->id,
                    'patient'      => $referral->patient_id,
                ])
                ->with('success', 'Imaging referral created. Create a lab referral.');
        }

        return redirect()
            ->route('referrals.imaging.show', $referral)
            ->with('success', 'Imaging referral created.');
    }

    public function show(ImagingReferral $imagingReferral)
    {
        $imagingReferral->load([
            'patient:id,first_name,surname,gender,date_of_birth,employee_number',
            'consultation:id,consultation_date',
            'referrer:id,name'
        ]);

        return Inertia::render('Referrals/Imaging/Show', [
            'referral' => $imagingReferral,
        ]);
    }

    public function edit(ImagingReferral $imagingReferral, ?Consultation $consultation = null, ?Patient $patient = null)
    {
        $imagingReferral->load([
            'patient:id,first_name,surname,gender,date_of_birth,employee_number',
            'consultation:id,consultation_date'
        ]);

        if ($consultation) abort_if((int) $imagingReferral->consultation_id !== (int) $consultation->id, 404);
        if ($patient)      abort_if((int) $imagingReferral->patient_id !== (int) $patient->id, 404);

        return Inertia::render('Referrals/Imaging/Edit', [
            'referral'     => $imagingReferral,
            'consultation' => $imagingReferral->consultation?->only(['id','consultation_date']),
            'patient'      => $imagingReferral->patient?->only(['id','first_name','surname','gender','date_of_birth','employee_number']),
            'defaults'     => [
                'requested_at' => optional($imagingReferral->requested_at)->format('Y-m-d\TH:i'),
            ],
        ]);
    }

    public function update(Request $request, ImagingReferral $imagingReferral)
    {
        $data = $request->validate([
            'provider'              => ['nullable','string','max:255'],
            'requested_at'          => ['nullable','date'],
            'exams'                 => ['nullable','array'],
            'exams.*'               => ['string', Rule::in(['ultrasound','x_rays','ct_scan','mri_scanning'])],
            'region_of_interest'    => ['nullable','string','max:500'],
            'history'               => ['nullable','string'],
            'lmp'                   => ['nullable','date'],
            'is_pregnant'           => ['nullable','boolean'],
            'referring_doctor_name' => ['nullable','string','max:255'],
            'notes'                 => ['nullable','string'],
            'status'                => ['nullable','in:draft,sent,completed,cancelled'],
            'attachment'            => ['nullable','file','max:8192'],
        ]);

        if ($request->hasFile('attachment')) {
            if ($imagingReferral->attachment_path) {
                Storage::disk('public')->delete($imagingReferral->attachment_path);
            }
            $data['attachment_path'] = $request->file('attachment')->store('referrals/attachments', 'public');
        }

        $imagingReferral->fill($data)->save();

        return redirect()->route('referrals.imaging.show', $imagingReferral)->with('success','Imaging referral updated.');
    }

    public function destroy(ImagingReferral $imagingReferral)
    {
        if ($imagingReferral->attachment_path) {
            Storage::disk('public')->delete($imagingReferral->attachment_path);
        }
        $imagingReferral->delete();

        return redirect()->route('referrals.imaging.index')->with('success','Imaging referral deleted.');
    }
}
