<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class InjuryOnDutyController extends Controller
{
    /**
     * Monthly Injury-On-Duty (IOD) report (Employees only):
     * - Required: year, month
     * - Optional: clinic_id, search (patient name / employee # / diagnosis / company)
     * - Scopes to clinics the authenticated user can access (superadmin can see all)
     * - Sheets: "All Clinics" + one per clinic ("Name (City)")
     * - Columns: Date, FirstName, Surname, Gender, Employee #, Company, [Diagnosis], [Notes], Clinic
     *   (Diagnosis & Notes are INCLUDED ONLY for doctor/nurse; hidden for other roles)
     */
    public function monthly(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $role = (string) ($user->role?->value ?? $user->role ?? '');
        $includeDxNotes = in_array($role, ['doctor', 'nurse'], true);
        $isSuperadmin   = ($role === 'superadmin');

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month, 422, 'Year and month are required.');

        $clinicId = $request->query('clinic_id'); // optional
        $search   = $request->query('search');    // optional

        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end   = (clone $start)->endOfMonth();
        $periodLabel = $start->isoFormat('MMMM YYYY');

        // Determine clinic scope
        $allowedClinicIds = $this->clinicIdsFor($user);

        // If an explicit clinic filter is given, enforce access unless superadmin
        if ($clinicId) {
            $clinicId = (int) $clinicId;
            if (!$isSuperadmin && !in_array($clinicId, $allowedClinicIds, true)) {
                // No access to the requested clinic -> return an empty workbook (with headers)
                $rows = collect();
                return $this->streamMonthlyWorkbook($rows, $periodLabel, $clinicId, $search, $includeDxNotes);
            }
        }

        // Build scoped query
        $rows = Consultation::query()
            ->with([
                'patient:id,first_name,surname,gender,employee_number,parent_patient_id,company_id',
                'patient.company:id,name',
                'clinic:id,name,city',
            ])
            ->where('injury_on_duty', 1)
            // Employees only
            ->whereHas('patient', fn($p) => $p->whereNull('parent_patient_id'))
            // Month filter: prefer consultation_date, fallback to created_at if null
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('consultation_date', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->whereNull('consultation_date')
                         ->whereBetween('created_at', [$start, $end]);
                  });
            })
            // Optional explicit clinic_id (still respects access above)
            ->when($clinicId, fn($q) => $q->where('clinic_id', $clinicId))
            // Scope to clinics the user can access (superadmin sees all when no explicit clinic)
            ->when(!$isSuperadmin && !$clinicId, fn($q) => $q->whereIn('clinic_id', $allowedClinicIds ?: [-1]))
            // Search (patient/company/diagnosis/notes)
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->orWhereHas('patient', function ($p) use ($search) {
                            $p->where('first_name', 'like', "%{$search}%")
                              ->orWhere('surname', 'like', "%{$search}%")
                              ->orWhere('employee_number', 'like', "%{$search}%");
                        })
                        ->orWhereHas('patient.company', fn($c) =>
                            $c->where('name', 'like', "%{$search}%")
                        )
                        ->orWhere('diagnosis', 'like', "%{$search}%")
                        ->orWhere('clinical_notes', 'like', "%{$search}%")
                        ->orWhere('presenting_complaint', 'like', "%{$search}%");
                });
            })
            ->orderBy('consultation_date')
            ->orderBy('created_at')
            ->get();

        return $this->streamMonthlyWorkbook($rows, $periodLabel, $clinicId, $search, $includeDxNotes);
    }

    /** Stream the XLSX workbook for the given dataset. */
    private function streamMonthlyWorkbook($rows, string $periodLabel, $clinicId, ?string $search, bool $includeDxNotes)
    {
        // Group by clinic label for per-clinic sheets
        $byClinic = $rows->groupBy(function ($c) {
            $clinic = $c->clinic;
            if (!$clinic) return 'Unknown Clinic';
            $name = (string) ($clinic->name ?? '');
            $city = (string) ($clinic->city ?? '');
            return trim($name . ($city ? " ({$city})" : ''));
        })->sortKeys();

        $generatedAt = now()->format('Y-m-d H:i:s');
        $filters = ["Period: {$periodLabel}"];
        if ($clinicId) $filters[] = "Clinic ID: {$clinicId}";
        if ($search)   $filters[] = "Search: {$search}";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Injury_On_Duty_%s.xlsx', str_replace(' ', '_', $periodLabel));

        return response()->streamDownload(function () use ($rows, $byClinic, $generatedAt, $filtersLine, $includeDxNotes) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // ===== All Clinics =====
            $writer->getCurrentSheet()->setName($this->safeSheetName('All Clinics'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['INJURY ON DUTY — ALL CLINICS'],
                ['Generated At', $generatedAt],
                ['Filters', $filtersLine],
            ]);
            $this->writeIodTable($writer, $rows, $headerStyle, $wrapStyle, true, $includeDxNotes);
            $this->writeIodTotals($writer, $rows);

            // ===== Per Clinic =====
            $used = ['All Clinics' => true];
            foreach ($byClinic as $clinicLabel => $dataset) {
                $writer->addNewSheetAndMakeItCurrent();
                $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($clinicLabel));
                $writer->getCurrentSheet()->setName($sheetName);

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ["INJURY ON DUTY — {$clinicLabel}"],
                    ['Generated At', $generatedAt],
                    ['Filters', $filtersLine],
                ]);
                // On per-clinic sheets, omit the Clinic column (implied)
                $this->writeIodTable($writer, $dataset, $headerStyle, $wrapStyle, false, $includeDxNotes);
                $this->writeIodTotals($writer, $dataset);
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /** Table for IOD rows
     *  @param bool $includeClinic   Include the Clinic column (true for "All Clinics", false per clinic)
     *  @param bool $includeDxNotes  Include Diagnosis & Notes columns (only for doctor/nurse)
     */
    private function writeIodTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle,
        bool $includeClinic,
        bool $includeDxNotes
    ): void {
        $headers = [
            'Date',
            'Patient FirstName',
            'Patient Surname',
            'Gender',
            'Employee #',
            'Company',
        ];

        if ($includeDxNotes) {
            $headers[] = 'Diagnosis';
            $headers[] = 'Notes';
        }

        if ($includeClinic) {
            $headers[] = 'Clinic';
        }

        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Column widths
        $opts = $writer->getOptions();
        $setW = function (string $label, int $width) use ($headers, $opts) {
            $idx = array_search($label, $headers, true);
            if ($idx !== false) $opts->setColumnWidth($width, $idx + 1);
        };
        $setW('Date', 14);
        $setW('Patient FirstName', 24);
        $setW('Patient Surname', 24);
        $setW('Gender', 10);
        $setW('Employee #', 16);
        $setW('Company', 26);
        if ($includeDxNotes) {
            $setW('Diagnosis', 40);
            $setW('Notes', 40);
        }
        if ($includeClinic) $setW('Clinic', 24);

        foreach ($dataset as $c) {
            $patient  = $c->patient;
            $company  = optional($patient->company)->name ?? '';
            $clinic   = $this->resolveClinicLabelFromConsultation($c);

            $dateCell = optional($c->consultation_date)->format('Y-m-d')
                ?? (optional($c->created_at)->format('Y-m-d') ?? '');

            // Prefer clinical_notes; fallback to presenting_complaint
            $notes = (string) ($c->clinical_notes ?? $c->presenting_complaint ?? '');

            $row = [
                $dateCell,
                (string) ($patient->first_name ?? ''),
                (string) ($patient->surname ?? ''),
                (string) ($patient->gender ?? ''),
                (string) ($patient->employee_number ?? ''),
                (string) $company,
            ];

            if ($includeDxNotes) {
                $row[] = (string) ($c->diagnosis ?? '');
                $row[] = $notes;
            }

            if ($includeClinic) {
                $row[] = $clinic;
            }

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    private function writeIodTotals(XLSXWriter $writer, $dataset): void
    {
        $writer->addRow(Row::fromValues([]));
        $writer->addRow(Row::fromValues(['Total IOD Cases', $dataset->count()]));
    }

    /** "Clinic Name (City)" from Consultation relation. */
    private function resolveClinicLabelFromConsultation($consultation): string
    {
        $c = $consultation->clinic;
        if (!$c) return '';
        $name = (string) ($c->name ?? '');
        $city = (string) ($c->city ?? '');
        return trim($name . ($city ? " ({$city})" : ''));
    }

    /* ---------------- helpers ---------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }

    /** Get clinic IDs the user can access (primary + pivot). */
    private function clinicIdsFor($user): array
    {
        if (method_exists($user, 'accessibleClinicIds')) {
            $ids = $user->accessibleClinicIds() ?: [];
            return array_values(array_unique(array_map('intval', $ids)));
        }

        $ids = [];
        if (method_exists($user, 'accessibleClinics')) {
            $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
        }
        if (!empty($user->clinic_id) && !in_array((int)$user->clinic_id, $ids, true)) {
            $ids[] = (int)$user->clinic_id;
        }
        return array_values(array_unique(array_map('intval', $ids)));
    }
}
