<?php

namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Consultation;
use App\Models\ImagingReferral;
use App\Models\LabReferral;
use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class LabReferralController extends Controller
{
    public function index(Request $request)
    {
        $q       = $request->string('q');
        $status  = $request->string('status');
        $perPage = (int) $request->input('per_page', 20);

        $items = LabReferral::query()
            ->with([
                'patient:id,first_name,surname,gender,date_of_birth,employee_number',
                'consultation:id,consultation_date',
                'referrer:id,name'
            ])
            ->when($q, fn($qq) =>
                $qq->whereHas('patient', fn($p) =>
                    $p->where('first_name', 'like', "%{$q}%")
                      ->orWhere('surname', 'like', "%{$q}%")
                )->orWhere('tests_requested', 'like', "%{$q}%")
            )
            ->when($status, fn($qq) => $qq->where('status', $status))
            ->latest()
            ->paginate($perPage)
            ->withQueryString();

        return Inertia::render('Referrals/Lab/Index', [
            'filters' => compact('q', 'status', 'perPage'),
            'items'   => $items,
        ]);
    }

    /** CREATE — consultation & patient via route params */
    public function create(Consultation $consultation, Patient $patient)
    {
        abort_if((int) $consultation->patient_id !== (int) $patient->id, 404);

        // include the patient's company
        $patient->load('company:id,name');

        // latest imaging referral for this consultation & patient
        $imaging = ImagingReferral::query()
            ->where('consultation_id', $consultation->id)
            ->where('patient_id', $patient->id)
            ->latest('id')
            ->first(['id','provider','requested_at','exams','status','notes','attachment_path']);

        return Inertia::render('Referrals/Lab/Create', [
            // richer consultation context (safe subset)
            'consultation' => $consultation->only([
                'id', 'consultation_date',
                // vitals
                'systolic','diastolic','temperature','heart_rate',
                'respiratory_rate','oxygen_saturation','weight','height','bmi',
                // glucose
                'blood_sugar','blood_sugar_unit','blood_sugar_context','blood_sugar_measured_at',
                // clinical notes
                'presenting_complaint','diagnosis','treatment_plan',
                // follow-ups & flags
                'follow_up_required','follow_up_date','injury_on_duty',
                'refer_for_lab','refer_for_radiology',
            ]),
            // patient plus company
            'patient'      => array_merge(
                $patient->only(['id','first_name','surname','gender','date_of_birth','employee_number']),
                ['company' => $patient->company ? $patient->company->only(['id','name']) : null]
            ),
            'defaults'     => [
                'provider'     => 'Africa Diagnostic Laboratory',
                'requested_at' => now()->format('Y-m-d\TH:i'),
            ],
            // latest imaging ref (nullable)
            'imaging_latest' => $imaging,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'consultation_id'      => ['required', 'exists:consultations,id'],
            'patient_id'           => ['required', 'exists:patients,id'],
            'provider'             => ['nullable', 'string', 'max:255'],
            'lab_ref_no'           => ['nullable', 'string', 'max:255'],
            'requested_at'         => ['nullable', 'date'],
            'clinical_data'        => ['nullable', 'string'],
            'specimen_types'       => ['nullable', 'array'],
            'specimen_types.*'     => ['string', Rule::in(['blood', 'urine', 'swab', 'fluid', 'stool', 'serum', 'other'])],
            'tests_requested'      => ['nullable', 'string'],

            'medical_aid_provider' => ['nullable', 'string', 'max:255'],
            'membership_no'        => ['nullable', 'string', 'max:255'],
            'membership_suffix'    => ['nullable', 'string', 'max:50'],
            'relation_to_member'   => ['nullable', 'string', 'max:100'],

            'copies_to'            => ['nullable', 'string', 'max:255'],
            'patient_bled_by'      => ['nullable', 'string', 'max:255'],
            'tubes_edta'           => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_sst'            => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_plain'          => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_fluoride'       => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_citrate'        => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_heparin'        => ['nullable', 'integer', 'min:0', 'max:50'],
            'collected_at'         => ['nullable', 'date'],
            'received_by'          => ['nullable', 'string', 'max:255'],
            'received_at'          => ['nullable', 'date'],

            'referral_notes'       => ['nullable', 'string'],
            'status'               => ['nullable', 'in:draft,sent,completed,cancelled'],
            'attachment'           => ['nullable', 'file', 'max:8192'],

            // Optional post-save redirect
            'next'                 => ['nullable', Rule::in(['notes', 'dispense', 'imaging', 'show'])],
        ]);

        $consultation = Consultation::findOrFail($data['consultation_id']);
        abort_if((int) $consultation->patient_id !== (int) $data['patient_id'], 422, 'Patient does not match the consultation.');

        $path = $request->hasFile('attachment')
            ? $request->file('attachment')->store('referrals/attachments', 'public')
            : null;

        $referral = LabReferral::create([
            'consultation_id'      => $data['consultation_id'],
            'patient_id'           => $data['patient_id'],
            'referrer_id'          => auth()->id(),
            'provider'             => $data['provider'] ?? 'Africa Diagnostic Laboratory',
            'lab_ref_no'           => $data['lab_ref_no'] ?? null,
            'requested_at'         => $data['requested_at'] ?? now(),
            'clinical_data'        => $data['clinical_data'] ?? null,
            'specimen_types'       => $data['specimen_types'] ?? [],
            'tests_requested'      => $data['tests_requested'] ?? null,

            'medical_aid_provider' => $data['medical_aid_provider'] ?? null,
            'membership_no'        => $data['membership_no'] ?? null,
            'membership_suffix'    => $data['membership_suffix'] ?? null,
            'relation_to_member'   => $data['relation_to_member'] ?? null,

            'copies_to'            => $data['copies_to'] ?? null,
            'patient_bled_by'      => $data['patient_bled_by'] ?? null,
            'tubes_edta'           => (int) ($data['tubes_edta'] ?? 0),
            'tubes_sst'            => (int) ($data['tubes_sst'] ?? 0),
            'tubes_plain'          => (int) ($data['tubes_plain'] ?? 0),
            'tubes_fluoride'       => (int) ($data['tubes_fluoride'] ?? 0),
            'tubes_citrate'        => (int) ($data['tubes_citrate'] ?? 0),
            'tubes_heparin'        => (int) ($data['tubes_heparin'] ?? 0),

            'collected_at'         => $data['collected_at'] ?? null,
            'received_by'          => $data['received_by'] ?? null,
            'received_at'          => $data['received_at'] ?? null,

            'referral_notes'       => $data['referral_notes'] ?? null,
            'status'               => $data['status'] ?? 'sent',
            'attachment_path'      => $path,
        ]);

        // --- Post-save routing based on "next"
        $next = (string) $request->input('next'); // 'notes' | 'dispense' | 'imaging' | 'show' | ''
        $role = auth()->user()?->role ?? null;

        $isDoctor = $role instanceof UserRole
            ? ($role === UserRole::Doctor)
            : (strtolower((string) $role) === UserRole::Doctor->value);

        if ($next === 'notes' && $isDoctor) {
            return redirect()
                ->route('consultations.doctor-notes.create', ['consultation' => $consultation->id])
                ->with('success', 'Lab referral created. Proceed to doctor notes.');
        }

        if ($next === 'dispense') {
            return redirect()
                ->route('consultations.dispense', $consultation->id)
                ->with('success', 'Lab referral created. Proceed to dispensing.');
        }

        if ($next === 'imaging') {
            return redirect()
                ->route('referrals.imaging.create', [
                    'consultation' => $consultation->id,
                    'patient'      => $referral->patient_id,
                ])
                ->with('success', 'Lab referral created. Proceed to imaging referral.');
        }

        return redirect()
            ->route('referrals.lab.show', $referral)
            ->with('success', 'Lab referral created.');
    }

    public function show(LabReferral $labReferral)
    {
        $labReferral->load([
            'patient:id,first_name,surname,gender,date_of_birth,employee_number',
            'consultation:id,consultation_date',
            'referrer:id,name'
        ]);

        return Inertia::render('Referrals/Lab/Show', [
            'referral' => $labReferral,
        ]);
    }

    public function edit(LabReferral $labReferral, ?Consultation $consultation = null, ?Patient $patient = null)
    {
        $labReferral->load([
            'patient:id,first_name,surname,gender,date_of_birth,employee_number',
            'consultation:id,consultation_date'
        ]);

        if ($consultation) abort_if((int) $labReferral->consultation_id !== (int) $consultation->id, 404);
        if ($patient)      abort_if((int) $labReferral->patient_id !== (int) $patient->id, 404);

        return Inertia::render('Referrals/Lab/Edit', [
            'referral'     => $labReferral,
            'consultation' => $labReferral->consultation?->only(['id', 'consultation_date']),
            'patient'      => $labReferral->patient?->only(['id','first_name','surname','gender','date_of_birth','employee_number']),
            'defaults'     => [
                'requested_at' => optional($labReferral->requested_at)->format('Y-m-d\TH:i'),
                'collected_at' => optional($labReferral->collected_at)->format('Y-m-d\TH:i'),
                'received_at'  => optional($labReferral->received_at)->format('Y-m-d\TH:i'),
            ],
        ]);
    }

    public function update(Request $request, LabReferral $labReferral)
    {
        $data = $request->validate([
            'provider'             => ['nullable', 'string', 'max:255'],
            'lab_ref_no'           => ['nullable', 'string', 'max:255'],
            'requested_at'         => ['nullable', 'date'],
            'clinical_data'        => ['nullable', 'string'],
            'specimen_types'       => ['nullable', 'array'],
            'specimen_types.*'     => ['string', Rule::in(['blood', 'urine', 'swab', 'fluid', 'stool', 'serum', 'other'])],
            'tests_requested'      => ['nullable', 'string'],

            'medical_aid_provider' => ['nullable', 'string', 'max:255'],
            'membership_no'        => ['nullable', 'string', 'max:255'],
            'membership_suffix'    => ['nullable', 'string', 'max:50'],
            'relation_to_member'   => ['nullable', 'string', 'max:100'],

            'copies_to'            => ['nullable', 'string', 'max:255'],
            'patient_bled_by'      => ['nullable', 'string', 'max:255'],
            'tubes_edta'           => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_sst'            => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_plain'          => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_fluoride'       => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_citrate'        => ['nullable', 'integer', 'min:0', 'max:50'],
            'tubes_heparin'        => ['nullable', 'integer', 'min:0', 'max:50'],
            'collected_at'         => ['nullable', 'date'],
            'received_by'          => ['nullable', 'string', 'max:255'],
            'received_at'          => ['nullable', 'date'],

            'referral_notes'       => ['nullable', 'string'],
            'status'               => ['nullable', 'in:draft,sent,completed,cancelled'],
            'attachment'           => ['nullable', 'file', 'max:8192'],
        ]);

        if ($request->hasFile('attachment')) {
            if ($labReferral->attachment_path) {
                Storage::disk('public')->delete($labReferral->attachment_path);
            }
            $data['attachment_path'] = $request->file('attachment')->store('referrals/attachments', 'public');
        }

        $labReferral->fill($data)->save();

        return redirect()->route('referrals.lab.show', $labReferral)->with('success', 'Lab referral updated.');
    }

    public function destroy(LabReferral $labReferral)
    {
        if ($labReferral->attachment_path) {
            Storage::disk('public')->delete($labReferral->attachment_path);
        }
        $labReferral->delete();

        return redirect()->route('referrals.lab.index')->with('success', 'Lab referral deleted.');
    }
}
