<?php
namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Consultation;
use App\Models\LabTest;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class LabTestController extends Controller
{
    public function create($consultationId)
{
    $consultation = Consultation::with([
        'patient:id,first_name,surname',
        'doctor:id,name'
    ])->findOrFail($consultationId);

    return Inertia::render('LabTests/Create', [
        'consultationId' => $consultation->id,
        'patient' => $consultation->patient,
        'consultation' => $consultation->only([
            'id',
            'consultation_date',
            'presenting_complaint',
            'clinical_notes',
            'diagnosis',
            'treatment_plan',
            'doctor'
        ]),
        'labTechnicians' => User::where('role', UserRole::LabTechnician)->get(['id', 'name']),
        'availableTests' => [
            'FBC',
            'Malaria',
            'COVID-19 PCR',
            'Urea & Electrolytes',
            'Liver Function Test',
        ],
    ]);
}

    public function store(Request $request, $consultationId)
    {
        $consultation = Consultation::findOrFail($consultationId);

        $validated = $request->validate([
            'test_name'         => 'required|string|max:255',
            'status'            => 'required|in:requested,in_progress,completed',
            'result_value'      => 'nullable|string',
            'result_file'       => 'nullable|string',
            'lab_technician_id' => 'nullable|exists:users,id',
        ]);

        LabTest::create([
            'patient_id'        => $consultation->patient_id,
            'consultation_id'   => $consultation->id,
            'test_name'         => $validated['test_name'],
            'status'            => $validated['status'],
            'result_value'      => $validated['result_value'] ?? null,
            'result_file'       => $validated['result_file'] ?? null,
            'ordered_by'        => Auth::id(),
            'lab_technician_id' => $validated['lab_technician_id'] ?? null,
            'completed_at'      => $validated['status'] === 'completed' ? now() : null,
        ]);

        return redirect()->route('consultations.show', $consultationId)->with('success', 'Lab test created.');
    }

    public function edit(LabTest $labTest)
    {
        $labTest->load(['patient', 'consultation']);

        return Inertia::render('LabTests/Edit', [
            'test'           => $labTest,
            'testNames'      => $this->getAvailableTestNames(),
            'labTechnicians' => User::where('role', UserRole::LabTechnician)->get(['id', 'first_name', 'surname']),
        ]);
    }

    public function update(Request $request, LabTest $labTest)
    {
        $validated = $request->validate([
            'test_name'         => 'required|string|max:255',
            'status'            => 'required|in:requested,in_progress,completed',
            'result_value'      => 'nullable|string',
            'result_file'       => 'nullable|string',
            'lab_technician_id' => 'nullable|exists:users,id',
        ]);

        $labTest->update([
             ...$validated,
            'completed_at' => $validated['status'] === 'completed' ? now() : null,
        ]);

        return redirect()->route('consultations.show', $labTest->consultation_id)->with('success', 'Lab test updated.');
    }

    private function getAvailableTestNames()
    {
        return [
            'Full Blood Count (FBC)',
            'Malaria Rapid Test',
            'COVID-19 PCR',
            'Urinalysis',
            'Stool Microscopy',
            'Blood Glucose',
            'Liver Function Test',
            'Renal Panel',
        ];
    }
}
