<?php
namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\MedicalFundMember;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

// OpenSpout v4
use OpenSpout\Reader\CSV\Options as CSVOptions;
use OpenSpout\Reader\CSV\Reader as CSVReader;
use OpenSpout\Reader\XLSX\Reader as XLSXReader;

class MedicalFundMemberController extends Controller
{
    public function searchMembers(Request $request)
    {
        $q   = trim($request->string('q')->toString());
        $tag = $request->string('tag')->toString() ?: 'Providence Fund';

        if ($q === '') {
            return response()->json([]);
        }

        $results = MedicalFundMember::query()
            ->with('company:id,name')
            ->where('tag', $tag)
            ->where(function ($w) use ($q) {
                $w->where('employee_number', 'like', "%{$q}%")
                    ->orWhere('first_name', 'like', "%{$q}%")
                    ->orWhere('surname', 'like', "%{$q}%");
            })
            ->orderBy('surname')
            ->limit(20)
            ->get([
                'id', 'employee_number', 'first_name', 'middle_name', 'surname',
                'company_id', 'tag', 'created_at',
            ]);

        return response()->json($results);
    }

    /* ================== INDEX (list + basic search) ================== */
    public function index(Request $request)
    {
        $q = MedicalFundMember::query()->with('company');

        if ($search = trim((string) $request->get('q', ''))) {
            $q->where(function ($w) use ($search) {
                $w->where('employee_number', 'like', "%{$search}%")
                    ->orWhere('first_name', 'like', "%{$search}%")
                    ->orWhere('middle_name', 'like', "%{$search}%")
                    ->orWhere('surname', 'like', "%{$search}%")
                    ->orWhere('tag', 'like', "%{$search}%");
            });
        }

        if ($tag = $request->string('tag')->toString()) {
            $q->where('tag', $tag);
        }

        if ($companyId = $request->integer('company_id')) {
            $q->where('company_id', $companyId);
        }

        $members = $q->latest()->paginate(20)->appends($request->query());

        return inertia('FundMembers/Index', [
            'members' => $members,
            'filters' => [
                'q'          => $search ?? '',
                'tag'        => $tag ?? '',
                'company_id' => $companyId ?: null,
            ],
        ]);
    }

    /* ================== STORE (single create) ================== */
    public function store(Request $request)
    {
        $data = $request->validate([
            'employee_number' => 'required|string|max:64',
            'first_name'      => 'required|string|max:120',
            'middle_name'     => 'nullable|string|max:120',
            'surname'         => 'required|string|max:120',
            'company_id'      => 'nullable|exists:companies,id',
            'company'         => 'nullable|string|max:255', // if provided and company_id missing, auto-create
            'tag'             => 'nullable|string|max:120',
        ]);

        $companyId = $data['company_id'] ?? null;
        if (! $companyId && ! empty($data['company'])) {
            $companyId = $this->firstOrCreateCompanyByName($data['company']);
        }

        $member = MedicalFundMember::updateOrCreate(
            [
                'employee_number' => $data['employee_number'],
                'company_id'      => $companyId,
                'tag'             => $data['tag'] ?? 'Providence Fund',
            ],
            [
                'first_name'  => $data['first_name'],
                'middle_name' => $data['middle_name'] ?? null,
                'surname'     => $data['surname'],
            ]
        );

        return redirect()->back()->with('success', 'Member saved: ' . $member->full_name);
    }

    /* ================== BULK IMPORT (UI) ================== */
    public function bulkForm()
    {
        return inertia('FundMembers/BulkImport', [
            'templates' => [
                'members' => [
                    'headers' => [
                                           // Keep it minimal to your spec
                        'employee_number', // a.k.a code
                        'first_name',
                        'middle_name',
                        'surname',
                        'company',    // company name (preferred)
                        'company_id', // (optional) will override name if valid
                        'tag',        // optional; default "Providence Fund"
                    ],
                ],
            ],
        ]);
    }

    /* ================== BULK IMPORT (PREVIEW) ================== */
    public function bulkPreview(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,txt',
        ]);

        [$headers, $rows] = $this->readSpreadsheetToArrays($request->file('file'));

        $preview = [];
        $valid   = 0;
        $invalid = 0;

        foreach ($rows as $i => $rowValues) {
            $row                   = $this->mapRowByHeaders($headers, $rowValues);
            [$ok, $errs, $payload] = $this->validateMemberRow($row);

            $preview[] = [
                'index'  => $i + 2,
                'data'   => $payload,
                'errors' => $errs,
                'raw'    => $row,
            ];
            $ok ? $valid++ : $invalid++;
        }

        return response()->json([
            'columns' => $headers,
            'rows'    => $preview,
            'stats'   => ['total' => count($rows), 'valid' => $valid, 'invalid' => $invalid],
        ]);
    }

    /* ================== BULK IMPORT (COMMIT) ================== */
    public function bulkCommit(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,txt',
        ]);

        [$headers, $rows] = $this->readSpreadsheetToArrays($request->file('file'));

        $created = 0;
        $updated = 0;
        $failed  = 0;
        $errors  = [];

        DB::beginTransaction();
        try {
            foreach ($rows as $i => $rowValues) {
                $row                   = $this->mapRowByHeaders($headers, $rowValues);
                [$ok, $errs, $payload] = $this->validateMemberRow($row);

                if (! $ok) {
                    $failed++;
                    $errors[] = ['row' => $i + 2, 'errors' => $errs];
                    continue;
                }

                // Resolve/create company by name if no valid company_id
                $companyId = $payload['company_id'] ?? null;
                if (! $companyId && ! empty($payload['company'])) {
                    $companyId = $this->firstOrCreateCompanyByName($payload['company']);
                }
                unset($payload['company']); // not a column
                $payload['company_id'] = $companyId;

                // Upsert by unique key (employee_number + company_id + tag)
                $existing = MedicalFundMember::where([
                    'employee_number' => $payload['employee_number'],
                    'company_id'      => $payload['company_id'],
                    'tag'             => $payload['tag'] ?? 'Providence Fund',
                ])->first();

                if ($existing) {
                    $existing->fill([
                        'first_name'  => $payload['first_name'],
                        'middle_name' => $payload['middle_name'] ?? null,
                        'surname'     => $payload['surname'],
                    ])->save();
                    $updated++;
                } else {
                    MedicalFundMember::create($payload);
                    $created++;
                }
            }
            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['ok' => false, 'message' => 'Import failed: ' . $e->getMessage()], 500);
        }

        return response()->json([
            'ok'      => true,
            'created' => $created,
            'updated' => $updated,
            'failed'  => $failed,
            'errors'  => $errors,
        ]);
    }

    /* ================== Helpers ================== */

    /** Read XLSX/CSV into [headers, rows] */
    private function readSpreadsheetToArrays(UploadedFile $file): array
    {
        $ext = strtolower($file->getClientOriginalExtension() ?: $file->guessExtension() ?: '');

        if ($ext === 'xlsx') {
            $reader = new XLSXReader();
        } else {
            $csvOptions                 = new CSVOptions();
            $csvOptions->fieldDelimiter = ',';
            $csvOptions->fieldEnclosure = '"';
            $csvOptions->encoding       = 'UTF-8';
            $reader                     = new CSVReader($csvOptions);
        }

        $reader->open($file->getRealPath());

        $headers = [];
        $rows    = [];

        foreach ($reader->getSheetIterator() as $sheet) {
            foreach ($sheet->getRowIterator() as $idx => $row) {
                $cells = $row->getCells();
                $vals  = array_map(fn($c) => is_string($c->getValue()) ? trim($c->getValue()) : $c->getValue(), $cells);

                if ($idx === 1) {
                    $headers = array_map(fn($h) => strtolower(trim((string) $h)), $vals);
                    continue;
                }

                if (count(array_filter($vals, fn($v) => $v !== null && $v !== '')) === 0) {
                    continue;
                }

                $rows[] = $vals;
            }
            break; // only first sheet
        }

        $reader->close();
        return [$headers, $rows];
    }

    /** Map row by headers */
    private function mapRowByHeaders(array $headers, array $vals): array
    {
        $out = [];
        foreach ($headers as $i => $key) {
            $out[$key] = $vals[$i] ?? null;
        }

        return $out;
    }

    /** Validate/shape a fund member row (very simple) */
    private function validateMemberRow(array $row): array
    {
        $errs = [];

        // allow multiple header variants (code/idp/id)
        $employeeNumber = $this->coalesce($row, ['employee_number', 'code', 'idp', 'id']);
        $first          = $this->coalesce($row, ['first_name', 'firstname']);
        $middle         = $row['middle_name'] ?? null;
        $surname        = $this->coalesce($row, ['surname', 'last_name']);
        $company        = $this->coalesce($row, ['company']);
        $companyId      = $row['company_id'] ?? null;
        $tag            = $row['tag'] ?? 'Providence Fund';

        if (! $employeeNumber) {
            $errs[] = 'employee_number (or code/idp) is required';
        }

        if (! $first) {
            $errs[] = 'first_name is required';
        }

        if (! $surname) {
            $errs[] = 'surname is required';
        }

        $payload = [
            'employee_number' => $employeeNumber ? (string) $employeeNumber : null,
            'first_name'      => $first ? Str::title(trim($first)) : null,
            'middle_name'     => $middle ? Str::title(trim($middle)) : null,
            'surname'         => $surname ? Str::upper(trim($surname)) : null,
            'company'         => $company ? trim($company) : null, // transient
            'company_id'      => $companyId ?: null,
            'tag'             => trim($tag) !== '' ? trim($tag) : 'Providence Fund',
        ];

        return [count($errs) === 0, $errs, $payload];
    }

    private function coalesce(array $row, array $keys)
    {
        foreach ($keys as $k) {
            if (isset($row[$k]) && $row[$k] !== '') {
                return $row[$k];
            }

        }
        return null;
    }

    /** Find or create a company by name; generate a unique email from the name; other fields default */
    private function firstOrCreateCompanyByName(string $name): int
    {
        $name = trim($name);
        if ($name === '') {
            return null;
        }

        $existing = Company::where('name', $name)->first();
        if ($existing) {
            return $existing->id;
        }

        // Generate unique email like: info+well-page-staffing-solutions@providencefund.local
        $slug  = Str::slug($name);
        $email = $this->uniqueCompanyEmail("info+{$slug}@providencefund.local");

        $company = Company::create([
            'name'           => $name,
            'address'        => null,
            'site_telephone' => null,
            'company_email'  => $email,
            'contact_person' => null,
            'province'       => null,
            'designation'    => 'N/A',
            'contact_number' => null,
        ]);

        return $company->id;
    }

    private function uniqueCompanyEmail(string $baseEmail): string
    {
        [$local, $domain] = explode('@', $baseEmail, 2);
        $email            = $baseEmail;
        $i                = 1;

        while (Company::where('company_email', $email)->exists()) {
            $email = "{$local}{$i}@{$domain}";
            $i++;
        }
        return $email;
    }
}
