<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Attendee;
use App\Models\Medicals\Patient;
use App\Models\Medicals\Company;
use App\Models\Medicals\Auscultate;
use App\Models\Medicals\Certificate;
use App\Models\Medicals\Disease;
use App\Models\Medicals\Illness;
use App\Models\Medicals\MedicalRecord;
use App\Models\Medicals\Pneumoconiosis;
use App\Models\Medicals\SkinCondition;
use App\Models\Medicals\TobaccoUse;
use App\Models\Medicals\Category; // ⬅️ NEW
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class AttendeeController extends Controller
{
    /**
     * GET /medicals/attendees
     * Query params: search, per_page, sort (created_at|time_of_entry|last_name), direction (asc|desc)
     */
    public function index(Request $request)
    {
        $search    = $request->string('search')->toString();
        $perPage   = (int) ($request->integer('per_page') ?: 15);
        $sort      = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc') === 'asc' ? 'asc' : 'desc';

        $query = Attendee::query()
            ->with([
                'company',
                'patient.certificates' => fn ($q) => $q->latest(),
                'categories:id,slug,name',            // ⬅️ list attendee categories
                'patient.categories:id,slug,name',    // ⬅️ list patient categories (usually same)
            ])
            ->when($search !== '', function ($q) use ($search) {
                $like = '%' . trim($search) . '%';
                $q->where(function ($qq) use ($like) {
                    $qq->where('swab_number', 'like', $like)
                        ->orWhere('first_name', 'like', $like)
                        ->orWhere('last_name', 'like', $like)
                        ->orWhere('national_id', 'like', $like);
                });
            });

        // Safe sort
        $allowedSorts = ['created_at', 'time_of_entry', 'last_name'];
        if (! in_array($sort, $allowedSorts, true)) {
            $sort = 'created_at';
        }
        $query->orderBy($sort, $direction);

        $attendees = $query->paginate($perPage)->withQueryString();

        return Inertia::render('Medicals/Attendees/Index', [
            'attendees' => $attendees->through(function (Attendee $a) {
                $age = null;
                if (!empty($a->date_of_birth)) {
                    try {
                        $age = Carbon::parse($a->date_of_birth)->diffInYears(Carbon::now());
                    } catch (\Throwable $e) {
                        $age = null;
                    }
                }

                $toe = $a->time_of_entry
                    ? Carbon::parse($a->time_of_entry)->timezone('Africa/Harare')->toDateTimeString()
                    : null;

                $latestCert = optional($a->patient)->certificates()->latest()->first();

                return [
                    'id'               => $a->id,
                    'company'          => $a->relationLoaded('company') && $a->company ? [
                        'id'           => $a->company->id,
                        'company_name' => $a->company->company_name,
                    ] : null,
                    'employee_number'  => optional($a->patient)->employee_number,
                    'exam_purpose'     => optional($a->patient)->exam_purpose,
                    'category'         => optional($a->patient)->category, // legacy single category (first selected)
                    'categories'       => $a->categories?->map(fn ($c) => [
                        'id'   => $c->id,
                        'slug' => $c->slug,
                        'name' => $c->name,
                    ])->values() ?? [],
                    'patient_categories' => $a->patient?->categories?->map(fn ($c) => [
                        'id'   => $c->id,
                        'slug' => $c->slug,
                        'name' => $c->name,
                    ])->values() ?? [],
                    'last_x_ray'       => optional($a->patient)->last_x_ray,
                    'swab_number'      => $a->swab_number,
                    'x_ray_status'     => $a->x_ray_status,
                    'time_of_entry'    => $toe,
                    'first_name'       => $a->first_name,
                    'last_name'        => $a->last_name,
                    'date_of_birth'    => $a->date_of_birth,
                    'age'              => $age,
                    'gender'           => $a->gender,
                    'national_id'      => $a->national_id,
                    'phone_number'     => $a->phone_number,
                    'latest_certificate' => $latestCert ? [
                        'id'                   => $latestCert->id,
                        'patient_id'           => $latestCert->patient_id,
                        'status'               => $latestCert->status,
                        'certificate_location' => $latestCert->certificate_location,
                        'created_at'           => Carbon::parse($latestCert->created_at)->format('Y-m-d'),
                        'updated_at'           => Carbon::parse($latestCert->updated_at)->format('Y-m-d'),
                    ] : null,
                    'created_at'       => optional($a->created_at)?->toIso8601String(),
                    'updated_at'       => optional($a->updated_at)?->toIso8601String(),
                ];
            }),
            'filters' => [
                'search'    => $search,
                'per_page'  => $perPage,
                'sort'      => $sort,
                'direction' => $direction,
            ],
        ]);
    }

    /**
     * GET /medicals/attendees/create
     */
    public function create()
    {
        $companies  = Company::orderBy('company_name')->get(['id', 'company_name']);
        $categories = Category::orderBy('name')->get(['id', 'slug', 'name']); // ⬅️ send to UI

        return Inertia::render('Medicals/Attendees/Create', [
            'companies'  => $companies,
            'categories' => $categories,
        ]);
    }

    /**
     * POST /medicals/attendees
     * Creates Attendee + related Patient + Certificate + MedicalRecord + category pivots.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_id'      => ['required', 'numeric', 'exists:medicals.companies,id'],
            'first_name'      => ['required', 'string'],
            'last_name'       => ['required', 'string'],
            'date_of_birth'   => ['nullable', 'string'],
            'gender'          => ['nullable', 'string'],
            'national_id'     => ['required', 'string'],
            'phone_number'    => ['nullable', 'string'],
            'employee_number' => ['nullable', 'string'],
            'x_ray_status'    => ['nullable', 'string'],
            'exam_purpose'    => ['nullable', 'numeric'], // (1..4) legacy mapping
            'country_code'    => ['nullable', 'string'],
            'last_x_ray'      => ['nullable', 'string'],
            'category'        => ['nullable', 'string'],  // legacy single (kept for compatibility)
            'created_at'      => ['nullable', 'string'],
            'location'        => ['nullable', 'string'],

            // ⬅️ NEW: multi-categories (array of slugs or IDs)
            'categories'      => ['required', 'array', 'min:1'],
            'categories.*'    => ['string'], // we’ll resolve to Category rows (slug or numeric id passed as string)
        ]);

        // Parse created_at (exam datetime) – affects child records' timestamps
        $createdAt = $request->has('created_at')
            ? Carbon::parse($request->input('created_at'))
            : now();

        // Compose phone into final field (keep both pieces if you need them)
        if (!empty($data['country_code'])) {
            $data['phone_number'] = ($data['country_code'] ?? '') . ($data['phone_number'] ?? '');
        }
        unset($data['country_code']);

        // Resolve categories passed from UI
        // Accept either slug or numeric ID. We map to full rows for name checks.
        $incoming = $data['categories'] ?? [];
        $categories = Category::query()
            ->where(function ($q) use ($incoming) {
                $q->whereIn('slug', $incoming)
                  ->orWhereIn('id', collect($incoming)->filter(fn($x) => is_numeric($x))->all());
            })
            ->get(['id', 'slug', 'name']);

        if ($categories->isEmpty()) {
            return back()->withErrors(['categories' => 'Please choose at least one valid category.'])->withInput();
        }

        // Legacy single category string (keep old reports/screens working):
        // set patient.category = FIRST selected category's name (display name)
        $primaryCategoryName = $categories->first()->name;

        // Decide if we should generate a swab number (Food Handler or In House chosen)
        $requiresSwab = $categories->contains(fn ($c) => in_array($c->name, ['Food Handler (COH)', 'In House'], true));

        // Swab number generator
        $generateSwab = function (int $len = 5): string {
            $chars = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
            $s = '';
            for ($i = 0; $i < $len; $i++) {
                $s .= $chars[random_int(0, strlen($chars) - 1)];
            }
            return $s;
        };

        DB::beginTransaction();
        try {
            // Ensure unique swab when needed
            $swab = null;
            if ($requiresSwab) {
                do {
                    $swab = $generateSwab(5);
                } while (Attendee::where('swab_number', $swab)->exists());
            }

            $data['swab_number']   = $swab;
            $data['time_of_entry'] = now();

            // Avoid duplicate national IDs for same company – same behavior as before
            $existing = Attendee::where('national_id', $data['national_id'])->first();
            if ($existing) {
                if ((int)$existing->company_id === (int)$data['company_id']) {
                    DB::rollBack();
                    return back()->withErrors(['national_id' => 'National ID already exists for this company.'])->withInput();
                } else {
                    // Move attendee to new company
                    $existing->update(['company_id' => $data['company_id']]);
                    $attendee = $existing;
                }
            } else {
                $attendee = Attendee::create(array_merge($data, [
                    'created_at' => $createdAt,
                    'updated_at' => $createdAt,
                ]));
            }

            // Map numeric exam_purpose to label
            $purpose = match ((int)($data['exam_purpose'] ?? 2)) {
                1 => 'Pre-Placement',
                2 => 'Periodical',
                3 => 'Exit(Employment Termination)',
                4 => 'Post(Employment Employment Follow Up)',
                default => 'Periodical',
            };

            // Build or reuse patient relation
            $patient = $attendee->patient;
            if (!$patient) {
                $patient = Patient::create([
                    'attendee_id'  => $attendee->id,
                    'company_id'   => $data['company_id'],
                    'exam_purpose' => $purpose,
                    'last_x_ray'   => $data['last_x_ray'] ?? null,
                    'category'     => $primaryCategoryName, // ⬅️ legacy string (first selected)
                    'created_at'   => $createdAt,
                    'updated_at'   => $createdAt,
                ]);
            } else {
                $patient->update([
                    'company_id'   => $data['company_id'],
                    'exam_purpose' => $purpose,
                    'last_x_ray'   => $data['last_x_ray'] ?? $patient->last_x_ray,
                    'category'     => $primaryCategoryName, // keep legacy aligned
                ]);
            }

            // ⬇️ Sync multi-categories on attendee + patient
            $attendee->categories()->sync($categories->pluck('id')->all());
            $patient->categories()->sync($categories->pluck('id')->all());

            // Seed other pivots based on category membership
            $names = $categories->pluck('name')->map(fn($s) => trim($s))->all();

            // Food Handler / In House: illnesses + tobacco uses
            if (in_array('Food Handler (COH)', $names, true) || in_array('In House', $names, true)) {
                $patient->illnesses()->syncWithoutDetaching(Illness::query()->pluck('id'));
                $patient->tobacco_uses()->syncWithoutDetaching(TobaccoUse::query()->pluck('id'));
            }

            // Skin conditions + Auscultates for everyone (as before)
            $patient->skin_conditions()->syncWithoutDetaching(SkinCondition::query()->pluck('id'));
            $patient->auscultates()->syncWithoutDetaching(Auscultate::query()->pluck('id'));

            // Employment categories: diseases
            if (in_array('Pre-Employement', $names, true) || in_array('Exit-Employement', $names, true)) {
                $patient->diseases()->syncWithoutDetaching(Disease::query()->pluck('id'));
            }

            // Pneumo branch
            if (in_array('Pneumoconiosis', $names, true) || in_array('Exit-Pneumoconiosis', $names, true)) {
                Pneumoconiosis::firstOrCreate(
                    ['patient_id' => $patient->id],
                    [
                        'company_id'   => $data['company_id'],
                        'exam_purpose' => $patient->exam_purpose,
                        'created_at'   => $createdAt,
                        'updated_at'   => $createdAt,
                    ]
                );
            }

            // Create default certificate + medical record (timestamps aligned to created_at if provided)
            Certificate::create([
                'patient_id'           => $patient->id,
                'validity'             => 1,
                'status'               => 'PENDING',
                'certificate_location' => $request->input('location'),
                'created_at'           => $createdAt,
                'updated_at'           => $createdAt,
            ]);

            MedicalRecord::create([
                'patient_id'   => $patient->id,
                'year_of_exam' => $createdAt->year,
                'created_at'   => $createdAt,
                'updated_at'   => $createdAt,
            ]);

            DB::commit();

            return redirect()
                ->route('medicals.attendees.show', $attendee->id)
                ->with('success', 'Attendee created successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * GET /medicals/attendees/{attendee}
     */
    public function show(Attendee $attendee)
    {
        $attendee->load([
            'company',
            'patient.certificates' => fn ($q) => $q->latest(),
            'categories:id,slug,name',
            'patient.categories:id,slug,name',
        ]);

        $age = null;
        if (!empty($attendee->date_of_birth)) {
            try {
                $age = Carbon::parse($attendee->date_of_birth)->diffInYears(now());
            } catch (\Throwable $e) {
                $age = null;
            }
        }

        $toe = $attendee->time_of_entry
            ? Carbon::parse($attendee->time_of_entry)->timezone('Africa/Harare')->toDateTimeString()
            : null;

        $latestCert = optional($attendee->patient)->certificates()->latest()->first();

        return Inertia::render('Medicals/Attendees/Show', [
            'attendee' => [
                'id'               => $attendee->id,
                'company'          => $attendee->company ? [
                    'id'           => $attendee->company->id,
                    'company_name' => $attendee->company->company_name,
                ] : null,
                'employee_number'  => optional($attendee->patient)->employee_number,
                'exam_purpose'     => optional($attendee->patient)->exam_purpose,
                'category'         => optional($attendee->patient)->category, // legacy
                'categories'       => $attendee->categories?->map(fn ($c) => [
                    'id'   => $c->id, 'slug' => $c->slug, 'name' => $c->name,
                ])->values() ?? [],
                'patient_categories' => $attendee->patient?->categories?->map(fn ($c) => [
                    'id'   => $c->id, 'slug' => $c->slug, 'name' => $c->name,
                ])->values() ?? [],
                'last_x_ray'       => optional($attendee->patient)->last_x_ray,
                'swab_number'      => $attendee->swab_number,
                'x_ray_status'     => $attendee->x_ray_status,
                'time_of_entry'    => $toe,
                'first_name'       => $attendee->first_name,
                'last_name'        => $attendee->last_name,
                'date_of_birth'    => $attendee->date_of_birth,
                'age'              => $age,
                'gender'           => $attendee->gender,
                'national_id'      => $attendee->national_id,
                'phone_number'     => $attendee->phone_number,
                'latest_certificate' => $latestCert ? [
                    'id'                   => $latestCert->id,
                    'patient_id'           => $latestCert->patient_id,
                    'status'               => $latestCert->status,
                    'certificate_location' => $latestCert->certificate_location,
                    'created_at'           => Carbon::parse($latestCert->created_at)->format('Y-m-d'),
                    'updated_at'           => Carbon::parse($latestCert->updated_at)->format('Y-m-d'),
                ] : null,
                'created_at'       => optional($attendee->created_at)?->toIso8601String(),
                'updated_at'       => optional($attendee->updated_at)?->toIso8601String(),
            ],
        ]);
    }

    /**
     * GET /medicals/attendees/{attendee}/edit
     */
    public function edit(Attendee $attendee)
    {
        $attendee->load(['company', 'patient', 'categories:id,slug,name', 'patient.categories:id,slug,name']);
        $companies  = Company::orderBy('company_name')->get(['id', 'company_name']);
        $categories = Category::orderBy('name')->get(['id', 'slug', 'name']);

        return Inertia::render('Medicals/Attendees/Edit', [
            'attendee' => [
                'id'               => $attendee->id,
                'company_id'       => $attendee->company_id,
                'first_name'       => $attendee->first_name,
                'last_name'        => $attendee->last_name,
                'date_of_birth'    => $attendee->date_of_birth,
                'gender'           => $attendee->gender,
                'national_id'      => $attendee->national_id,
                'phone_number'     => $attendee->phone_number,
                'employee_number'  => optional($attendee->patient)->employee_number,
                'x_ray_status'     => $attendee->x_ray_status,
                'last_x_ray'       => optional($attendee->patient)->last_x_ray,
                'exam_purpose'     => optional($attendee->patient)->exam_purpose,
                'category'         => optional($attendee->patient)->category, // legacy
                'categories'       => $attendee->categories?->pluck('slug')->values() ?? [], // form default
            ],
            'companies'  => $companies,
            'categories' => $categories,
        ]);
    }

    /**
     * PUT /medicals/attendees/{attendee}
     * Align attendee & attached patient and re-sync categories.
     * If attendance_day supplied, move latest certificate timestamps to it.
     */
    public function update(Request $request, Attendee $attendee)
    {
        $data = $request->validate([
            'company_id'      => ['required', 'numeric', 'exists:medicals.companies,id'],
            'first_name'      => ['required', 'string'],
            'last_name'       => ['required', 'string'],
            'date_of_birth'   => ['nullable', 'string'],
            'gender'          => ['nullable', 'string'],
            'national_id'     => ['required', 'string'],
            'phone_number'    => ['nullable', 'string'],
            'employee_number' => ['nullable', 'string'],
            'x_ray_status'    => ['nullable', 'string'],
            'exam_purpose'    => ['nullable', 'string'], // textual now
            'last_x_ray'      => ['nullable', 'string'],
            'category'        => ['nullable', 'string'], // legacy single
            'attendance_day'  => ['nullable', 'date'],

            // ⬅️ NEW: allow updating multi-categories
            'categories'      => ['required', 'array', 'min:1'],
            'categories.*'    => ['string'],
        ]);

        // Resolve categories (slug or ID)
        $incoming = $data['categories'] ?? [];
        $categories = Category::query()
            ->where(function ($q) use ($incoming) {
                $q->whereIn('slug', $incoming)
                  ->orWhereIn('id', collect($incoming)->filter(fn($x) => is_numeric($x))->all());
            })
            ->get(['id', 'slug', 'name']);

        if ($categories->isEmpty()) {
            return back()->withErrors(['categories' => 'Please choose at least one valid category.'])->withInput();
        }

        $primaryCategoryName = $categories->first()->name;
        $requiresSwab = $categories->contains(fn ($c) => in_array($c->name, ['Food Handler (COH)', 'In House'], true));

        DB::beginTransaction();
        try {
            $patient = $attendee->patient;

            // Update attendee basics
            $attendee->update([
                'company_id'      => $data['company_id'],
                'first_name'      => $data['first_name'],
                'last_name'       => $data['last_name'],
                'date_of_birth'   => $data['date_of_birth'] ?? null,
                'gender'          => $data['gender'] ?? null,
                'national_id'     => $data['national_id'],
                'phone_number'    => $data['phone_number'] ?? null,
                'x_ray_status'    => $data['x_ray_status'] ?? null,
                'employee_number' => $data['employee_number'] ?? null,
                // keep swab_number as-is; assign one if it’s missing and now required
                'swab_number'     => $attendee->swab_number ?: ($requiresSwab ? $this->generateSwabUnique() : null),
            ]);

            // Update patient basics & legacy category string
            if ($patient) {
                $patient->update([
                    'company_id'   => $data['company_id'],
                    'exam_purpose' => $data['exam_purpose'] ?? $patient->exam_purpose,
                    'category'     => $primaryCategoryName, // legacy
                    'last_x_ray'   => $data['last_x_ray'] ?? $patient->last_x_ray,
                ]);

                // Shift latest certificate to attendance_day if provided
                if (!empty($data['attendance_day'])) {
                    $latest = $patient->certificates()->latest()->first();
                    if ($latest) {
                        $latest->created_at = $data['attendance_day'];
                        $latest->updated_at = $data['attendance_day'];
                        $latest->save();
                    }
                }

                // Re-sync category pivots
                $attendee->categories()->sync($categories->pluck('id')->all());
                $patient->categories()->sync($categories->pluck('id')->all());
            }

            DB::commit();
            return redirect()
                ->route('medicals.attendees.show', $attendee->id)
                ->with('success', 'Attendee updated successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * DELETE /medicals/attendees/{attendee}
     * Relies on Attendee::boot deleting cascade you added on the model.
     */
    public function destroy(Attendee $attendee)
    {
        DB::beginTransaction();
        try {
            $attendee->delete();
            DB::commit();

            return redirect()
                ->route('medicals.attendees.index')
                ->with('success', 'Attendee and related records deleted successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Helper: generate a unique swab number if needed in update().
     */
    private function generateSwabUnique(int $len = 5): string
    {
        $chars = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        do {
            $s = '';
            for ($i = 0; $i < $len; $i++) {
                $s .= $chars[random_int(0, strlen($chars) - 1)];
            }
        } while (Attendee::where('swab_number', $s)->exists());

        return $s;
    }
}
