<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Auscultate;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Validation\Rule;

class AuscultateController extends Controller
{
    public function index(Request $request)
    {
        $search    = (string) $request->string('search');
        $perPage   = (int) ($request->integer('per_page') ?: 20);
        $sort      = $request->input('sort', 'name'); // name|id|created_at|updated_at
        $direction = strtolower($request->input('direction', 'asc')) === 'desc' ? 'desc' : 'asc';

        $query = Auscultate::query();

        if ($search !== '') {
            $query->where('name', 'like', "%{$search}%");
        }

        // safe sort
        if (! in_array($sort, ['name', 'id', 'created_at', 'updated_at'], true)) {
            $sort = 'name';
        }

        $query->orderBy($sort, $direction);

        $paginator = $query->paginate($perPage)->withQueryString();

        return Inertia::render('Medicals/Auscultates/Index', [
            'conditions' => $paginator,
            'filters' => [
                'search'    => $search,
                'per_page'  => $perPage,
                'sort'      => $sort,
                'direction' => $direction,
            ],
        ]);
    }

    public function create()
    {
        return Inertia::render('Medicals/Auscultates/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:255', Rule::unique('auscultates','name')],
        ]);

        $cond = Auscultate::create($data);

        return redirect()
            ->route('medicals.auscultates.index')
            ->with('success', 'Auscultate created.');
    }

    public function edit(Auscultate $auscultate)
    {
        return Inertia::render('Medicals/Auscultates/Edit', [
            'condition' => $auscultate,
        ]);
    }

    public function update(Request $request, Auscultate $auscultate)
    {
        $data = $request->validate([
            'name' => [
                'required','string','max:255',
                Rule::unique('auscultates','name')->ignore($auscultate->id),
            ],
        ]);

        $auscultate->update($data);

        return redirect()
            ->route('medicals.auscultates.index')
            ->with('success', 'Auscultate updated.');
    }

    public function destroy(Auscultate $auscultate)
    {
        $auscultate->delete();

        return redirect()
            ->back()
            ->with('success', 'Auscultate deleted.');
    }
}
