<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Http\Resources\AllPatientsResource;
use App\Models\Medicals\Certificate;
use App\Models\Medicals\MedicalRecord;
use App\Models\Medicals\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class CertificateController extends Controller
{
    /**
     * PATCH /medicals/patients/{patient}/certificate/status
     * Re-evaluates and updates the latest certificate status based on patient category & findings.
     */
    public function updateCertificateStatus(Request $request, int $patientId)
    {
        try {
            /** @var Patient $patient */
            $patient = Patient::with([
                // Eager-load the relationships you reference in helpers to avoid N+1/null issues.
                'certificates'           => fn ($q) => $q->latest(),
                'physical_exams'         => fn ($q) => $q->latest(),
                'fremarks',
                'pneumoconiosis.results_investigation',
                'icomments_remarks',
            ])->findOrFail($patientId);

            // Legacy logic depends on the single-string category on Patient
            $category = $patient->category;

            switch ($category) {
                case 'Food Handler (COH)':
                    $certificateStatus = $this->getCertificateStatusForCityOfHarare($patient);
                    break;

                case 'Pneumoconiosis':
                    $certificateStatus = $this->getCertificateStatusForPneumoconiosis($patient);
                    break;

                case 'Pre-Employement':
                    $certificateStatus = $this->getCertificateStatusForIndustry($patient);
                    break;

                case 'In House':
                    $certificateStatus = $this->getCertificateStatusForInHouse($patient);
                    break;

                case 'Exit-Employement':
                    $certificateStatus = $this->getCertificateStatusForExitEmployment($patient);
                    break;

                case 'Exit-Pneumoconiosis':
                    $certificateStatus = $this->getCertificateStatusForExitPneumo($patient);
                    break;

                default:
                    return response()->json(['message' => 'Invalid or unsupported patient category'], 422);
            }

            $latestCertificate = $patient->certificates->first();
            if ($latestCertificate) {
                $latestCertificate->status = $certificateStatus;
                $latestCertificate->save();
            }

            return response()->json([
                'message'   => 'Certificate status updated successfully',
                'status'    => $certificateStatus,
                'certificate_id' => $latestCertificate?->id,
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'message' => 'Failed to update certificate status',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /* =========================
     * CATEGORY STATUS HELPERS
     * ========================= */

    private function getCertificateStatusForCityOfHarare(Patient $patient): string
    {
        // If there are no open items (referral/lab/radiology/prescription), rely on fremarks
        if (
            ! $patient->referral &&
            ! $patient->laboratory &&
            ! $patient->radiology &&
            ! $patient->prescription
        ) {
            // fremarks is assumed to be a relation returning a collection
            $fitToWork = optional($patient->fremarks)->where('chest_x_ray', true)->where('fit_to_work', true)->count() ?? 0;
            return $fitToWork > 0 ? 'READY' : 'MONITORING';
        }

        // Otherwise, fall back to latest physical exam BP status
        return $this->statusFromLatestBP($patient);
    }

    private function getCertificateStatusForInHouse(Patient $patient): string
    {
        if (
            ! $patient->referral &&
            ! $patient->laboratory &&
            ! $patient->radiology &&
            ! $patient->prescription
        ) {
            $fitToWork = optional($patient->fremarks)->where('chest_x_ray', true)->where('fit_to_work', true)->count() ?? 0;
            return $fitToWork > 0 ? 'READY' : 'MONITORING';
        }

        return $this->statusFromLatestBP($patient);
    }

    private function getCertificateStatusForPneumoconiosis(Patient $patient): string
    {
        if (
            ! $patient->referral &&
            ! $patient->laboratory &&
            ! $patient->radiology &&
            ! $patient->prescription
        ) {
            $ri = optional(optional($patient->pneumoconiosis)->results_investigation);
            $fitToWork = $ri
                ? $ri->where('chest_x_ray', true)
                     ->where('suitable_for_dusty', true)
                     ->where('tb_free', true)
                     ->where('referral', false)
                     ->count()
                : 0;

            return $fitToWork > 0 ? 'READY' : 'MONITORING';
        }

        return $this->statusFromLatestBP($patient);
    }

    private function getCertificateStatusForExitPneumo(Patient $patient): string
    {
        if (
            ! $patient->referral &&
            ! $patient->laboratory &&
            ! $patient->radiology &&
            ! $patient->prescription
        ) {
            $ri = optional(optional($patient->pneumoconiosis)->results_investigation);
            $fitToWork = $ri
                ? $ri->where('chest_x_ray', true)
                     ->where('suitable_for_dusty', true)
                     ->where('tb_free', true)
                     ->where('referral', false)
                     ->count()
                : 0;

            return $fitToWork > 0 ? 'READY' : 'MONITORING';
        }

        return $this->statusFromLatestBP($patient);
    }

    private function getCertificateStatusForIndustry(Patient $patient): string
    {
        if (
            ! $patient->referral &&
            ! $patient->laboratory &&
            ! $patient->radiology &&
            ! $patient->prescription
        ) {
            $fitToWork = optional($patient->icomments_remarks)->where('fit_to_work', true)->count() ?? 0;
            return $fitToWork > 0 ? 'READY' : 'MONITORING';
        }

        return $this->statusFromLatestBP($patient);
    }

    private function getCertificateStatusForExitEmployment(Patient $patient): string
    {
        if (
            ! $patient->referral &&
            ! $patient->laboratory &&
            ! $patient->radiology &&
            ! $patient->prescription
        ) {
            $fitToWork = optional($patient->icomments_remarks)->where('fit_to_work', true)->count() ?? 0;
            return $fitToWork > 0 ? 'READY' : 'MONITORING';
        }

        return $this->statusFromLatestBP($patient);
    }

    /**
     * Re-usable BP-based decision helper to minimize repetition.
     */
    private function statusFromLatestBP(Patient $patient): string
    {
        $latestPhysicalExam = $patient->physical_exams->first(); // already eager-loaded & latest
        if (!$latestPhysicalExam) {
            return 'UNKNOWN';
        }

        $bpStatus = $latestPhysicalExam->bp_status;

        if (in_array($bpStatus, [
            'Low Blood Pressure',
            'Stage 1 Hypertension',
            'Stage 2 Hypertension',
            'Hypertensive Crisis',
        ], true)) {
            return 'MONITORING';
        }

        if (in_array($bpStatus, [
            'Normal Blood Pressure',
            'Pre Hypertension',
        ], true)) {
            return 'READY';
        }

        return 'UNKNOWN';
    }

    /**
     * GET /medicals/patients/due-medicals
     * Returns patients whose certificates expire within 30 days (or already expired).
     */
    public function getPatientsWithDueMedicals()
    {
        $patients = Patient::with(['certificates', 'attendee', 'company'])
            ->whereHas('certificates', function ($query) {
                $query->where(function ($q) {
                    // released_at is set and the difference between released_at + validity(years) and today <= 30 days
                    $q->where('released_at', '<=', Carbon::now())
                      ->whereRaw('DATEDIFF(DATE_ADD(released_at, INTERVAL validity YEAR), CURDATE()) <= ?', [30]);
                });
            })
            ->get();

        return $patients;
    }

    /**
     * PATCH /medicals/certificates/{certificate}/manual-update
     * Doctor manually updates certificate status (password required).
     */
    public function doctorManualCertificateUpdate(Request $request, int $certificateId)
    {
        $validated = $request->validate([
            'password'      => ['required'],
            'status'        => ['required', 'string'],
            'update_reason' => ['nullable', 'string'],
        ]);

        $certificate = Certificate::find($certificateId);
        if (! $certificate) {
            return response()->json(['error' => 'Certificate not found'], 404);
        }

        $user = $request->user();
        if (! $user || ! Hash::check($validated['password'], $user->password)) {
            return response()->json(['error' => 'Invalid password'], 401);
        }

        // (Optional) Gate/role check can go here
        // if (! in_array($user->role_id, [4, 6], true)) { ... }

        $certificate->status        = $validated['status'];
        $certificate->update_reason = $validated['update_reason'] ?? null;
        $certificate->released_at   = Carbon::today(); // mirror legacy behavior
        $certificate->save();

        return response()->json(['message' => 'Certificate status updated successfully']);
    }

    /**
     * POST /medicals/patients/{patient}/certificate/renew
     * Renews medical certificate, making a PENDING one unless a same-day certificate exists.
     * If a same-day certificate exists and is PENDING, auto-release it (legacy flow).
     */
    public function renewMedicalCertificate(Request $request, int $patientId)
    {
        /** @var Patient $patient */
        $patient = Patient::findOrFail($patientId);

        // Optional: update category from request (legacy behavior)
        if ($request->filled('medicalCategory')) {
            $patient->update([
                'category' => $request->input('medicalCategory'),
            ]);
        }

        $existingToday = $patient->certificates()
            ->whereDate('created_at', Carbon::today())
            ->latest()
            ->first();

        if ($existingToday) {
            if ($existingToday->status === 'PENDING') {
                $existingToday->update([
                    'status'        => 'RELEASED',
                    'update_reason' => 'Auto-released due to duplicate renewal request',
                ]);

                return response()->json([
                    'message' => 'Existing pending certificate released instead of creating a new one.',
                ], 200);
            }

            return response()->json([
                'message' => 'A certificate has already been created today.',
            ], 400);
        }

        $certificateData = [
            'patient_id'           => $patientId,
            'status'               => 'PENDING',
            'validity'             => 1, // in years
            'released_at'          => Carbon::now(),
            'certificate_number'   => uniqid('CERT-'),
            'certificate_location' => $request->input('certificateLocation', 'HARARE'),
            'update_reason'        => 'Renewal requested by doctor',
        ];

        Certificate::create($certificateData);

        return response()->json([
            'message' => 'Medical certificate renewed successfully.',
        ], 200);
    }

    /**
     * POST /medicals/patients/{patient}/certificate/renew-simple
     * Very simple renewal: create MedicalRecord (current year) + PENDING certificate.
     */
    public function renewPatientCertificate(Request $request, int $patientId)
    {
        MedicalRecord::create([
            'patient_id'   => $patientId,
            'year_of_exam' => (int) date('Y'),
        ]);

        Certificate::create([
            'patient_id' => $patientId,
            'validity'   => 1,
            'status'     => 'PENDING',
        ]);

        return response()->json([
            'message' => 'Certificate successfully renewed',
        ], 200);
    }
}
