<?php
namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class CompanyController extends Controller
{
    public function index(Request $request)
    {
        $search  = $request->string('search')->toString();
        $perPage = (int) ($request->integer('per_page') ?: 15);

        $query = Company::query();

        if ($search !== '') {
            $like = '%' . trim($search) . '%';
            $query->where('company_name', 'like', $like);
        }

        $companies = $query
            ->orderBy('company_name')
            ->paginate($perPage)
            ->withQueryString();

        return Inertia::render('Medicals/Companies/Index', [
            'companies' => $companies->through(fn(Company $c) => [
                'id'             => $c->id,
                'company_name'   => $c->company_name,
                'address'        => $c->address,
                'site_telephone' => $c->site_telephone,
                'company_email'  => $c->company_email,
                'contact_person' => $c->contact_person,
                'province'       => $c->province,
                'designation'    => $c->designation,
                'contact_number' => $c->contact_number,
            ]),
            'filters'   => [
                'search'   => $search,
                'per_page' => $perPage,
            ],
        ]);
    }

    /**
     * GET /medicals/companies/create
     */
    public function create()
    {
        return Inertia::render('Medicals/Companies/Create');
    }

    /**
     * POST /medicals/companies
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_name'   => ['required', 'string', 'max:255'],
            'address'        => ['required', 'string', 'max:1000'],
            'site_telephone' => ['required', 'string', 'max:255'],
            // Unique on the same (medicals) connection by passing the Model::class
            'company_email'  => ['required', 'email', 'max:255', Rule::unique(Company::class, 'company_email')],
            'contact_person' => ['nullable', 'string', 'max:255'],
            'province'       => ['nullable', 'string', 'max:255'],
            'designation'    => ['nullable', 'string', 'max:255'],
            'contact_number' => ['nullable', 'string', 'max:255'],
        ]);

        $company = Company::create($data);

        return redirect()
            ->route('medicals.companies.index')
            ->with('success', 'Company created successfully.');
    }

    /**
     * GET /medicals/companies/{company}
     */
    public function show(Company $company)
    {
        return Inertia::render('Medicals/Companies/Show', [
            'company' => [
                'id'             => $company->id,
                'company_name'   => $company->company_name,
                'address'        => $company->address,
                'site_telephone' => $company->site_telephone,
                'company_email'  => $company->company_email,
                'contact_person' => $company->contact_person,
                'province'       => $company->province,
                'designation'    => $company->designation,
                'contact_number' => $company->contact_number,
                'created_at'     => optional($company->created_at)?->toIso8601String(),
                'updated_at'     => optional($company->updated_at)?->toIso8601String(),
            ],
        ]);
    }

    /**
     * GET /medicals/companies/{company}/edit
     */
    public function edit(Company $company)
    {
        return Inertia::render('Medicals/Companies/Edit', [
            'company' => [
                'id'             => $company->id,
                'company_name'   => $company->company_name,
                'address'        => $company->address,
                'site_telephone' => $company->site_telephone,
                'company_email'  => $company->company_email,
                'contact_person' => $company->contact_person,
                'province'       => $company->province,
                'designation'    => $company->designation,
                'contact_number' => $company->contact_number,
            ],
        ]);
    }

    /**
     * PUT /medicals/companies/{company}
     */
    public function update(Request $request, Company $company)
    {
        $data = $request->validate([
            'company_name'   => ['required', 'string', 'max:255'],
            'address'        => ['required', 'string', 'max:1000'],
            'site_telephone' => ['required', 'string', 'max:255'],
            // Ignore current record (still validates on medicals connection)
            'company_email'  => [
                'required',
                'email',
                'max:255',
                Rule::unique(Company::class, 'company_email')->ignore($company->getKey()),
            ],
            'contact_person' => ['nullable', 'string', 'max:255'],
            'province'       => ['nullable', 'string', 'max:255'],
            'designation'    => ['nullable', 'string', 'max:255'],
            'contact_number' => ['nullable', 'string', 'max:255'],
        ]);

        $company->fill($data)->save();

        return redirect()
            ->route('medicals.companies.show', $company->id)
            ->with('success', 'Company updated successfully.');
    }

    /**
     * DELETE /medicals/companies/{company}
     */
    public function destroy(Company $company)
    {
        DB::connection('medicals')->transaction(function () use ($company) {
            $company->delete();
        });

        return redirect()
            ->route('medicals.companies.index')
            ->with('success', 'Company deleted successfully.');
    }
}
