<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Disease;
use App\Models\Medicals\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class DiseaseController extends Controller
{
    // GET /medicals/diseases
    public function index(Request $request): Response
    {
        $search   = (string) $request->string('search');
        $perPage  = (int) ($request->integer('per_page') ?: 20);

        $query = Disease::query();

        if ($search !== '') {
            $query->where('name', 'like', "%{$search}%");
        }

        $diseases = $query
            ->orderBy('name')
            ->paginate($perPage)
            ->withQueryString();

        return Inertia::render('Medicals/Diseases/Index', [
            'diseases' => $diseases,
            'filters'  => [
                'search'   => $search,
                'per_page' => $perPage,
            ],
        ]);
    }

    // GET /medicals/diseases/create
    public function create(): Response
    {
        return Inertia::render('Medicals/Diseases/Create');
    }

    // POST /medicals/diseases
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:255'],
        ]);

        Disease::create($data);

        return redirect()
            ->route('medicals.diseases.index')
            ->with('success', 'Disease created.');
    }

    // GET /medicals/diseases/{disease}/edit
    public function edit(Disease $disease): Response
    {
        return Inertia::render('Medicals/Diseases/Edit', [
            'disease' => [
                'id'   => $disease->id,
                'name' => $disease->name,
            ],
        ]);
    }

    // PATCH /medicals/diseases/{disease}
    public function update(Request $request, Disease $disease)
    {
        $data = $request->validate([
            'name' => ['required','string','max:255'],
        ]);

        $disease->update($data);

        return redirect()
            ->route('medicals.diseases.index')
            ->with('success', 'Disease updated.');
    }

    // DELETE /medicals/diseases/{disease}
    public function destroy(Disease $disease)
    {
        // Detach all patient pivots for safety
        DB::transaction(function () use ($disease) {
            $disease->patients()->detach();
            $disease->delete();
        });

        return redirect()
            ->route('medicals.diseases.index')
            ->with('success', 'Disease deleted.');
    }

    /**
     * Optional: Attach/update a Disease on a Patient (pivot).
     * POST /medicals/diseases/{disease}/patients/{patient}
     */
    public function updatePatientDisease(Request $request, Disease $disease, Patient $patient)
    {
        $payload = $request->validate([
            'has_disease' => ['nullable','boolean'],
            'date'        => ['nullable','string'],
        ]);

        DB::transaction(function () use ($patient, $disease, $payload) {
            $patient->diseases()->syncWithoutDetaching([
                $disease->id => [
                    'has_disease' => $payload['has_disease'] ?? null,
                    'date'        => $payload['date'] ?? null,
                    'created_at'  => now(),
                    'updated_at'  => now(),
                ],
            ]);
        });

        $diseases = $patient->diseases()
            ->wherePivot('has_disease', true)
            ->withPivot('date')
            ->get()
            ->map(fn($d) => [
                'name'        => $d->name,
                'has_disease' => (bool) $d->pivot->has_disease,
                'date'        => $d->pivot->date,
            ]);

        return response()->json([
            'message'  => 'Disease updated successfully!',
            'diseases' => $diseases,
        ]);
    }
}
