<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Illness;
use App\Models\Medicals\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class IllnessController extends Controller
{
    /**
     * List illnesses with optional search + pagination (Inertia).
     */
    public function index(Request $request)
    {
        $search  = (string) $request->string('search');
        $perPage = (int) ($request->integer('per_page') ?: 20);

        $query = Illness::query();

        if ($search !== '') {
            $query->where('illness_name', 'like', "%{$search}%");
        }

        $paginator = $query
            ->orderBy('illness_name')
            ->paginate($perPage)
            ->withQueryString();

        return Inertia::render('Medicals/Illnesses/Index', [
            'illnesses' => $paginator,
            'filters'   => [
                'search'   => $search,
                'per_page' => $perPage,
            ],
        ]);
    }

    /**
     * Show create page (optional; can be a modal on index).
     */
    public function create()
    {
        return Inertia::render('Medicals/Illnesses/Create');
    }

    /**
     * Store a new illness.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'illness_name' => 'required|string|max:255',
        ]);

        Illness::create($data);

        return redirect()
            ->route('medicals.illnesses.index')
            ->with('success', 'Illness created successfully.');
    }

    /**
     * Show a single illness (optional).
     */
    public function show(Illness $illness)
    {
        return Inertia::render('Medicals/Illnesses/Show', [
            'illness' => [
                'id'           => $illness->id,
                'illness_name' => $illness->illness_name,
                'created_at'   => $illness->created_at,
                'updated_at'   => $illness->updated_at,
            ],
        ]);
    }

    /**
     * Edit page (optional; can be a modal on index).
     */
    public function edit(Illness $illness)
    {
        return Inertia::render('Medicals/Illnesses/Edit', [
            'illness' => [
                'id'           => $illness->id,
                'illness_name' => $illness->illness_name,
            ],
        ]);
    }

    /**
     * Update an illness' name (mirrors your illnessUpdate).
     */
    public function update(Request $request, Illness $illness)
    {
        $data = $request->validate([
            'illness_name' => 'required|string|max:255',
        ]);

        $illness->update($data);

        return redirect()
            ->route('medicals.illnesses.index')
            ->with('success', 'Illness updated successfully.');
    }

    /**
     * Delete an illness and detach from all patients (mirrors your deleteAnddetach).
     */
    public function destroy(Illness $illness)
    {
        DB::transaction(function () use ($illness) {
            // Detach from all patients via pivot
            $illness->patients()->detach();
            $illness->delete();
        });

        return redirect()
            ->route('medicals.illnesses.index')
            ->with('success', 'Illness deleted successfully.');
    }

    /**
     * Update a patient's illness pivot (has_illness, treatment_year) safely.
     * Mirrors your updateIllness(Request $request, $id, $patientId).
     *
     * Accepts:
     *  - has_illness: boolean (required)
     *  - treatment_year: numeric|nullable
     *
     * Returns JSON (used inline in UIs).
     */
    public function updateIllness(Request $request, int $illnessId, int $patientId)
    {
        $patient = Patient::findOrFail($patientId);
        $illness = Illness::findOrFail($illnessId);

        $update = $request->validate([
            'has_illness'     => 'required|boolean',
            'treatment_year'  => 'nullable|numeric',
        ]);

        DB::transaction(function () use ($patient, $illnessId, $update) {
            $patient->illnesses()->syncWithoutDetaching([
                $illnessId => [
                    'has_illness'    => $update['has_illness'],
                    'treatment_year' => $update['treatment_year'],
                    'created_at'     => now(),
                    'updated_at'     => now(),
                ],
            ]);
        });

        // Return all positive illnesses for the patient (formatted)
        $illnesses = $patient->illnesses()
            ->wherePivot('has_illness', true)
            ->withPivot('treatment_year')
            ->orderBy('illness_name')
            ->get();

        $payload = $illnesses->map(function (Illness $i) {
            return [
                'name'           => $i->illness_name,
                'has_illness'    => (bool) $i->pivot->has_illness,
                'treatment_year' => $i->pivot->treatment_year,
            ];
        });

        return response()->json([
            'message'   => 'Illness updated successfully!',
            'illness'   => [
                'id'           => $illness->id,
                'illness_name' => $illness->illness_name,
            ],
            'illnesses' => $payload,
        ], 200);
    }
}
