<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Carbon\Carbon;

class PatientsController extends Controller
{
    /**
     * GET /medicals/patients
     * Filters:
     *  - search (last_name / national_id / first_name via attendee)
     *  - location (latest certificate location)
     *  - company (company id)
     *  - swab_status (patients.swab_result)
     *  - certificate_status (latest certificate status)
     *  - per_page (default 50)
     *  - sort: created_at | cert_created_at (latest cert) | last_name
     *  - direction: asc|desc
     */
    public function index(Request $request)
    {
        $search            = (string) $request->string('search');
        $location          = (string) $request->string('location');
        $companyId         = $request->input('company');
        $swabStatus        = (string) $request->string('swab_status');
        $certificateStatus = (string) $request->string('certificate_status');
        $perPage           = (int) ($request->integer('per_page') ?: 50);
        $sort              = $request->input('sort', 'cert_created_at'); // default by latest certificate
        $direction         = strtolower($request->input('direction', 'desc')) === 'asc' ? 'asc' : 'desc';

        $query = Patient::query()
            ->with(['attendee', 'company', 'certificates', 'swabs', 'physical_exams'])
            ->withMax('certificates as latest_certificate_created_at', 'created_at');

        // Search on attendee names / ID
        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->whereHas('attendee', function ($qa) use ($search) {
                    $qa->where('first_name', 'like', "%{$search}%")
                       ->orWhere('last_name', 'like', "%{$search}%")
                       ->orWhere('national_id', 'like', "%{$search}%");
                });
            });
        }

        // Filter: latest certificate location
        if ($location !== '') {
            $query->whereHas('certificates', function ($q) use ($location) {
                $q->where('certificate_location', $location)
                  ->whereColumn('id', DB::raw('(SELECT MAX(id) FROM certificates c2 WHERE c2.patient_id = certificates.patient_id)'));
            });
        }

        // Filter: company
        if (!empty($companyId)) {
            $query->whereHas('company', fn ($q) => $q->where('id', $companyId));
        }

        // Filter: swab status (on patients.swab_result)
        if ($swabStatus !== '') {
            $query->where('swab_result', $swabStatus);
        }

        // Filter: certificate status (latest)
        if ($certificateStatus !== '') {
            $query->whereHas('certificates', function ($q) use ($certificateStatus) {
                $q->where('status', $certificateStatus)
                  ->whereColumn('id', DB::raw('(SELECT MAX(id) FROM certificates c2 WHERE c2.patient_id = certificates.patient_id)'));
            });
        }

        // Sorting
        if ($sort === 'last_name') {
            $query->joinRelationship('attendee')  // requires Laravel 9+; otherwise use join
                  ->orderBy('attendee.last_name', $direction)
                  ->select('patients.*'); // ensure columns correct
        } elseif ($sort === 'created_at') {
            $query->orderBy('patients.created_at', $direction);
        } else { // cert_created_at (default)
            $query->orderBy('latest_certificate_created_at', $direction)
                  ->orderBy('patients.created_at', 'desc');
        }

        $paginator = $query->paginate($perPage)->withQueryString();

        // Map like AllPatientsResource but inline
        $collection = $paginator->getCollection()->map(function (Patient $p) {
            $att = $p->attendee;
            $comp = $p->company;

            // compute age from attendee DOB (if available)
            $age = null;
            if ($att?->date_of_birth) {
                try {
                    $dob = Carbon::parse($att->date_of_birth);
                    $age = $dob->diffInYears(Carbon::now());
                } catch (\Throwable $e) {
                    $age = null;
                }
            }

            // latest certificate (by created_at)
            $latestCert = $p->certificates->sortByDesc('created_at')->first();

            return [
                'id'               => $p->id,
                'first_name'       => $att?->first_name,
                'last_name'        => $att?->last_name,
                'company'          => $comp?->company_name,
                'company_email'    => $comp?->company_email,
                'company_phone'    => $comp?->contact_number,
                'company_phone2'   => $comp?->site_telephone,
                'age'              => $age,
                'national_id'      => $att?->national_id,
                'date_of_birth'    => $att?->date_of_birth,
                'phone_number'     => $att?->phone_number,
                'updated_at'       => $att?->updated_at,
                'created_at'       => $att?->created_at,
                'employee_number'  => $att?->employee_number,
                'last_x_ray'       => $p->last_x_ray,
                'category'         => $p->category,
                'certificate_status' => $latestCert?->status ?? 'PENDING',
                // if you need full swabs/certs/exams on index, keep; otherwise you can omit for performance
                'swabs'            => $p->swabs->map(fn($s) => [
                    'id'         => $s->id,
                    'patient_id' => $s->patient_id,
                    'status'     => $s->status,
                    'file'       => $s->file,
                    'comment'    => $s->comment,
                    'created_at' => $s->created_at,
                    'updated_at' => $s->updated_at,
                ]),
                'certificates'     => $p->certificates->sortByDesc('created_at')->map(fn($c) => [
                    'id'                   => $c->id,
                    'patient_id'           => $c->patient_id,
                    'status'               => $c->status,
                    'validity'             => $c->validity,
                    'released_at'          => $c->released_at,
                    'certificate_number'   => $c->certificate_number,
                    'update_reason'        => $c->update_reason,
                    'certificate_location' => $c->certificate_location,
                    'created_at'           => $c->created_at,
                    'updated_at'           => $c->updated_at,
                ]),
                'physical_records' => $p->physical_exams->map(fn($pe) => [
                    'id'                => $pe->id,
                    'bmi'               => number_format($pe->bmi, 2),
                    'bmi_status'        => $pe->bmi_status,
                    'bp_dia'            => $pe->bp_dia,
                    'bp_repeat_dia'     => $pe->bp_repeat_dia,
                    'bp_repeat_sys'     => $pe->bp_repeat_sys,
                    'bp_status'         => $pe->bp_status,
                    'bp_sys'            => $pe->bp_sys,
                    'created_at'        => $pe->created_at,
                    'first_bp_time'     => $pe->first_bp_time,
                    'height'            => $pe->height,
                    'referral_comment'  => $pe->referral_comment,
                    'temp'              => $pe->temp,
                    'pulse'             => $pe->pulse,
                    'last_bp_time'      => $pe->last_bp_time,
                    'left_vision'       => $pe->left_vision,
                    'patient_id'        => $pe->patient_id,
                    'right_vision'      => $pe->right_vision,
                    'updated_at'        => $pe->updated_at,
                    'weight'            => $pe->weight,
                ]),
            ];
        });

        $paginator->setCollection($collection);

        return Inertia::render('Medicals/Patients/Index', [
            'patients' => $paginator,
            'filters'  => [
                'search'            => $search,
                'location'          => $location,
                'company'           => $companyId,
                'swab_status'       => $swabStatus,
                'certificate_status'=> $certificateStatus,
                'per_page'          => $perPage,
                'sort'              => $sort,
                'direction'         => $direction,
            ],
        ]);
    }

    /**
     * GET /medicals/patients/{patient}
     * Detailed patient view.
     */
     public function show(Patient $patient)
    {
        // Load everything we need, including categories on both patient and attendee.
        $patient->load([
            'attendee',
            'company',
            'certificates'   => fn ($q) => $q->orderByDesc('created_at'),
            'swabs'          => fn ($q) => $q->orderByDesc('created_at'),
            'physical_exams' => fn ($q) => $q->orderByDesc('created_at'),
            'categories:id,slug,name,form_key',             // ⬅️ NEW: patient categories
            'attendee.categories:id,slug,name,form_key',    // ⬅️ Optional: attendee categories too
        ]);

        $att  = $patient->attendee;
        $comp = $patient->company;
        $latestCert = $patient->certificates->first();

        // Derive the "latest examination categories"
        // In this model, categories attached to the Patient represent the latest intended examination set.
        $latestExamCategories = $patient->categories
            ->map(fn ($c) => [
                'id'       => $c->id,
                'slug'     => $c->slug,
                'name'     => $c->name,
                'form_key' => $c->form_key, // optional helper to decide which form to show
            ])
            ->values();

        // Age helper
        $age = null;
        if ($att?->date_of_birth) {
            try {
                $age = Carbon::parse($att->date_of_birth)->diffInYears(now());
            } catch (\Throwable $e) {
                $age = null;
            }
        }

        return Inertia::render('Medicals/Patients/Show', [
            'patient' => [
                'id'               => $patient->id,
                'first_name'       => $att?->first_name,
                'last_name'        => $att?->last_name,
                'age'              => $age,
                'employee_number'  => $att?->employee_number,
                'national_id'      => $att?->national_id,
                'date_of_birth'    => $att?->date_of_birth,
                'gender'           => $att?->gender,
                'phone_number'     => $att?->phone_number,
                'company'          => $comp ? [
                    'id'   => $comp->id,
                    'name' => $comp->company_name,
                    'email'=> $comp->company_email,
                    'phone'=> $comp->contact_number,
                ] : null,

                // Legacy single category string (kept for backward compatibility)
                'category'         => $patient->category,

                // NEW: full category collection for the planned/latest examination
                'categories'             => $latestExamCategories,     // patient-level categories
                'latest_exam_categories' => $latestExamCategories,     // alias for clarity on the frontend

                'last_x_ray'       => $patient->last_x_ray,
                'created_at'       => $att?->created_at,
                'updated_at'       => $att?->updated_at,

                'latest_certificate'=> $latestCert ? [
                    'id'                   => $latestCert->id,
                    'status'               => $latestCert->status,
                    'certificate_location' => $latestCert->certificate_location,
                    'created_at'           => $latestCert->created_at,
                    'updated_at'           => $latestCert->updated_at,
                ] : null,

                'certificates'     => $patient->certificates->map(fn($c) => [
                    'id'                   => $c->id,
                    'status'               => $c->status,
                    'certificate_location' => $c->certificate_location,
                    'created_at'           => $c->created_at,
                    'updated_at'           => $c->updated_at,
                    'validity'             => $c->validity,
                    'released_at'          => $c->released_at,
                    'certificate_number'   => $c->certificate_number,
                    'update_reason'        => $c->update_reason,
                ]),

                'swabs'            => $patient->swabs->map(fn($s) => [
                    'id'         => $s->id,
                    'status'     => $s->status,
                    'file'       => $s->file,
                    'comment'    => $s->comment,
                    'created_at' => $s->created_at,
                    'updated_at' => $s->updated_at,
                ]),

                'physical_exams'   => $patient->physical_exams->map(fn($pe) => [
                    'id'               => $pe->id,
                    'bmi'              => number_format($pe->bmi, 2),
                    'bmi_status'       => $pe->bmi_status,
                    'bp_dia'           => $pe->bp_dia,
                    'bp_repeat_dia'    => $pe->bp_repeat_dia,
                    'bp_repeat_sys'    => $pe->bp_repeat_sys,
                    'bp_status'        => $pe->bp_status,
                    'bp_sys'           => $pe->bp_sys,
                    'created_at'       => $pe->created_at,
                    'first_bp_time'    => $pe->first_bp_time,
                    'height'           => $pe->height,
                    'referral_comment' => $pe->referral_comment,
                    'temp'             => $pe->temp,
                    'pulse'            => $pe->pulse,
                    'last_bp_time'     => $pe->last_bp_time,
                    'left_vision'      => $pe->left_vision,
                    'patient_id'       => $pe->patient_id,
                    'right_vision'     => $pe->right_vision,
                    'updated_at'       => $pe->updated_at,
                    'weight'           => $pe->weight,
                ]),
            ],
        ]);
    }

    /**
     * DELETE /medicals/patients/{patient}
     * Mirrors your legacy cascade delete.
     */
    public function destroy(Patient $patient)
    {
        try {
            DB::beginTransaction();

            // Certificates
            $patient->certificates()->each(function ($cert) {
                $cert->delete();
            });

            // Physical exams
            $patient->physical_exams()->each(function ($exam) {
                $exam->delete();
            });

            // X-rays (if relation exists)
            if (method_exists($patient, 'xrays')) {
                $patient->xrays()->each(function ($x) {
                    $x->delete();
                });
            }

            // Detach many-to-many
            if (method_exists($patient, 'illnesses'))     $patient->illnesses()->detach();
            if (method_exists($patient, 'tobacco_uses'))  $patient->tobacco_uses()->detach();

            // Delete attendee (if exists)
            if ($patient->attendee) {
                $patient->attendee->delete();
            }

            $patient->delete();

            DB::commit();

            return redirect()
                ->route('medicals.patients.index')
                ->with('success', 'Patient deleted successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to delete patient: ' . $e->getMessage());
        }
    }

     /** Pneumoconiosis list -> its own page */
    public function pneumoPatients(Request $request)
    {
        return $this->renderCategoryPatientsToView(
            $request,
            'Pneumoconiosis',
            'Medicals/Patients/PneumoIndex' // <-- unique page
        );
    }

    /** Food Handler (COH) list -> its own page */
    public function cofhPatients(Request $request)
    {
        return $this->renderCategoryPatientsToView(
            $request,
            'Food Handler (COH)',
            'Medicals/Patients/COFHIndex' // <-- unique page
        );
    }

    /** Pre-Employement list -> its own page */
    public function industryPatients(Request $request)
    {
        return $this->renderCategoryPatientsToView(
            $request,
            'Pre-Employement',
            'Medicals/Patients/IndustryIndex' // <-- unique page
        );
    }

    /** Exit-Employement list -> its own page */
    public function exitEmploymentPatients(Request $request)
    {
        return $this->renderCategoryPatientsToView(
            $request,
            'Exit-Employement',
            'Medicals/Patients/ExitEmploymentIndex' // <-- unique page
        );
    }

    /** Exit-Pneumoconiosis list -> its own page */
    public function exitPneumoPatients(Request $request)
    {
        return $this->renderCategoryPatientsToView(
            $request,
            'Exit-Pneumoconiosis',
            'Medicals/Patients/ExitPneumoIndex' // <-- unique page
        );
    }

    /**
     * Same logic as index(), but with a fixed category AND a custom view.
     * The payload (patients paginator + filters) matches index().
     */
    private function renderCategoryPatientsToView(Request $request, string $category, string $view)
    {
        $search            = (string) $request->string('search');
        $location          = (string) $request->string('location');
        $companyId         = $request->input('company');
        $swabStatus        = (string) $request->string('swab_status');
        $certificateStatus = (string) $request->string('certificate_status');
        $perPage           = (int) ($request->integer('per_page') ?: 50);
        $sort              = $request->input('sort', 'cert_created_at');
        $direction         = strtolower($request->input('direction', 'desc')) === 'asc' ? 'asc' : 'desc';

        $query = Patient::query()
            ->with(['attendee', 'company', 'certificates', 'swabs', 'physical_exams'])
            ->withMax('certificates as latest_certificate_created_at', 'created_at')
            ->where('category', $category);

        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->whereHas('attendee', function ($qa) use ($search) {
                    $qa->where('first_name', 'like', "%{$search}%")
                       ->orWhere('last_name', 'like', "%{$search}%")
                       ->orWhere('national_id', 'like', "%{$search}%");
                });
            });
        }

        if ($location !== '') {
            $query->whereHas('certificates', function ($q) use ($location) {
                $q->where('certificate_location', $location)
                  ->whereColumn('id', DB::raw('(SELECT MAX(id) FROM certificates c2 WHERE c2.patient_id = certificates.patient_id)'));
            });
        }

        if (!empty($companyId)) {
            $query->whereHas('company', fn ($q) => $q->where('id', $companyId));
        }

        if ($swabStatus !== '') {
            $query->where('swab_result', $swabStatus);
        }

        if ($certificateStatus !== '') {
            $query->whereHas('certificates', function ($q) use ($certificateStatus) {
                $q->where('status', $certificateStatus)
                  ->whereColumn('id', DB::raw('(SELECT MAX(id) FROM certificates c2 WHERE c2.patient_id = certificates.patient_id)'));
            });
        }

        if ($sort === 'last_name') {
            $query->joinRelationship('attendee')
                  ->orderBy('attendee.last_name', $direction)
                  ->select('patients.*');
        } elseif ($sort === 'created_at') {
            $query->orderBy('patients.created_at', $direction);
        } else {
            $query->orderBy('latest_certificate_created_at', $direction)
                  ->orderBy('patients.created_at', 'desc');
        }

        $paginator = $query->paginate($perPage)->withQueryString();

        $collection = $paginator->getCollection()->map(function (Patient $p) {
            $att  = $p->attendee;
            $comp = $p->company;

            $age = null;
            if ($att?->date_of_birth) {
                try {
                    $age = Carbon::parse($att->date_of_birth)->diffInYears(now());
                } catch (\Throwable $e) { $age = null; }
            }

            $latestCert = $p->certificates->sortByDesc('created_at')->first();

            return [
                'id'                 => $p->id,
                'first_name'         => $att?->first_name,
                'last_name'          => $att?->last_name,
                'company'            => $comp?->company_name,
                'company_email'      => $comp?->company_email,
                'company_phone'      => $comp?->contact_number,
                'company_phone2'     => $comp?->site_telephone,
                'age'                => $age,
                'national_id'        => $att?->national_id,
                'date_of_birth'      => $att?->date_of_birth,
                'phone_number'       => $att?->phone_number,
                'updated_at'         => $att?->updated_at,
                'created_at'         => $att?->created_at,
                'employee_number'    => $att?->employee_number,
                'last_x_ray'         => $p->last_x_ray,
                'category'           => $p->category,
                'certificate_status' => $latestCert?->status ?? 'PENDING',
                'swabs'              => $p->swabs->map(fn($s) => [
                    'id'         => $s->id,
                    'patient_id' => $s->patient_id,
                    'status'     => $s->status,
                    'file'       => $s->file,
                    'comment'    => $s->comment,
                    'created_at' => $s->created_at,
                    'updated_at' => $s->updated_at,
                ]),
                'certificates'       => $p->certificates->sortByDesc('created_at')->map(fn($c) => [
                    'id'                   => $c->id,
                    'patient_id'           => $c->patient_id,
                    'status'               => $c->status,
                    'validity'             => $c->validity,
                    'released_at'          => $c->released_at,
                    'certificate_number'   => $c->certificate_number,
                    'update_reason'        => $c->update_reason,
                    'certificate_location' => $c->certificate_location,
                    'created_at'           => $c->created_at,
                    'updated_at'           => $c->updated_at,
                ]),
                'physical_records'   => $p->physical_exams->map(fn($pe) => [
                    'id'                => $pe->id,
                    'bmi'               => number_format($pe->bmi, 2),
                    'bmi_status'        => $pe->bmi_status,
                    'bp_dia'            => $pe->bp_dia,
                    'bp_repeat_dia'     => $pe->bp_repeat_dia,
                    'bp_repeat_sys'     => $pe->bp_repeat_sys,
                    'bp_status'         => $pe->bp_status,
                    'bp_sys'            => $pe->bp_sys,
                    'created_at'        => $pe->created_at,
                    'first_bp_time'     => $pe->first_bp_time,
                    'height'            => $pe->height,
                    'referral_comment'  => $pe->referral_comment,
                    'temp'              => $pe->temp,
                    'pulse'             => $pe->pulse,
                    'last_bp_time'      => $pe->last_bp_time,
                    'left_vision'       => $pe->left_vision,
                    'patient_id'        => $pe->patient_id,
                    'right_vision'      => $pe->right_vision,
                    'updated_at'        => $pe->updated_at,
                    'weight'            => $pe->weight,
                ]),
            ];
        });

        $paginator->setCollection($collection);

        return Inertia::render($view, [
            'patients' => $paginator,
            'filters'  => [
                'search'             => $search,
                'location'           => $location,
                'company'            => $companyId,
                'swab_status'        => $swabStatus,
                'certificate_status' => $certificateStatus,
                'per_page'           => $perPage,
                'sort'               => $sort,
                'direction'          => $direction,
                'category'           => $category, // helpful to show a header/badge
            ],
        ]);
    }






}
