<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Patient;
use App\Models\Medicals\PhysicalExam;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class PhysicalExamController extends Controller
{
    // ─────────────────────────────────────────────────────────────
    // Helpers
    // ─────────────────────────────────────────────────────────────

    /**
     * Convert a height input to meters.
     * Rules:
     *  - If it contains a dot => treat as meters (return as float).
     *  - If it's an integer with >= 3 digits => treat as centimeters (177 -> 1.77).
     *  - If it's empty/invalid => return null.
     */
    private static function parseHeightToMeters(?string $raw): ?float
    {
        if ($raw === null) {
            return null;
        }
        $v = trim((string) $raw);
        if ($v === '') {
            return null;
        }

        // Keep digits and dot
        $v = preg_replace('/[^0-9.]/', '', $v) ?? '';

        if ($v === '') {
            return null;
        }

        if (str_contains($v, '.')) {
            // already meters-ish; clamp to 2 decimals
            $num = (float) $v;
            if ($num <= 0) {
                return null;
            }
            return round($num, 2);
        }

        // No dot present: treat >=3 digits as centimeters
        if (strlen($v) >= 3 && ctype_digit($v)) {
            $cm = (int) $v;
            if ($cm <= 0) {
                return null;
            }
            return round($cm / 100, 2);
        }

        // Too short or odd input -> treat as meters if sensible
        $num = (float) $v;
        if ($num > 0 && $num < 3.0) {
            return round($num, 2);
        }

        return null;
    }

    private static function calculateBmi(?float $weightKg, ?float $heightM): ?float
    {
        if (!$weightKg || !$heightM || $heightM <= 0) {
            return null;
        }
        return round($weightKg / ($heightM * $heightM), 2);
    }

    private static function bmiStatus(?float $bmi): ?string
    {
        if ($bmi === null) {
            return null;
        }
        if ($bmi <= 18.5) return 'UNDERWEIGHT';
        if ($bmi <= 24.9) return 'NORMAL';
        if ($bmi <= 29.9) return 'OVERWEIGHT';
        return 'OBESE';
    }

    public static function calculateBloodPressureStatus(float $sys, float $dia): string
    {
        if ($sys < 90 && $dia < 60) {
            return "Low Blood Pressure";
        } elseif (($sys < 120 && $sys >= 90) && ($dia <= 80 && $dia >= 60)) {
            return "Normal Blood Pressure";
        } elseif (($sys >= 120 && $sys <= 139 && $dia <= 80) || ($dia >= 80 && $dia <= 89 && $sys <= 139)) {
            return "Pre Hypertension";
        } elseif (($sys >= 140 && $sys <= 159 && $dia <= 99) || ($dia >= 90 && $dia <= 99 && $sys <= 159)) {
            return "Stage 1 Hypertension";
        } elseif (($sys >= 160 && $sys <= 179 && $dia <= 119) || ($dia >= 100 && $dia <= 119 && $sys <= 179)) {
            return "Stage 2 Hypertension";
        } elseif ($sys >= 180 || $dia >= 120) {
            return "Hypertensive Crisis";
        } else {
            return "Invalid Blood Pressure Reading";
        }
    }

    private static function mapPatientForView(Patient $p): array
    {
        $att  = $p->attendee;     // if you have attendee relation
        $comp = $p->company;      // if you have company relation

        return [
            'id'             => $p->id,
            'first_name'     => $att?->first_name ?? null,
            'last_name'      => $att?->last_name ?? null,
            'gender'         => $att?->gender ?? null,
            'date_of_birth'  => $att?->date_of_birth ?? null,
            'phone_number'   => $att?->phone_number ?? null,
            'category'       => $p->category,
            'company'        => $comp?->company_name,
            'created_at'     => $p->created_at?->toDateTimeString(),
            'updated_at'     => $p->updated_at?->toDateTimeString(),
        ];
    }

    private static function mapExamForView(?PhysicalExam $e): ?array
    {
        if (!$e) return null;

        return [
            'id'               => $e->id,
            'created_at'       => optional($e->created_at)?->toDateTimeString(),
            'updated_at'       => optional($e->updated_at)?->toDateTimeString(),
            'height'           => $e->height,        // meters
            'weight'           => $e->weight,        // kg
            'bmi'              => $e->bmi,
            'bmi_status'       => $e->bmi_status,
            'bp_sys'           => $e->bp_sys,
            'bp_dia'           => $e->bp_dia,
            'bp_status'        => $e->bp_status,
            'left_vision'      => $e->left_vision,
            'right_vision'     => $e->right_vision,
            'temp'             => $e->temp,
            'pulse'            => $e->pulse,
            'blood_sugar'      => $e->blood_sugar,
            'first_bp_time'    => optional($e->first_bp_time)?->toDateTimeString(),
            'last_bp_time'     => optional($e->last_bp_time)?->toDateTimeString(),
            'referral_comment' => $e->referral_comment,
        ];
    }

    // ─────────────────────────────────────────────────────────────
    // Pages / Actions
    // ─────────────────────────────────────────────────────────────

    /**
     * GET /medicals/patients/{patient}/exams/start
     * Shows the Start Exam page.
     */
    public function start(int $patientId)
    {
        /** @var Patient $patient */
        $patient = Patient::with(['attendee', 'company'])->findOrFail($patientId);

        $latestExam = PhysicalExam::query()
            ->where('patient_id', $patient->id)
            ->orderByDesc('created_at')
            ->first();

        $flashExam = session('flashExam'); // array we pass after store

        return Inertia::render('Medicals/PhysicalExams/Start', [
            'patient'    => self::mapPatientForView($patient),
            'latestExam' => self::mapExamForView($latestExam),
            'flashExam'  => $flashExam ?: null,
        ]);
    }

    /**
     * POST /medicals/patients/{patient}/exams
     * Create a physical exam and redirect back to start with flash exam.
     */
    public function store(Request $request, int $patientId)
    {
        /** @var Patient $patient */
        $patient = Patient::findOrFail($patientId);

        $data = $request->validate([
            'weight'        => ['required', 'numeric'],      // kg
            'height'        => ['required'],                 // string/number input; we'll normalize to meters
            'bp_sys'        => ['required', 'numeric'],
            'bp_dia'        => ['required', 'numeric'],
            'left_vision'   => ['nullable', 'numeric'],
            'right_vision'  => ['nullable', 'numeric'],
            'temp'          => ['nullable', 'numeric'],
            'pulse'         => ['nullable', 'numeric'],
            'blood_sugar'   => ['nullable', 'string'],
        ]);

        // Normalize height to meters
        $heightMeters = self::parseHeightToMeters((string) $data['height']);
        $weightKg     = isset($data['weight']) ? (float) $data['weight'] : null;

        // Calculate BMI info
        $bmi       = self::calculateBmi($weightKg, $heightMeters);
        $bmiStatus = self::bmiStatus($bmi);

        // BP status
        $bpSys = (float) $data['bp_sys'];
        $bpDia = (float) $data['bp_dia'];
        $bpStatus = self::calculateBloodPressureStatus($bpSys, $bpDia);

        // Timezone: Africa/Harare
        $firstBpTime = Carbon::now('Africa/Harare');

        /** @var PhysicalExam $exam */
        $exam = null;

        DB::connection('medicals')->transaction(function () use (
            &$exam,
            $patient,
            $data,
            $heightMeters,
            $weightKg,
            $bmi,
            $bmiStatus,
            $bpStatus,
            $firstBpTime
        ) {
            $exam = PhysicalExam::create([
                'patient_id'     => $patient->id,
                'height'         => $heightMeters,          // meters
                'weight'         => $weightKg,              // kg
                'bmi'            => $bmi,
                'bmi_status'     => $bmiStatus,
                'bp_sys'         => $data['bp_sys'],
                'bp_dia'         => $data['bp_dia'],
                'bp_status'      => $bpStatus,
                'left_vision'    => $data['left_vision']   ?? null,
                'right_vision'   => $data['right_vision']  ?? null,
                'temp'           => $data['temp']          ?? null,
                'pulse'          => $data['pulse']         ?? null,
                'blood_sugar'    => $data['blood_sugar']   ?? null,
                'first_bp_time'  => $firstBpTime,
            ]);
        });

        // Pass just-saved exam back so sidebar updates immediately
        $flashExam = self::mapExamForView($exam);

        return redirect()
            ->route('medicals.exams.start', $patient->id)
            ->with('flashExam', $flashExam);
    }

    /**
     * (Optional) GET /medicals/patients/{patient}/exams/latest
     * JSON endpoint if you still want to fetch latest via XHR.
     */
    public function latest(int $patientId)
    {
        $patient = Patient::findOrFail($patientId);

        $latestExam = PhysicalExam::query()
            ->where('patient_id', $patient->id)
            ->orderByDesc('created_at')
            ->first();

        return response()->json(self::mapExamForView($latestExam));
    }

    /**
     * (Optional) PATCH /medicals/physical-exams/{exam}/second-bp
     * Save repeat BP reading and update status/time.
     */
    public function takeSecondBPReading(Request $request, int $examId)
    {
        $data = $request->validate([
            'bp_repeat_sys' => ['required', 'numeric'],
            'bp_repeat_dia' => ['required', 'numeric'],
        ]);

        /** @var PhysicalExam $exam */
        $exam = PhysicalExam::findOrFail($examId);

        $sys = (float) $data['bp_repeat_sys'];
        $dia = (float) $data['bp_repeat_dia'];
        $status = self::calculateBloodPressureStatus($sys, $dia);

        $exam->bp_repeat_sys = $sys;
        $exam->bp_repeat_dia = $dia;
        $exam->bp_status     = $status;
        $exam->last_bp_time  = Carbon::now('Africa/Harare');
        $exam->save();

        return response()->json(self::mapExamForView($exam));
    }

    /**
     * (Optional) PATCH /medicals/physical-exams/{exam}/referral
     */
    public function addReferralComment(Request $request, int $examId)
    {
        $data = $request->validate([
            'referral_comment' => ['nullable', 'string'],
        ]);

        $exam = PhysicalExam::findOrFail($examId);
        $exam->referral_comment = $data['referral_comment'] ?? null;
        $exam->save();

        return response()->json(self::mapExamForView($exam));
    }
}
