<?php
namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Certificate;
use App\Models\Medicals\Company;
use App\Models\Medicals\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ReportsController extends Controller
{
    /**
     * GET /medicals/reports/generated
     * Query params:
     *  - year (int)
     *  - month (1..12)
     *  - company (company_id)
     *  - category (string)
     *  - date_from (YYYY-MM-DD)
     *  - date_to   (YYYY-MM-DD)
     */
    public function generated(Request $request)
    {
        $year     = $request->integer('year');
        $month    = $request->integer('month');
        $company  = $request->input('company');
        $category = (string) $request->string('category');
        $dateFrom = $request->input('date_from'); // YYYY-MM-DD
        $dateTo   = $request->input('date_to');   // YYYY-MM-DD

        // Build certificate date window (priority to date range, else year/month)
        $certStart = null;
        $certEnd   = null;

        if ($dateFrom || $dateTo) {
            try {
                $certStart = $dateFrom ? Carbon::parse($dateFrom)->startOfDay() : null;
                $certEnd   = $dateTo ? Carbon::parse($dateTo)->endOfDay() : null;
            } catch (\Throwable $e) {
                $certStart = $certStart ?: null;
                $certEnd   = $certEnd ?: null;
            }
        } elseif ($year) {
            $m         = $month ?: 1;
            $certStart = Carbon::createFromDate($year, $m, 1)->startOfMonth();
            $certEnd   = Carbon::createFromDate($year, $m, 1)->endOfMonth();
        }

        $query = Patient::query()
            ->with([
                'attendee',
                'company',
                'latestCertificate', // latest certificate for each patient
            ]);

        // Filter: company
        if (! empty($company)) {
            $query->where('company_id', $company);
        }

        // Filter: category (on patients.category)
        if ($category !== '') {
            $query->where('category', $category);
        }

        // Filter: latest certificate by created_at window (when provided)
        if ($certStart || $certEnd) {
            $query->whereHas('latestCertificate', function ($q) use ($certStart, $certEnd) {
                if ($certStart) {
                    $q->where('created_at', '>=', $certStart);
                }

                if ($certEnd) {
                    $q->where('created_at', '<=', $certEnd);
                }

            });
        }

        // If no explicit date filter, but year/month provided (handled above),
        // we still filtered via $certStart/$certEnd.

        // Prepare lightweight payload for front-end table/export
        $patients = $query->orderByDesc('id')->get()->map(function (Patient $p) {
            $att  = $p->attendee;
            $comp = $p->company;
            $cert = $p->latestCertificate;

            return [
                'id'                 => $p->id,
                'category'           => $p->category,
                'created_at'         => $cert?->created_at ?? $p->created_at, // medical date for report
                'certificate_status' => $cert?->status,
                'attendee'           => [
                    'employee_number' => $att?->employee_number,
                    'first_name'      => $att?->first_name,
                    'last_name'       => $att?->last_name,
                    'date_of_birth'   => $att?->date_of_birth,
                    'gender'          => $att?->gender,
                    'phone_number'    => $att?->phone_number,
                    'national_id'     => $att?->national_id,
                    'company'         => [
                        'id'           => $comp?->id,
                        'company_name' => $comp?->company_name,
                    ],
                ],
            ];
        });

        // Companies for the filter select
        $companies = Company::query()
            ->orderBy('company_name')
            ->get(['id', 'company_name'])
            ->map(fn($c) => ['id' => $c->id, 'company_name' => $c->company_name]);

        return Inertia::render('Medicals/Reports/Generated', [
            'patients'  => $patients,
            'companies' => $companies,
            'filters'   => [
                'year'      => $year,
                'month'     => $month,
                'company'   => $company,
                'category'  => $category,
                'date_from' => $dateFrom,
                'date_to'   => $dateTo,
            ],
        ]);
    }

    public function index(Request $request)
    {
        [$from, $to] = $this->parseDateRange($request);

        // Heatmap-friendly activity array (keep ascending for calendar)
        $activity = $this->buildActivityByDay($from, $to);

                                                                  // Rich report rows grouped by day with category buckets (latest first)
        $reportsAsc = $this->buildPatientReportByDay($from, $to); // returns ASC by day
        $reports    = collect($reportsAsc)
            ->sortByDesc('day') // <-- sort latest day first
            ->values()
            ->all();

        return Inertia::render('Medicals/Reports/Index', [
            'activityByDay' => $activity,
            'reportByDay'   => $reports, // now DESC
            'filters'       => [
                'from' => $from->toDateString(),
                'to'   => $to->toDateString(),
            ],
        ]);
    }

    /**
     * Optional JSON endpoint: /medicals/reports/activity-by-day
     */
    public function activityByDay(Request $request)
    {
        [$from, $to] = $this->parseDateRange($request);
        return response()->json($this->buildActivityByDay($from, $to));
    }

    /**
     * Optional JSON endpoint: /medicals/reports/activity-by-day/report
     */
    public function patientReportByDay(Request $request)
    {
        [$from, $to] = $this->parseDateRange($request);
        return response()->json($this->buildPatientReportByDay($from, $to));
    }

    // ─────────────────────────────────────────────────────────────
    // Helpers
    // ─────────────────────────────────────────────────────────────

    /**
     * Parse optional ?from=YYYY-MM-DD&to=YYYY-MM-DD, default = current year
     * Returns [Carbon $from, Carbon $to]
     */
    private function parseDateRange(Request $request): array
    {
        $year = (int) ($request->query('year') ?: now()->year);

        $from = $request->query('from')
            ? Carbon::parse($request->query('from'))->startOfDay()
            : Carbon::create($year, 1, 1, 0, 0, 0);
        $to = $request->query('to')
            ? Carbon::parse($request->query('to'))->endOfDay()
            : Carbon::create($year, 12, 31, 23, 59, 59);

        if ($from->greaterThan($to)) {
            [$from, $to] = [$to->copy()->startOfDay(), $from->copy()->endOfDay()];
        }

        return [$from, $to];
    }

    /**
     * Build heatmap data: [{ day: 'YYYY-MM-DD', activity: int }, ...]
     */
    private function buildActivityByDay(Carbon $from, Carbon $to): array
    {
        return DB::connection('medicals')
            ->table('certificates')
            ->selectRaw('DATE(created_at) AS day, COUNT(*) AS activity')
        // compare by date range (inclusive) – clamp to dates to avoid TZ edge cases
            ->whereBetween(DB::raw('DATE(created_at)'), [
                $from->toDateString(),
                $to->toDateString(),
            ])
            ->groupByRaw('DATE(created_at)')
            ->orderBy('day', 'asc')
            ->get()
            ->map(fn($r) => [
                'day'      => (string) $r->day,   // 'YYYY-MM-DD'
                'activity' => (int) $r->activity, // count
            ])
            ->all();
    }

    /**
     * Build report-by-day with category buckets and totals.
     *
     * Returns an array like:
     * [
     *   {
     *     day: 'YYYY-MM-DD',
     *     certificates_count: n,
     *     patients_count: n, // alias
     *     'Pneumoconiosis': [{ ...patient summary... }, ...],
     *     'Food Handler (COH)': [...],
     *     ...
     *   },
     *   ...
     * ]
     */
    private function buildPatientReportByDay(Carbon $from, Carbon $to): array
    {
        $certificates = Certificate::with([
            'patient.attendee:id,first_name,last_name,date_of_birth,gender,national_id,phone_number',
            'patient.company:id,company_name',
        ])
            ->select('id', 'patient_id', 'created_at')
            ->whereBetween('created_at', [$from, $to])
            ->orderBy('created_at', 'asc')
            ->get();

        $result = [];

        foreach ($certificates as $certificate) {
            $entryDay = $certificate->created_at->format('Y-m-d');
            $patient  = $certificate->patient;
            $category = $patient?->category ?? 'Unknown';

            if (! isset($result[$entryDay])) {
                $result[$entryDay] = [
                    'day'                => $entryDay,
                    'certificates_count' => 0,
                    'patients_count'     => 0, // alias for your frontend
                ];
            }

            $result[$entryDay]['certificates_count']++;
            $result[$entryDay]['patients_count']++;

            if (! isset($result[$entryDay][$category])) {
                $result[$entryDay][$category] = [];
            }

            $att  = $patient?->attendee;
            $comp = $patient?->company;

            $result[$entryDay][$category][] = [
                'exam_purpose'  => $patient?->exam_purpose,
                'first_name'    => $att?->first_name ?? '',
                'last_name'     => $att?->last_name ?? '',
                'date_of_birth' => $att?->date_of_birth ?? '',
                'gender'        => $att?->gender ?? '',
                'national_id'   => $att?->national_id ?? '',
                'phone_number'  => $att?->phone_number ?? '',
                'company'       => $comp?->company_name ?? '',
                'category'      => $category,
                'patient_id'    => $patient?->id,
            ];
        }

        // return chronologically sorted array
        $rows = array_values($result);
        usort($rows, fn($a, $b) => strcmp($a['day'], $b['day']));
        return $rows;
    }

    public function day(Request $request, string $day)
    {
        // Validate/normalize the day param
        try {
            $date = Carbon::createFromFormat('Y-m-d', $day)->startOfDay();
        } catch (\Throwable $e) {
            abort(404, 'Invalid date.');
        }
        $from = $date->copy()->startOfDay();
        $to   = $date->copy()->endOfDay();

        // Pull certificates for that exact day with only the fields we need
        $certificates = Certificate::with([
            'patient.attendee:id,first_name,last_name,date_of_birth,gender,national_id,phone_number',
            'patient.company:id,company_name',
        ])
            ->select('id', 'patient_id', 'created_at')
            ->whereBetween('created_at', [$from, $to])
            ->orderBy('created_at', 'asc')
            ->get();

        // Build the single-day report structure
        $report = [
            'day'                => $from->toDateString(),
            'certificates_count' => 0,
            'patients_count'     => 0, // alias for convenience
        ];

        // Buckets by category
        foreach ($certificates as $certificate) {
            $patient  = $certificate->patient;
            $category = $patient?->category ?? 'Unknown';

            $report['certificates_count']++;
            $report['patients_count']++;

            if (! isset($report[$category])) {
                $report[$category] = [];
            }

            $att  = $patient?->attendee;
            $comp = $patient?->company;

            $report[$category][] = [
                'exam_purpose'  => $patient?->exam_purpose,
                'first_name'    => $att?->first_name ?? '',
                'last_name'     => $att?->last_name ?? '',
                'date_of_birth' => $att?->date_of_birth ?? '',
                'gender'        => $att?->gender ?? '',
                'national_id'   => $att?->national_id ?? '',
                'phone_number'  => $att?->phone_number ?? '',
                'company'       => $comp?->company_name ?? '',
                'category'      => $category,
                'patient_id'    => $patient?->id,
            ];
        }

        // Companies present that day (for the company filter in the UI)
        $companies = collect($report)
            ->only(['Pneumoconiosis', 'Food Handler (COH)', 'Pre-Employement', 'Exit-Employement', 'Exit-Pneumoconiosis', 'Unknown'])
            ->flatten(1)
            ->pluck('company')
            ->filter(fn($v) => $v !== null && $v !== '')
            ->unique()
            ->sort()
            ->values()
            ->all();

        return Inertia::render('Medicals/Reports/Day', [
            'day'       => $from->toDateString(),
            'report'    => $report,
            'companies' => $companies,
        ]);
    }

    public function corrections(Request $request)
    {
        $now = Carbon::now();

        // Certificates with created_at in the future
        $certs = Certificate::with([
            'patient.attendee:id,first_name,last_name',
            'patient.company:id,company_name',
        ])
            ->where('created_at', '>', $now)
            ->orderBy('created_at', 'asc')
            ->limit(1000)
            ->get()
            ->map(function (Certificate $c) {
                $p    = $c->patient;
                $att  = $p?->attendee;
                $comp = $p?->company;
                return [
                    'id'           => $c->id,
                    'entity'       => 'certificate',
                    'created_at'   => $c->created_at?->toDateTimeString(),
                    'patient_id'   => $p?->id,
                    'first_name'   => $att?->first_name,
                    'last_name'    => $att?->last_name,
                    'company_name' => $comp?->company_name,
                    'category'     => $p?->category,
                    'status'       => $c->status,
                ];
            });

        // Patients with created_at in the future
        $patients = Patient::with(['attendee:id,first_name,last_name', 'company:id,company_name'])
            ->where('created_at', '>', $now)
            ->orderBy('created_at', 'asc')
            ->limit(1000)
            ->get()
            ->map(function (Patient $p) {
                $att  = $p->attendee;
                $comp = $p->company;
                return [
                    'id'           => $p->id,
                    'entity'       => 'patient',
                    'created_at'   => $p->created_at?->toDateTimeString(),
                    'patient_id'   => $p->id,
                    'first_name'   => $att?->first_name,
                    'last_name'    => $att?->last_name,
                    'company_name' => $comp?->company_name,
                    'category'     => $p->category,
                    'status'       => null,
                ];
            });

        return Inertia::render('Medicals/Reports/Corrections', [
            'now'          => $now->toDateTimeString(),
            'certificates' => $certs,
            'patients'     => $patients,
        ]);
    }

    /**
     * PATCH /medicals/reports/corrections/fix
     * body: { entity: 'certificate'|'patient', id: int, new_created_at: 'YYYY-MM-DDTHH:mm' or 'YYYY-MM-DD HH:mm:ss' }
     */
    public function fixCreatedAt(Request $request)
    {
        $validated = $request->validate([
            'entity'         => 'required|in:certificate,patient',
            'id'             => 'required|integer',
            'new_created_at' => 'required|date',
        ]);

        $entity = $validated['entity'];
        $id     = (int) $validated['id'];
        $new    = Carbon::parse($validated['new_created_at']);

        // Do not allow future dates
        if ($new->greaterThan(now())) {
            return back()->with('error', 'New date/time cannot be in the future.');
        }

        DB::connection('medicals')->transaction(function () use ($entity, $id, $new) {
            if ($entity === 'certificate') {
                $model = Certificate::findOrFail($id);
            } else {
                $model = Patient::findOrFail($id);
            }
            $model->created_at = $new;
            // If you also keep updated_at consistent, update it if it is earlier than created_at
            if ($model->updated_at && $model->updated_at->lessThan($new)) {
                $model->updated_at = $new;
            }
            $model->save();
        });

        return back()->with('success', 'Timestamp updated.');
    }

}
