<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\SkinCondition;
use App\Models\Medicals\Patient;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Validation\Rule;

class SkinConditionController extends Controller
{
    public function index(Request $request)
    {
        $search   = (string) $request->string('search');
        $perPage  = (int) ($request->integer('per_page') ?: 20);
        $sort     = $request->input('sort', 'name');
        $dir      = strtolower($request->input('direction', 'asc')) === 'desc' ? 'desc' : 'asc';

        $query = SkinCondition::query();
        if ($search !== '') {
            $query->where('name', 'like', "%{$search}%");
        }

        if (! in_array($sort, ['id','name','created_at','updated_at'], true)) {
            $sort = 'name';
        }

        $conditions = $query->orderBy($sort, $dir)->paginate($perPage)->withQueryString();

        return Inertia::render('Medicals/SkinConditions/Index', [
            'conditions' => $conditions,
            'filters'    => [
                'search'    => $search,
                'per_page'  => $perPage,
                'sort'      => $sort,
                'direction' => $dir,
            ],
        ]);
    }

    public function create()
    {
        return Inertia::render('Medicals/SkinConditions/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:255', Rule::unique('skin_conditions','name')],
        ]);

        $cond = SkinCondition::create($data);

        return redirect()
            ->route('medicals.skin-conditions.edit', $cond->id)
            ->with('success', 'Skin condition created.');
    }

    public function edit(SkinCondition $skinCondition)
    {
        return Inertia::render('Medicals/SkinConditions/Edit', [
            'condition' => [
                'id'   => $skinCondition->id,
                'name' => $skinCondition->name,
                'created_at' => $skinCondition->created_at,
                'updated_at' => $skinCondition->updated_at,
            ],
        ]);
    }

    public function update(Request $request, SkinCondition $skinCondition)
    {
        $data = $request->validate([
            'name' => ['required','string','max:255', Rule::unique('skin_conditions','name')->ignore($skinCondition->id)],
        ]);

        $skinCondition->update($data);

        return redirect()
            ->route('medicals.skin-conditions.edit', $skinCondition->id)
            ->with('success', 'Skin condition updated.');
    }

    public function destroy(SkinCondition $skinCondition)
    {
        // detach all patients first (safe delete)
        $skinCondition->patients()->detach();
        $skinCondition->delete();

        return redirect()
            ->route('medicals.skin-conditions.index')
            ->with('success', 'Skin condition deleted.');
    }

    /**
     * Attach/update a condition on a specific patient (pivot)
     * Like Disease::updatePatientDisease — same idea.
     */
    public function updatePatientSkinCondition(
        Request $request,
        SkinCondition $skinCondition,
        Patient $patient
    ) {
        $data = $request->validate([
            'has_skin_condition' => ['required','boolean'],
            'date'               => ['nullable','string'],
        ]);

        $patient->skin_conditions()->syncWithoutDetaching([
            $skinCondition->id => [
                'has_skin_condition' => $data['has_skin_condition'],
                'date'               => $data['date'] ?? null,
                'updated_at'         => now(),
                'created_at'         => now(),
            ],
        ]);

        // Return the patient's TRUE conditions list (like your Disease/Illness style)
        $attached = $patient->skin_conditions()
            ->wherePivot('has_skin_condition', true)
            ->withPivot(['date'])
            ->get()
            ->map(function ($c) {
                return [
                    'name'               => $c->name,
                    'has_skin_condition' => (bool)$c->pivot->has_skin_condition,
                    'date'               => $c->pivot->date,
                ];
            });

        return response()->json([
            'message' => 'Skin condition updated successfully.',
            'conditions' => $attached,
        ], 200);
    }
}
