<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\Company;
use App\Models\Medicals\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;

class SwabController extends Controller
{
    /**
     * GET /medicals/swabs
     * Renders the Inertia page with filter defaults and companies.
     */
    public function index(Request $request)
    {
        $start = $request->query('start_date')
            ? Carbon::parse($request->query('start_date'))->toDateString()
            : Carbon::now()->startOfMonth()->toDateString();

        $end = $request->query('end_date')
            ? Carbon::parse($request->query('end_date'))->toDateString()
            : Carbon::now()->toDateString();

        $company = $request->query('company');

        $companies = Company::query()
            ->orderBy('company_name')
            ->get(['id', 'company_name'])
            ->map(fn ($c) => ['id' => $c->id, 'company_name' => $c->company_name])
            ->values();

        return Inertia::render('Medicals/Swabs/Index', [
            'companies' => $companies,
            'filters'   => [
                'start_date' => $start,
                'end_date'   => $end,
                'company'    => $company,
            ],
        ]);
    }

    /**
     * POST /medicals/swabs/patients-by-status
     * Body: start_date, end_date, company (optional)
     * Returns a plain JSON array of patients (no Resource, no wrapper).
     */
    public function getPatientsBySwabStatus(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate   = $request->input('end_date');
        $companyId = $request->input('company');

        $start = $startDate ? Carbon::parse($startDate)->startOfDay() : null;
        $end   = $endDate   ? Carbon::parse($endDate)->endOfDay()   : null;

        $patients = Patient::query()
            ->where('category', 'Food Handler (COH)')
            ->when($companyId, fn ($q) => $q->where('company_id', $companyId))
            ->when($start && $end, function ($q) use ($start, $end) {
                $q->whereHas('certificates', function ($cq) use ($start, $end) {
                    $cq->whereBetween('certificates.created_at', [$start, $end]);
                });
            })
            ->when($start && ! $end, function ($q) use ($start) {
                $q->whereHas('certificates', function ($cq) use ($start) {
                    $cq->where('certificates.created_at', '>=', $start);
                });
            })
            ->with([
                'attendee.company' => fn ($q) => $q->select('companies.id', 'companies.company_name'),
                'company'          => fn ($q) => $q->select('companies.id', 'companies.company_name'),
                'certificates'     => fn ($q) => $q->select(
                    'certificates.id',
                    'certificates.patient_id',
                    'certificates.status',
                    'certificates.created_at'
                ),
                'latestCertificate'=> fn ($q) => $q->select(
                    'certificates.id',
                    'certificates.patient_id',
                    'certificates.status',
                    'certificates.certificate_location',
                    'certificates.created_at'
                ),
                'swabs'            => fn ($q) => $q->select(
                    'swabs.id',
                    'swabs.patient_id',
                    'swabs.status',
                    'swabs.file',
                    'swabs.comment',
                    'swabs.created_at',
                    'swabs.updated_at'
                ),
            ])
            ->orderByDesc('id')
            ->get();

        $data = $patients->map(function ($p) {
            $att  = $p->attendee;
            $comp = $p->company;
            $lc   = $p->latestCertificate;

            return [
                'id'           => $p->id,
                'category'     => $p->category,
                'exam_purpose' => $p->exam_purpose,
                'created_at'   => optional($p->created_at)->toDateTimeString(),

                'attendee' => [
                    'id'            => $att?->id,
                    'first_name'    => $att?->first_name,
                    'last_name'     => $att?->last_name,
                    'date_of_birth' => $att?->date_of_birth,
                    'gender'        => $att?->gender,
                    'national_id'   => $att?->national_id,
                    'phone_number'  => $att?->phone_number,
                    'time_of_entry' => $att?->time_of_entry
                        ? \Carbon\Carbon::parse($att->time_of_entry)->timezone('Africa/Harare')->toDateTimeString()
                        : null,
                    'company'       => [
                        'id'           => $att?->company?->id,
                        'company_name' => $att?->company?->company_name,
                    ],
                ],

                'company' => [
                    'id'           => $comp?->id,
                    'company_name' => $comp?->company_name,
                ],

                'latest_certificate' => $lc ? [
                    'id'                   => $lc->id,
                    'patient_id'           => $lc->patient_id,
                    'status'               => $lc->status,
                    'certificate_location' => $lc->certificate_location,
                    'created_at'           => optional($lc->created_at)->toDateTimeString(),
                ] : null,

                'certificates_count' => $p->certificates?->count() ?? 0,

                'swabs' => ($p->swabs ?? collect())->map(function ($s) {
                    return [
                        'id'         => $s->id,
                        'patient_id' => $s->patient_id,
                        'status'     => $s->status,
                        'file'       => $s->file,
                        'comment'    => $s->comment,
                        'created_at' => optional($s->created_at)->toDateTimeString(),
                        'updated_at' => optional($s->updated_at)->toDateTimeString(),
                    ];
                })->values(),
            ];
        })->values();

        // Return the array directly (no {data: …} wrapper)
        return response()->json($data);
    }
}
