<?php

namespace App\Http\Controllers\Medicals;

use App\Http\Controllers\Controller;
use App\Models\Medicals\TobaccoUse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class TobaccoUseController extends Controller
{
    public function index(Request $request)
    {
        $search    = (string) $request->string('search');
        $perPage   = (int) ($request->integer('per_page') ?: 20);
        $sort      = $request->input('sort', 'name'); // name|id|created_at|updated_at
        $direction = strtolower($request->input('direction', 'asc')) === 'desc' ? 'desc' : 'asc';

        $query = TobaccoUse::query();

        if ($search !== '') {
            $query->where('name', 'like', "%{$search}%");
        }

        if (! in_array($sort, ['name', 'id', 'created_at', 'updated_at'], true)) {
            $sort = 'name';
        }

        $query->orderBy($sort, $direction);

        $paginator = $query->paginate($perPage)->withQueryString();

        return Inertia::render('Medicals/TobaccoUses/Index', [
            'conditions' => $paginator,
            'filters'    => [
                'search'    => $search,
                'per_page'  => $perPage,
                'sort'      => $sort,
                'direction' => $direction,
            ],
        ]);
    }

    public function create()
    {
        return Inertia::render('Medicals/TobaccoUses/Create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:255', Rule::unique('tobacco_uses','name')],
        ]);

        TobaccoUse::create($data);

        return redirect()
            ->route('medicals.tobacco-uses.index')
            ->with('success', 'Tobacco use created.');
    }

    public function edit(TobaccoUse $tobacco_use)
    {
        return Inertia::render('Medicals/TobaccoUses/Edit', [
            'condition' => $tobacco_use,
        ]);
    }

    public function update(Request $request, TobaccoUse $tobacco_use)
    {
        $data = $request->validate([
            'name' => [
                'required','string','max:255',
                Rule::unique('tobacco_uses','name')->ignore($tobacco_use->id),
            ],
        ]);

        $tobacco_use->update($data);

        return redirect()
            ->route('medicals.tobacco-uses.index')
            ->with('success', 'Tobacco use updated.');
    }

    public function destroy(TobaccoUse $tobacco_use)
    {
        $tobacco_use->delete();

        return redirect()
            ->back()
            ->with('success', 'Tobacco use deleted.');
    }
}
