<?php
namespace App\Http\Controllers;

use App\Models\Medication;
use Illuminate\Http\Request;
use Inertia\Inertia;

class MedicationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $medications = Medication::query()
            ->when(request('search'), function ($query, $search) {
                $query->where('name', 'like', "%{$search}%");
            })
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('Medications/Index', [
            'medications' => $medications,
            'filters'     => request()->only(['search']),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Medications/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'         => 'required|string|max:255',
            'dosage'       => 'nullable|string|max:100',
            'form'         => 'nullable|string|max:100',
            'unit'         => 'nullable|string|max:100',
            'instructions' => 'nullable|string',
        ]);

        $medication = Medication::create($validated);

        // Check if redirect back to a central store form is requested
        if ($request->has('redirect_to_store') && $request->filled('central_store_id')) {
            return redirect()
                ->route('central-stores.receive-batch.form', $request->input('central_store_id'))
                ->with('success', 'Medication created successfully.');
        }

        // Default redirect
        return redirect()
            ->route('medications.index')
            ->with('success', 'Medication created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Medication $medication)
    {
        return Inertia::render('Medications/Show', [
            'medication' => $medication,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Medication $medication)
    {
        return Inertia::render('Medications/Edit', [
            'medication' => $medication,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Medication $medication)
    {
        $request->validate([
            'name'         => 'required|string|max:255',
            'dosage'       => 'nullable|string|max:100',
            'form'         => 'nullable|string|max:100',
            'unit'         => 'nullable|string|max:100',
            'instructions' => 'nullable|string',
        ]);

        $medication->update($request->all());

        return redirect()->route('medications.index')
            ->with('success', 'Medication updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Medication $medication)
    {
        $medication->delete();

        return redirect()->route('medications.index')
            ->with('success', 'Medication deleted successfully.');
    }
}
