<?php
namespace App\Http\Controllers;

use App\Models\Clinic;
use App\Models\Medication;
use App\Models\MedicationBatch;
use App\Models\MedicationRequest;
use Illuminate\Http\Request;
use Inertia\Inertia;

class MedicationRequestController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Clinic $clinic, Request $request)
    {
        // Eager load the medication relationship for each medication request
        $requests = $clinic->medicationRequests()->with('medication')->paginate(10);

        // Transform the requests to include detailed medication information
        $requests->getCollection()->transform(function ($request) {
            return [
                'id'           => $request->id,
                'clinic_id'    => $request->clinic_id,
                'medication'   => [
                    'id'           => $request->medication->id,
                    'name'         => $request->medication->name,
                    'dosage'       => $request->medication->dosage,
                    'form'         => $request->medication->form,
                    'unit'         => $request->medication->unit,
                    'instructions' => $request->medication->instructions,
                ],
                'quantity'     => $request->quantity,
                'request_date' => $request->request_date,
                'status'       => $request->status,
            ];
        });

        // Return an Inertia response
        return Inertia::render('Clinics/MedicationRequests/Index', [
            'clinic'   => [
                'id'             => $clinic->id,
                'name'           => $clinic->name,
                'city'           => $clinic->city,
                'manager'        => $clinic->manager,
                'contact_person' => $clinic->contact_person,
                'email_address'  => $clinic->email_address,
            ],
            'requests' => $requests,
            'filters'  => [
                'search' => $request->input('search'),
            ],
        ]);
    }

    public function indexAll(Request $request)
    {
        $requests = MedicationRequest::with('medication', 'clinic')->paginate(10);

        // Get all clinics
        $clinics = Clinic::orderBy('name')->get();
        $batches = MedicationBatch::with(['medication:id,name', 'supplier:id,name'])->get();

        return Inertia::render('Clinics/MedicationRequests/RequestAll', [
            'requests' => $requests,
            'batches'  => $batches,
            'clinics'  => $clinics,
            'filters'  => [
                'search' => $request->input('search'),
            ],
        ]);
    }

    public function indexJson(Clinic $clinic)
    {
        // Eager load the medication relationship for each medication request
        $requests = $clinic->medicationRequests()->with('medication')->paginate(10);

        // Transform the requests to include detailed medication information
        $requests->getCollection()->transform(function ($request) {
            return [
                'id'           => $request->id,
                'clinic_id'    => $request->clinic_id,
                'medication'   => [
                    'id'           => $request->medication->id,
                    'name'         => $request->medication->name,
                    'dosage'       => $request->medication->dosage,
                    'form'         => $request->medication->form,
                    'unit'         => $request->medication->unit,
                    'instructions' => $request->medication->instructions,
                ],
                'quantity'     => $request->quantity,
                'request_date' => $request->request_date,
                'status'       => $request->status,
            ];
        });

        return response()->json([
            'clinic'   => $clinic->only(['id', 'name', 'address', 'city', 'afhoz_number', 'contact_person', 'manager', 'country', 'email_address']),
            'requests' => $requests->toArray(),
        ]);
    }
    /**
     * Show the form for creating a new resource.
     */

    public function create(Clinic $clinic)
    {
        $medications = Medication::all();

        // load the requests that belong to this clinic
        $requests = $clinic->medicationRequests()
            ->with('medication') // optional: if you need medication details
            ->latest('request_date')
            ->get();

        return Inertia::render('Clinics/MedicationRequests/RequestMedication', [
            'clinic'      => $clinic,
            'medications' => $medications,
            'requests'    => $requests, // ← pass them to the page
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, Clinic $clinic)
    {
        $validated = $request->validate([
            'medication_id' => 'required|exists:medications,id',
            'quantity'      => 'required|integer|min:1',
        ]);

        MedicationRequest::create([
            'clinic_id'     => $clinic->id,
            'medication_id' => $validated['medication_id'],
            'quantity'      => $validated['quantity'],
            'request_date'  => now(),
            'status'        => 'pending',
        ]);

        return redirect()->route('clinics.request-medication.create', $clinic->id)->with('success', 'Medication request submitted successfully.');
    }
    /**
     * Display the specified resource.
     */
    public function show(MedicationRequest $medicationRequest)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MedicationRequest $medicationRequest)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MedicationRequest $medicationRequest)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:1',
            'status'   => 'required|string',
        ]);

        $medicationRequest->update([
            'quantity' => $validated['quantity'],
            'status'   => $validated['status'],
        ]);

        return redirect()->back()->with('success', 'Medication request updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MedicationRequest $medicationRequest)
    {
        $medicationRequest->delete();

        return redirect()->back()->with('success', 'Medication request deleted successfully.');
    }
}
