<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class MonthlyAilmentsController extends Controller
{
    /**
     * Excel: Diagnosis Summary for a month.
     * - Sheet 1: Monthly Totals (Diagnosis | Total | Nurse1 | Nurse2 | …)
     * - Sheet 2: Daily Summary  (Date | Diagnosis | Total | Nurse1 | Nurse2 | …)
     * Scope: clinics the authenticated user can access (unless superadmin).
     */
    public function monthly(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        // Allowed roles incl. receptionist
        $role = (string)($user->role?->value ?? $user->role ?? '');
        $allowed = ['receptionist','nurse','doctor','admin','superadmin'];
        abort_if(!in_array($role, $allowed, true), 403, 'You are not allowed to access this report.');

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month || $month < 1 || $month > 12, 422, 'Both valid year and month are required.');

        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end   = (clone $start)->endOfMonth();

        // Clinics this user can access
        $clinicIds = $this->clinicIdsFor($user);
        $seeAll    = in_array($role, ['superadmin'], true);

        // Pull consultations within month; prefer consultation_date, fallback to created_at
        $rows = Consultation::query()
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('consultation_date', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->whereNull('consultation_date')
                         ->whereBetween('created_at', [$start, $end]);
                  });
            })
            ->when(!$seeAll, fn ($q) => $q->whereIn('clinic_id', $clinicIds ?: [-1]))
            ->whereNotNull('diagnosis')
            ->get(['id','doctor_id','clinic_id','diagnosis','consultation_date','created_at']);

        // Resolve doctor (nurse/doctor) names for columns
        $doctorIds = $rows->pluck('doctor_id')->filter()->unique()->values();
        $doctorNames = $doctorIds->isNotEmpty()
            ? User::whereIn('id', $doctorIds)->pluck('name', 'id')
            : collect();
        // Ensure an "Unassigned" bucket
        $UNASSIGNED_KEY = 0;
        $doctorNames->put($UNASSIGNED_KEY, 'Unassigned');

        // Aggregates
        $monthly = [];            // [diagnosis => ['total'=>int, 'by'=>[doctorId=>count]]]
        $daily   = [];            // [Y-m-d => [diagnosis => ['total'=>int, 'by'=>[doctorId=>count]]]]
        $nurseIdSet = [];         // collect all doctor_ids encountered (including unassigned)

        foreach ($rows as $c) {
            $doctorId = $c->doctor_id ? (int)$c->doctor_id : $UNASSIGNED_KEY;
            $dt = $c->consultation_date ?: $c->created_at;
            $dayKey = Carbon::parse($dt)->format('Y-m-d');

            foreach ($this->explodeDiagnosis($c->diagnosis) as $diag) {
                // Monthly
                if (!isset($monthly[$diag])) {
                    $monthly[$diag] = ['total' => 0, 'by' => []];
                }
                $monthly[$diag]['total']++;
                $monthly[$diag]['by'][$doctorId] = ($monthly[$diag]['by'][$doctorId] ?? 0) + 1;

                // Daily
                if (!isset($daily[$dayKey])) $daily[$dayKey] = [];
                if (!isset($daily[$dayKey][$diag])) {
                    $daily[$dayKey][$diag] = ['total' => 0, 'by' => []];
                }
                $daily[$dayKey][$diag]['total']++;
                $daily[$dayKey][$diag]['by'][$doctorId] = ($daily[$dayKey][$diag]['by'][$doctorId] ?? 0) + 1;

                $nurseIdSet[$doctorId] = true;
            }
        }

        // Sort nurses by display name (Unassigned last)
        $nurseIds = array_keys($nurseIdSet);
        usort($nurseIds, function ($a, $b) use ($UNASSIGNED_KEY, $doctorNames) {
            if ($a === $UNASSIGNED_KEY && $b !== $UNASSIGNED_KEY) return 1;
            if ($b === $UNASSIGNED_KEY && $a !== $UNASSIGNED_KEY) return -1;
            return strcasecmp((string)$doctorNames->get($a, ''), (string)$doctorNames->get($b, ''));
        });

        // Sort diagnoses by total desc
        uksort($monthly, function ($a, $b) use ($monthly) {
            $da = $monthly[$a]['total'] ?? 0;
            $db = $monthly[$b]['total'] ?? 0;
            return $db <=> $da ?: strcasecmp($a, $b);
        });

        // Sort daily keys
        ksort($daily); // ASC by date

        $periodLabel  = $start->isoFormat('MMMM YYYY');
        $generatedAt  = now()->format('Y-m-d H:i:s');
        $clinicLabel  = $seeAll ? 'All Clinics' : ($this->clinicNamesList($clinicIds) ?: '—');

        $fileName = sprintf('Diagnosis_Summary_%d_%02d.xlsx', $year, $month);

        return response()->streamDownload(function () use (
            $periodLabel, $generatedAt, $clinicLabel,
            $monthly, $daily, $nurseIds, $doctorNames
        ) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            /* -------- Sheet 1: Monthly Totals -------- */
            $writer->getCurrentSheet()->setName($this->safeSheetName('Monthly Totals'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['DIAGNOSIS SUMMARY — MONTHLY TOTALS'],
                ['Period', $periodLabel],
                ['Clinics', $clinicLabel],
                ['Generated At', $generatedAt],
            ]);

            // Build header: Diagnosis | Total | Nurse1 | Nurse2 | ... | Unassigned
            $header = ['Diagnosis', 'Total'];
            foreach ($nurseIds as $nid) {
                $header[] = (string)($doctorNames->get($nid) ?? 'Unassigned');
            }
            $writer->addRow(Row::fromValues($header, $headerStyle));

            // Column widths
            $opts = $writer->getOptions();
            $opts->setColumnWidth(36, 1);         // Diagnosis
            $opts->setColumnWidth(10, 2);         // Total
            for ($i = 0; $i < count($nurseIds); $i++) {
                $opts->setColumnWidth(18, 3 + $i); // Nurse columns
            }

            foreach ($monthly as $diag => $agg) {
                $row = [$diag, (int)($agg['total'] ?? 0)];
                foreach ($nurseIds as $nid) {
                    $row[] = (int)($agg['by'][$nid] ?? 0);
                }
                $writer->addRow(Row::fromValues($row, $wrapStyle));
            }

            /* -------- Sheet 2: Daily Summary -------- */
            $writer->addNewSheetAndMakeItCurrent();
            $writer->getCurrentSheet()->setName($this->safeSheetName('Daily Summary'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['DIAGNOSIS SUMMARY — DAILY'],
                ['Period', $periodLabel],
                ['Clinics', $clinicLabel],
                ['Generated At', $generatedAt],
            ]);

            // Header: Date | Diagnosis | Total | Nurse1 | ...
            $dHeader = ['Date', 'Diagnosis', 'Total'];
            foreach ($nurseIds as $nid) {
                $dHeader[] = (string)($doctorNames->get($nid) ?? 'Unassigned');
            }
            $writer->addRow(Row::fromValues($dHeader, $headerStyle));

            // Column widths
            $opts = $writer->getOptions();
            $opts->setColumnWidth(14, 1);  // Date
            $opts->setColumnWidth(36, 2);  // Diagnosis
            $opts->setColumnWidth(10, 3);  // Total
            for ($i = 0; $i < count($nurseIds); $i++) {
                $opts->setColumnWidth(18, 4 + $i);
            }

            foreach ($daily as $day => $bag) {
                // Sort diagnoses for this day by total desc
                uksort($bag, function ($a, $b) use ($bag) {
                    $da = $bag[$a]['total'] ?? 0;
                    $db = $bag[$b]['total'] ?? 0;
                    return $db <=> $da ?: strcasecmp($a, $b);
                });

                foreach ($bag as $diag => $agg) {
                    $row = [$day, $diag, (int)($agg['total'] ?? 0)];
                    foreach ($nurseIds as $nid) {
                        $row[] = (int)($agg['by'][$nid] ?? 0);
                    }
                    $writer->addRow(Row::fromValues($row, $wrapStyle));
                }

                // Spacer between dates
                $writer->addRow(Row::fromValues([]));
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /* ---------------- helpers ---------------- */

    /**
     * Get clinic IDs the user can access (primary + pivot).
     */
    private function clinicIdsFor($user): array
    {
        if (method_exists($user, 'accessibleClinicIds')) {
            $ids = $user->accessibleClinicIds() ?: [];
            return array_values(array_unique(array_map('intval', $ids)));
        }

        $ids = [];
        if (method_exists($user, 'accessibleClinics')) {
            $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
        }
        if (!empty($user->clinic_id) && !in_array((int)$user->clinic_id, $ids, true)) {
            $ids[] = (int)$user->clinic_id;
        }
        return array_values(array_unique(array_map('intval', $ids)));
    }

    /** A display list of clinic names for the header (comma-separated). */
    private function clinicNamesList(array $ids): string
    {
        if (empty($ids)) return '';
        $rows = \DB::table('clinics')->whereIn('id', $ids)->orderBy('name')->pluck('name')->all();
        return implode(', ', $rows);
    }

    /** Split/normalize diagnosis text to Title Case tokens. */
    private function explodeDiagnosis(?string $text): array
    {
        $pieces = preg_split('/[,\;\|\n\/]+/u', (string)$text);
        $out = [];
        foreach ($pieces as $p) {
            $p = trim($p);
            if ($p === '' || mb_strlen($p) < 2) continue;
            $out[] = mb_convert_case($p, MB_CASE_TITLE, 'UTF-8');
        }
        return $out;
    }

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /**
     * Simple header block writer.
     * Each $row can be:
     * - ['Huge Title'] or
     * - ['Label', 'Value'] or
     * - null (skipped)
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
