<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth; // ⬅️ add
// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class MonthlyAttendanceReportController extends Controller
{
    /**
     * Attendance report (monthly):
     * - Required: year, month
     * - Optional: clinic_id, search (patient name/employee number)
     * - One sheet per day of the month (only for days with consultations)
     * - Row marks Employee vs Dependent; dependents show Account Holder/Employee #/Relationship/Company
     * - Shows Clinic column
     * - Time column uses created_at in Africa/Harare timezone (same as Cape Town offset).
     * - ⬅️ RESTRICTED to clinics the authenticated user can access.
     */
    public function monthly(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month, 422, 'Year and month are required.');

        // Figure out clinics this user can access
        if (method_exists($user, 'accessibleClinicIds')) {
            $accessibleClinicIds = array_map('intval', (array) $user->accessibleClinicIds());
        } else {
            $ids = [];
            if (method_exists($user, 'accessibleClinics')) {
                $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            }
            if (!empty($user->clinic_id)) {
                $ids[] = (int) $user->clinic_id;
            }
            $accessibleClinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        // If none, return an empty workbook (headers only)
        $hasAccess = !empty($accessibleClinicIds);

        // Optional filters
        $clinicId = $request->query('clinic_id');
        $search   = $request->query('search');

        // If clinic_id provided, ensure it's in the user's access
        $effectiveClinicIds = $hasAccess ? $accessibleClinicIds : [-1];
        if ($clinicId !== null && $clinicId !== '') {
            $clinicId = (int) $clinicId;
            abort_if(!in_array($clinicId, $accessibleClinicIds, true), 403, 'You do not have access to this clinic.');
            $effectiveClinicIds = [$clinicId];
        }

        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end   = (clone $start)->endOfMonth();
        $periodLabel = $start->isoFormat('MMMM YYYY');

        // Build base query (scoped to accessible clinics)
        $query = Consultation::query()
            ->with([
                'patient:id,first_name,surname,gender,employee_number,parent_patient_id,relationship,company_id',
                'patient.company:id,name',
                'clinic:id,name,city',
            ])
            // In-month filter: prefer consultation_date, fallback to created_at if consultation_date is null
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('consultation_date', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->whereNull('consultation_date')
                         ->whereBetween('created_at', [$start, $end]);
                  });
            })
            ->whereIn('clinic_id', $effectiveClinicIds) // ⬅️ scope to user's clinics
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->orWhereHas('patient', function ($p) use ($search) {
                        $p->where('first_name', 'like', "%{$search}%")
                          ->orWhere('surname', 'like', "%{$search}%")
                          ->orWhere('employee_number', 'like', "%{$search}%");
                    });
                });
            })
            ->orderBy('consultation_date')
            ->orderBy('created_at');

        $rows = $query->get();

        // Preload account holders for dependents (include company)
        $parentIds = $rows->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentsById = $parentIds->isNotEmpty()
            ? Patient::whereIn('id', $parentIds)
                ->with(['company:id,name'])
                ->get(['id', 'first_name', 'surname', 'employee_number', 'company_id'])
                ->keyBy('id')
            : collect();

        // Group by day label (YYYY-MM-DD)
        $byDay = $rows->groupBy(function ($c) {
            /** @var \App\Models\Consultation $c */
            $dt = $c->consultation_date ?: $c->created_at;
            return Carbon::parse($dt)->format('Y-m-d');
        })->sortKeys();

        // Header filters line
        $generatedAt = now()->format('Y-m-d H:i:s');
        $filters = ["Period: {$periodLabel}"];
        if (isset($clinicId)) {
            $filters[] = "Clinic ID: {$clinicId}";
        } else {
            // Note that all data is already constrained to user's accessible clinics
            $filters[] = 'Clinics: (your accessible clinics)';
        }
        if ($search)   $filters[] = "Search: {$search}";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Attendance_%d_%02d.xlsx', $year, $month);

        // timezone to use for the Time column
        $tz = 'Africa/Harare'; // same offset as Cape Town (no DST)

        return response()->streamDownload(function () use ($byDay, $parentsById, $generatedAt, $filtersLine, $tz) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            $usedSheets = [];

            $first = true;
            foreach ($byDay as $day => $dataset) {
                // Name sheet as "YYYY-MM-DD (ddd)"
                $dayCarbon = Carbon::parse($day);
                $sheetLabel = $dayCarbon->format('Y-m-d') . ' (' . $dayCarbon->format('D') . ')';
                $sheetName  = $this->uniqueSheetName($usedSheets, $this->safeSheetName($sheetLabel));

                if ($first) {
                    $writer->getCurrentSheet()->setName($sheetName);
                    $first = false;
                } else {
                    $writer->addNewSheetAndMakeItCurrent();
                    $writer->getCurrentSheet()->setName($sheetName);
                }

                // Header
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ["ATTENDANCE — " . $dayCarbon->isoFormat('D MMMM YYYY')],
                    ['Generated At', $generatedAt],
                    $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                ]);

                // Table (pass timezone)
                $this->writeAttendanceTable($writer, $dataset, $headerStyle, $wrapStyle, $parentsById, $tz);

                // Footer totals per sheet/day
                $this->writeAttendanceTotals($writer, $dataset);
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /** Table writer for a day's consultations. */
    private function writeAttendanceTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle,
        $parentsById,
        string $tz = 'Africa/Harare'
    ): void {
        $headers = [
            'Time',
            'Patient FirstName',
            'Patient Surname',
            'Gender',
            'Category',                 // Employee | Dependent
            'Account Holder',           // for dependents
            'Holder Employee #',        // for dependents
            'Account Holder Company',   // company of the account holder
            'Relationship',             // for dependents
            'Clinic',
        ];
        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Column widths (by header)
        $opts = $writer->getOptions();
        $setW = function (string $label, int $width) use ($headers, $opts) {
            $idx = array_search($label, $headers, true);
            if ($idx !== false) {
                $opts->setColumnWidth($width, $idx + 1); // 1-based
            }
        };
        $setW('Time', 12);
        $setW('Patient FirstName', 26);
        $setW('Patient Surname', 26);
        $setW('Gender', 10);
        $setW('Category', 14);
        $setW('Account Holder', 28);
        $setW('Holder Employee #', 20);
        $setW('Account Holder Company', 28);
        $setW('Relationship', 16);
        $setW('Clinic', 24);

        // Ensure in-day ordering by created_at time (timezone-adjusted for display)
        $sorted = $dataset->sortBy(function ($c) {
            /** @var \App\Models\Consultation $c */
            return Carbon::parse($c->created_at)->format('H:i:s');
        });

        foreach ($sorted as $c) {
            $patient = $c->patient;
            $isDependent = !empty(optional($patient)->parent_patient_id);

            $holderName = '';
            $holderEmp  = '';
            $holderCompanyName = '';
            $relationship = (string) ($patient->relationship ?? '');

            if ($isDependent) {
                $parentId = $patient->parent_patient_id;
                $holder   = $parentId ? $parentsById->get($parentId) : null;
                $holderName = $holder ? trim(($holder->first_name ?? '') . ' ' . ($holder->surname ?? '')) : '';
                $holderEmp  = $holder->employee_number ?? '';
                $holderCompanyName = $holder && $holder->company ? (string) $holder->company->name : '';
            }

            // Time column: use created_at in Africa/Harare timezone
            $timeCell = $c->created_at
                ? Carbon::parse($c->created_at)->setTimezone($tz)->format('H:i')
                : '';

            $clinic = $this->resolveClinicLabelFromConsultation($c);

            $row = [
                $timeCell,
                $patient->first_name ?? '',
                $patient->surname ?? '',
                (string) ($patient->gender ?? ''),
                $isDependent ? 'Dependent' : 'Employee',
                $holderName,
                $holderEmp,
                $holderCompanyName,
                $relationship,
                $clinic,
            ];

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    private function writeAttendanceTotals(XLSXWriter $writer, $dataset): void
    {
        $employees  = $dataset->filter(fn ($c) => empty(optional($c->patient)->parent_patient_id))->count();
        $dependents = $dataset->filter(fn ($c) => !empty(optional($c->patient)->parent_patient_id))->count();

        $writer->addRow(Row::fromValues([]));
        $writer->addRow(Row::fromValues(['Total Attended', $dataset->count()]));
        $writer->addRow(Row::fromValues(['Employees', $employees]));
        $writer->addRow(Row::fromValues(['Dependents', $dependents]));
    }

    /** "Clinic Name (City)" from Consultation relation. */
    private function resolveClinicLabelFromConsultation($consultation): string
    {
        $c = $consultation->clinic;
        if (!$c) return '';
        $name = (string) ($c->name ?? '');
        $city = (string) ($c->city ?? '');
        return trim($name . ($city ? " ({$city})" : ''));
    }

    /* ---------------- helpers reused across reports ---------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
