<?php

namespace App\Http\Controllers;

use App\Mail\DailyDispenseReportMail;
use App\Models\Dispensation;
use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class MonthlyDispenseReportController extends Controller
{
    /**
     * Download an Excel of all medication DISPENSED within the given month/year.
     * - Respects filters (search, patient_id, dispensed_by, optional clinic_id).
     * - ALWAYS scopes to clinics the authenticated user can access.
     * - If user's role is NOT doctor or nurse, the Diagnosis column is hidden.
     * - If no "dispensed_by" filter is chosen: creates "All Dispensations", per-dispenser sheets,
     *   and per-clinic "Employees" & "Dependents" sheets.
     */
    public function monthly(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month, 422, 'Year and month are required.');

        $search      = $request->query('search');
        $patientId   = $request->query('patient_id');
        $dispensedBy = $request->query('dispensed_by'); // user id (optional)
        $clinicId    = $request->query('clinic_id');    // optional, still scoped to access

        $periodStart = Carbon::create($year, $month, 1)->startOfMonth();
        $periodEnd   = (clone $periodStart)->endOfMonth();
        $periodLabel = $periodStart->isoFormat('MMMM YYYY');

        // Determine role & clinic scope
        $role = (string)($user->role?->value ?? $user->role ?? '');
        $includeDiagnosis = in_array($role, ['doctor','nurse'], true);

        $allowedClinicIds = $this->clinicIdsFor($user);
        $isSuperadmin     = ($role === 'superadmin');

        // If a clinic filter is provided, intersect with allowed clinics.
        // Superadmin: allow explicit clinic_id or all if none given.
        if ($clinicId) {
            $clinicId = (int)$clinicId;
            if (!$isSuperadmin && !in_array($clinicId, $allowedClinicIds, true)) {
                // No access -> return empty workbook
                $rows = collect();
                return $this->streamMonthlyWorkbook(
                    $rows, $periodLabel, $search, $patientId, $dispensedBy, $clinicId,
                    $includeDiagnosis
                );
            }
            $effectiveClinics = [$clinicId];
        } else {
            $effectiveClinics = $isSuperadmin ? [] : $allowedClinicIds;
        }

        // Build base query
        $query = Dispensation::query()
            ->with([
                'patient:id,first_name,surname,employee_number,parent_patient_id,relationship,company_id',
                'dispensedBy:id,name',
                'consultation:id,diagnosis,sick_leave,clinic_id,consultation_date',
                'consultation.clinic:id,name,city',
                'clinicMedicationStock.clinic:id,name,city',
                'nurseMedicationStock.clinic:id,name,city',
                'clinicMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
                'nurseMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
                'clinicMedicationStock.medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
                'nurseMedicationStock.medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
            ])
            // Month filter: prefer dispensed_at, fallback to created_at
            ->where(function ($q) use ($periodStart, $periodEnd) {
                $q->whereBetween('dispensed_at', [$periodStart, $periodEnd])
                  ->orWhere(function ($qq) use ($periodStart, $periodEnd) {
                      $qq->whereNull('dispensed_at')
                         ->whereBetween('created_at', [$periodStart, $periodEnd]);
                  });
            })
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('notes', 'like', "%{$search}%")
                       ->orWhereHas('patient', fn($p) =>
                           $p->where('first_name', 'like', "%{$search}%")
                             ->orWhere('surname', 'like', "%{$search}%")
                       )
                       ->orWhereHas('clinicMedicationStock.medicationBatch', fn($b) =>
                           $b->where('batch_number', 'like', "%{$search}%")
                       )
                       ->orWhereHas('nurseMedicationStock.medicationBatch', fn($b) =>
                           $b->where('batch_number', 'like', "%{$search}%")
                       )
                       ->orWhereHas('clinicMedicationStock.medicationBatch.medication', fn($m) =>
                           $m->where('name', 'like', "%{$search}%")
                       )
                       ->orWhereHas('nurseMedicationStock.medicationBatch.medication', fn($m) =>
                           $m->where('name', 'like', "%{$search}%")
                       );
                });
            })
            ->when($patientId, fn($q) => $q->where('patient_id', $patientId))
            ->when($dispensedBy, fn($q) => $q->where('dispensed_by', $dispensedBy))
            // 🔒 Scope to clinics the user can access (or explicit filter above)
            ->when(!$isSuperadmin || !empty($effectiveClinics), function ($q) use ($effectiveClinics, $isSuperadmin) {
                // If superadmin and no effectiveClinics set, skip restriction (see all)
                if ($isSuperadmin && empty($effectiveClinics)) {
                    return;
                }
                $q->where(function ($qq) use ($effectiveClinics) {
                    $qq->whereHas('consultation', fn($c) => $c->whereIn('clinic_id', $effectiveClinics))
                       ->orWhereHas('clinicMedicationStock.clinic', fn($c) => $c->whereIn('id', $effectiveClinics))
                       ->orWhereHas('nurseMedicationStock.clinic', fn($c) => $c->whereIn('id', $effectiveClinics));
                });
            })
            ->orderBy('dispensed_at');

        $rows = $query->get();

        return $this->streamMonthlyWorkbook(
            $rows, $periodLabel, $search, $patientId, $dispensedBy, $clinicId,
            $includeDiagnosis
        );
    }

    /** Stream monthly workbook (shared by monthly()). */
    private function streamMonthlyWorkbook($rows, string $periodLabel, ?string $search, $patientId, $dispensedBy, $clinicId, bool $includeDiagnosis)
    {
        // Preload account holders for dependents (to avoid N+1)
        $parentIds   = $rows->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentsById = $parentIds->isNotEmpty()
            ? Patient::whereIn('id', $parentIds)->get(['id', 'first_name', 'surname', 'employee_number'])->keyBy('id')
            : collect();

        // Filters line
        $generatedAt = now()->format('Y-m-d H:i:s');
        $filters     = ["Period: {$periodLabel}"];
        if ($search)    { $filters[] = "Search: {$search}"; }
        if ($patientId) { $filters[] = "Patient ID: {$patientId}"; }
        if ($dispensedBy) { $filters[] = "Dispensed By: {$dispensedBy}"; }
        if ($clinicId)  { $filters[] = "Clinic ID: {$clinicId}"; }
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Dispensed_Medication_%s.xlsx', str_replace(' ', '_', $periodLabel));

        return response()->streamDownload(function () use ($rows, $generatedAt, $filtersLine, $periodLabel, $dispensedBy, $parentsById, $includeDiagnosis) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // ====== Sheet A: All Dispensations (always) ======
            $writer->getCurrentSheet()->setName($this->safeSheetName('All Dispensations'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['DISPENSATION REPORT — ALL'],
                ['Generated At', $generatedAt],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);
            $this->writeDispensationsTable($writer, $rows, $headerStyle, $wrapStyle, [
                'includeClinic'    => true,
                'dependentsExtra'  => false,
                'parentsById'      => $parentsById,
                'includeDiagnosis' => $includeDiagnosis,
            ]);
            $this->writeDispensationsTotals($writer, $rows);

            // ====== If NO specific dispenser filter: add per-dispenser sheets ======
            if (!$dispensedBy) {
                $byDispenser = $rows->groupBy(fn($d) => optional($d->dispensedBy)->name ?? 'Unknown');
                $used        = ['All Dispensations' => true];

                foreach ($byDispenser as $dispenser => $dataset) {
                    $writer->addNewSheetAndMakeItCurrent();
                    $sheetName = $this->uniqueSheetName($used, $this->safeSheetName("By {$dispenser}"));
                    $writer->getCurrentSheet()->setName($sheetName);

                    $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                        ["DISPENSATION REPORT — {$dispenser}"],
                        ['Generated At', $generatedAt],
                        $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                    ]);
                    $this->writeDispensationsTable($writer, $dataset, $headerStyle, $wrapStyle, [
                        'includeClinic'    => true,
                        'dependentsExtra'  => false,
                        'parentsById'      => $parentsById,
                        'includeDiagnosis' => $includeDiagnosis,
                    ]);
                    $this->writeDispensationsTotals($writer, $dataset);
                }
            }

            // ====== Per-Clinic: Employees vs Dependents ======
            $byClinic   = $rows->groupBy(fn($d) => $this->resolveClinicLabel($d) ?: 'Unknown Clinic');
            $usedSheets = [];

            foreach ($byClinic as $clinicLabel => $dataset) {
                // Employees
                $employees = $dataset->filter(fn($d) => empty(optional($d->patient)->parent_patient_id));
                if ($employees->isNotEmpty()) {
                    $writer->addNewSheetAndMakeItCurrent();
                    $sheetName = $this->uniqueSheetName($usedSheets, $this->safeSheetName("{$clinicLabel} — Employees"));
                    $writer->getCurrentSheet()->setName($sheetName);

                    $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                        ["DISPENSATION REPORT — {$clinicLabel} (Employees)"],
                        ['Generated At', $generatedAt],
                    ]);
                    $this->writeDispensationsTable($writer, $employees, $headerStyle, $wrapStyle, [
                        'includeClinic'    => false,
                        'dependentsExtra'  => false,
                        'parentsById'      => $parentsById,
                        'includeDiagnosis' => $includeDiagnosis,
                    ]);
                    $this->writeDispensationsTotals($writer, $employees);
                }

                // Dependents
                $dependents = $dataset->filter(fn($d) => !empty(optional($d->patient)->parent_patient_id));
                if ($dependents->isNotEmpty()) {
                    $writer->addNewSheetAndMakeItCurrent();
                    $sheetName = $this->uniqueSheetName($usedSheets, $this->safeSheetName("{$clinicLabel} — Dependents"));
                    $writer->getCurrentSheet()->setName($sheetName);

                    $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                        ["DISPENSATION REPORT — {$clinicLabel} (Dependents)"],
                        ['Generated At', $generatedAt],
                    ]);
                    $this->writeDispensationsTable($writer, $dependents, $headerStyle, $wrapStyle, [
                        'includeClinic'    => false,
                        'dependentsExtra'  => true,
                        'parentsById'      => $parentsById,
                        'includeDiagnosis' => $includeDiagnosis,
                    ]);
                    $this->writeDispensationsTotals($writer, $dependents);
                }
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /** Daily email report:
     * - Scopes to clinics user can access.
     * - Hides Diagnosis unless user is doctor/nurse.
     */
    public function emailDaily(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $role = (string)($user->role?->value ?? $user->role ?? '');
        $includeDiagnosis = in_array($role, ['doctor','nurse'], true);

        $allowedClinicIds = $this->clinicIdsFor($user);
        $isSuperadmin     = ($role === 'superadmin');

        // --- Fixed recipients (single To + BCC) and current date ---
        $to  = 'jane@providencehumancapital.com';
        $bcc = ['gashiratafadzwa@gmail.com'];

        $date     = now(); // app timezone
        $dayStart = $date->copy()->startOfDay();
        $dayEnd   = $date->copy()->endOfDay();

        // Optional filters (still supported)
        $search      = $request->query('search');       // optional
        $patientId   = $request->query('patient_id');   // optional
        $dispensedBy = $request->query('dispensed_by'); // optional (ignored for sheet split)
        $clinicId    = $request->query('clinic_id');    // optional (still scoped)

        // If specific clinic provided, enforce access unless superadmin
        if ($clinicId) {
            $clinicId = (int)$clinicId;
            if (!$isSuperadmin && !in_array($clinicId, $allowedClinicIds, true)) {
                // Return empty (still generate workbook + email)
                $rows = collect();
                return $this->emailDailyWithRows($rows, $dayStart, $to, $bcc, $search, $patientId, $clinicId, $includeDiagnosis, $dispensedBy);
            }
            $effectiveClinics = [$clinicId];
        } else {
            $effectiveClinics = $isSuperadmin ? [] : $allowedClinicIds;
        }

        // Build query (same relationships as monthly())
        $query = Dispensation::query()
            ->with([
                'patient:id,first_name,surname,employee_number,parent_patient_id,relationship,company_id',
                'dispensedBy:id,name',
                'consultation:id,diagnosis,sick_leave,clinic_id,consultation_date',
                'consultation.clinic:id,name,city',
                'clinicMedicationStock.clinic:id,name,city',
                'nurseMedicationStock.clinic:id,name,city',
                'clinicMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
                'nurseMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
                'clinicMedicationStock.medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
                'nurseMedicationStock.medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
            ])
            // Day filter: prefer dispensed_at, fallback to created_at
            ->where(function ($q) use ($dayStart, $dayEnd) {
                $q->whereBetween('dispensed_at', [$dayStart, $dayEnd])
                  ->orWhere(function ($qq) use ($dayStart, $dayEnd) {
                      $qq->whereNull('dispensed_at')
                         ->whereBetween('created_at', [$dayStart, $dayEnd]);
                  });
            })
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('notes', 'like', "%{$search}%")
                       ->orWhereHas('patient', fn($p) =>
                           $p->where('first_name', 'like', "%{$search}%")
                             ->orWhere('surname', 'like', "%{$search}%")
                       )
                       ->orWhereHas('clinicMedicationStock.medicationBatch', fn($b) =>
                           $b->where('batch_number', 'like', "%{$search}%")
                       )
                       ->orWhereHas('nurseMedicationStock.medicationBatch', fn($b) =>
                           $b->where('batch_number', 'like', "%{$search}%")
                       )
                       ->orWhereHas('clinicMedicationStock.medicationBatch.medication', fn($m) =>
                           $m->where('name', 'like', "%{$search}%")
                       )
                       ->orWhereHas('nurseMedicationStock.medicationBatch.medication', fn($m) =>
                           $m->where('name', 'like', "%{$search}%")
                       );
                });
            })
            ->when($patientId, fn($q) => $q->where('patient_id', $patientId))
            ->when($dispensedBy, fn($q) => $q->where('dispensed_by', $dispensedBy))
            // 🔒 Scope to clinics the user can access (or explicit clinic)
            ->when(!$isSuperadmin || !empty($effectiveClinics), function ($q) use ($effectiveClinics, $isSuperadmin) {
                if ($isSuperadmin && empty($effectiveClinics)) {
                    return;
                }
                $q->where(function ($qq) use ($effectiveClinics) {
                    $qq->whereHas('consultation', fn($c) => $c->whereIn('clinic_id', $effectiveClinics))
                       ->orWhereHas('clinicMedicationStock.clinic', fn($c) => $c->whereIn('id', $effectiveClinics))
                       ->orWhereHas('nurseMedicationStock.clinic', fn($c) => $c->whereIn('id', $effectiveClinics));
                });
            })
            ->orderBy('dispensed_at');

        $rows = $query->get();

        return $this->emailDailyWithRows($rows, $dayStart, $to, $bcc, $search, $patientId, $clinicId, $includeDiagnosis, $dispensedBy);
    }

    private function emailDailyWithRows($rows, Carbon $dayStart, string $to, array $bcc, ?string $search, $patientId, $clinicId, bool $includeDiagnosis, $dispensedBy)
    {
        // Preload account holders for dependents
        $parentIds   = $rows->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentsById = $parentIds->isNotEmpty()
            ? Patient::whereIn('id', $parentIds)->get(['id', 'first_name', 'surname', 'employee_number'])->keyBy('id')
            : collect();

        // Filters line for header + email body
        $filters = ["Date: " . $dayStart->isoFormat('ddd, D MMM YYYY')];
        if ($search)    { $filters[] = "Search: {$search}"; }
        if ($patientId) { $filters[] = "Patient ID: {$patientId}"; }
        if ($clinicId)  { $filters[] = "Clinic ID: {$clinicId}"; }
        $filtersLine = implode(' | ', $filters);

        // Build XLSX to a temp file
        $fileName = sprintf('Dispensed_Medication_%s.xlsx', $dayStart->format('Y_m_d'));
        $dir      = 'reports/daily';
        Storage::makeDirectory($dir);
        $tmpPath = Storage::path($dir . '/' . $fileName);

        $writer = new XLSXWriter();
        $writer->openToFile($tmpPath);

        $titleStyle  = (new \OpenSpout\Common\Entity\Style\Style())->setFontBold();
        $labelStyle  = (new \OpenSpout\Common\Entity\Style\Style())->setFontBold();
        $headerStyle = (new \OpenSpout\Common\Entity\Style\Style())->setFontBold();
        $wrapStyle   = (new \OpenSpout\Common\Entity\Style\Style())->setShouldWrapText(true);

        // ===== Sheet 1: All Dispensations =====
        $writer->getCurrentSheet()->setName($this->safeSheetName('All Dispensations'));
        $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
            ['DAILY DISPENSATION REPORT — ALL'],
            ['Generated At', now()->format('Y-m-d H:i:s')],
            $filtersLine !== '' ? ['Filters', $filtersLine] : null,
        ]);
        $this->writeDispensationsTable($writer, $rows, $headerStyle, $wrapStyle, [
            'includeClinic'    => true,
            'dependentsExtra'  => false,
            'parentsById'      => $parentsById,
            'includeDiagnosis' => $includeDiagnosis,
        ]);
        $this->writeDispensationsTotals($writer, $rows);

        // ===== One sheet per dispenser =====
        $byDispenser = $rows->groupBy(fn($d) => optional($d->dispensedBy)->name ?? 'Unknown');
        $usedSheets  = ['All Dispensations' => true];

        foreach ($byDispenser as $dispenserName => $dataset) {
            $writer->addNewSheetAndMakeItCurrent();
            $sheetName = $this->uniqueSheetName($usedSheets, $this->safeSheetName($dispenserName));
            $writer->getCurrentSheet()->setName($sheetName);

            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ["DAILY DISPENSATION REPORT — {$dispenserName}"],
                ['Generated At', now()->format('Y-m-d H:i:s')],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);
            $this->writeDispensationsTable($writer, $dataset, $headerStyle, $wrapStyle, [
                'includeClinic'    => true,
                'dependentsExtra'  => false,
                'parentsById'      => $parentsById,
                'includeDiagnosis' => $includeDiagnosis,
            ]);
            $this->writeDispensationsTotals($writer, $dataset);
        }

        $writer->close();

        // Send email (single To + BCC)
        $mailable = new DailyDispenseReportMail($dayStart, $tmpPath, $fileName, $filtersLine);
        Mail::to($to)->bcc($bcc)->send($mailable);

        // Clean up temp file
        try { @unlink($tmpPath); } catch (\Throwable $e) {}

        return response()->json([
            'ok'      => true,
            'sent_to' => $to,
            'bcc'     => $bcc,
            'date'    => $dayStart->toDateString(),
            'sheets'  => array_merge(['All Dispensations'], array_keys($byDispenser->toArray())),
            'diagnosis_included' => $includeDiagnosis,
        ]);
    }

    /** Render table. Options:
     *  - includeClinic: include Clinic column
     *  - dependentsExtra: include Account Holder / Employee Number / Relationship
     *  - parentsById: map of parent patient (for dependents)
     *  - includeDiagnosis: include Diagnosis column (only for doctor/nurse)
     */
    private function writeDispensationsTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle,
        array $options = []
    ): void {
        $includeClinic    = (bool) ($options['includeClinic'] ?? true);
        $dependentsExtra  = (bool) ($options['dependentsExtra'] ?? false);
        $includeDiagnosis = (bool) ($options['includeDiagnosis'] ?? false);
        /** @var \Illuminate\Support\Collection $parentsById */
        $parentsById = $options['parentsById'] ?? collect();

        // Build headers
        $headers = [
            'Date/Time',
            'Patient FirstName',
            'Patient Surname',
        ];

        if ($dependentsExtra) {
            $headers[] = 'Account Holder';
            $headers[] = 'Employee Number';
            $headers[] = 'Relationship';
        }

        $headers = array_merge($headers, [
            'Medication',
            'Quantity',
            'Dosage/Route',
            'Dispensed By',
        ]);

        if ($includeClinic) {
            $headers[] = 'Clinic';
        }

        if ($includeDiagnosis) {
            $headers[] = 'Diagnosis';
        }

        $headers = array_merge($headers, [
            'Sick Leave',
            'Dispense Notes',
        ]);

        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Set useful column widths by header lookup
        $opts = $writer->getOptions();
        $setW = function (string $label, int $width) use ($headers, $opts) {
            $idx = array_search($label, $headers, true);
            if ($idx !== false) {
                $opts->setColumnWidth($width, $idx + 1); // 1-based index
            }
        };
        $setW('Date/Time', 22);
        $setW('Patient FirstName', 28);
        $setW('Patient Surname', 28);
        $setW('Dosage/Route', 42);
        if ($includeDiagnosis) {
            $setW('Diagnosis', 36);
        }
        $setW('Dispense Notes', 36);
        if ($includeClinic) {
            $setW('Clinic', 22);
        }
        if ($dependentsExtra) {
            $setW('Account Holder', 28);
            $setW('Employee Number', 18);
            $setW('Relationship', 16);
        }
        $setW('Medication', 28);

        foreach ($dataset as $d) {
            $patient   = $d->patient;
            $dispenser = $d->dispensedBy;
            $medName   = $this->resolveMedicationName($d);
            $diagnosis = $includeDiagnosis ? (optional($d->consultation)->diagnosis ?? '') : ''; // respect flag
            $sickLeave = optional($d->consultation)->sick_leave ? '✓' : '';
            $clinic    = $this->resolveClinicLabel($d);

            // Compact dosage/route string
            $dose = trim(collect([
                $d->route ?: null,
                $d->frequency ? "Freq: {$d->frequency}" : null,
                $d->duration ? "Dur: {$d->duration}" : null,
            ])->filter()->implode(' • '));

            // Date/time preferring dispensed_at
            $dateCell = optional($d->dispensed_at)->format('Y-m-d H:i') ?? (optional($d->created_at)->format('Y-m-d H:i') ?? '');

            $row = [
                $dateCell,
                $patient->first_name ?? '',
                $patient->surname ?? '',
            ];

            if ($dependentsExtra) {
                $parentId   = $patient->parent_patient_id ?? null;
                $holder     = $parentId ? $parentsById->get($parentId) : null;
                $holderName = $holder ? trim(($holder->first_name ?? '') . ' ' . ($holder->surname ?? '')) : '';
                $holderEmp  = $holder->employee_number ?? '';

                $row[] = $holderName;
                $row[] = $holderEmp;
                $row[] = (string) ($patient->relationship ?? '');
            }

            $row = array_merge($row, [
                $medName,
                (int) ($d->quantity ?? 0),
                $dose,
                $dispenser->name ?? '',
            ]);

            if ($includeClinic) {
                $row[] = $clinic;
            }

            if ($includeDiagnosis) {
                $row[] = $diagnosis;
            }

            $row = array_merge($row, [
                $sickLeave,
                (string) ($d->notes ?? ''),
            ]);

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    private function writeDispensationsTotals(XLSXWriter $writer, $dataset): void
    {
        $writer->addRow(Row::fromValues([]));
        $writer->addRow(Row::fromValues(['Total Dispensations', $dataset->count()]));
        $writer->addRow(Row::fromValues(['Total Quantity', (int) $dataset->sum('quantity')]));
    }

    /** Resolve medication name from clinic/nurse stock paths. */
    private function resolveMedicationName($dispensation): string
    {
        $paths = [
            'clinicMedicationStock.medicationBatch.medication.name',
            'nurseMedicationStock.medicationBatch.medication.name',
        ];
        foreach ($paths as $path) {
            $val = data_get($dispensation, $path);
            if (!empty($val)) {
                return (string) $val;
            }
        }
        return '';
    }

    /** Resolve a clinic label "Name (City)" from consultation or stock fallbacks. */
    private function resolveClinicLabel($dispensation): string
    {
        $c1 = data_get($dispensation, 'consultation.clinic');
        $c2 = data_get($dispensation, 'clinicMedicationStock.clinic');
        $c3 = data_get($dispensation, 'nurseMedicationStock.clinic');
        $c  = $c1 ?: ($c2 ?: $c3);

        if (!$c) return '';

        $name = (string) ($c->name ?? '');
        $city = (string) ($c->city ?? '');
        return trim($name . ($city ? " ({$city})" : ''));
    }

    /* --------------------- helpers --------------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';

        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i    = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name   = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }

    /** Get clinic IDs the user can access (primary + pivot). */
    private function clinicIdsFor($user): array
    {
        if (method_exists($user, 'accessibleClinicIds')) {
            $ids = $user->accessibleClinicIds() ?: [];
            return array_values(array_unique(array_map('intval', $ids)));
        }

        $ids = [];
        if (method_exists($user, 'accessibleClinics')) {
            $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
        }
        if (!empty($user->clinic_id) && !in_array((int)$user->clinic_id, $ids, true)) {
            $ids[] = (int)$user->clinic_id;
        }
        return array_values(array_unique(array_map('intval', $ids)));
    }
}
