<?php
namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class NurseDashboardReportsController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        abort_if(! $user, 401);

        // Only nurses, doctors, superadmins
        $role = $user->role?->value ?? (string) $user->role;
        abort_if(! in_array($role, ['nurse', 'doctor', 'superadmin']), 403);

        // Anchor date (defaults to today). We'll show last 7 days ending on this date.
        $anchor = $request->filled('date')
            ? Carbon::parse((string) $request->input('date'))
            : now();

        $startOfDay = (clone $anchor)->startOfDay();
        $endOfDay   = (clone $anchor)->endOfDay();

        $last7Start = (clone $anchor)->subDays(6)->startOfDay();
        $last7End   = (clone $anchor)->endOfDay();

        // Clinics this user can access (primary + pivot)
        if (method_exists($user, 'accessibleClinicIds')) {
            $clinicIds = $user->accessibleClinicIds();
        } else {
            $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            if (! empty($user->clinic_id) && ! in_array((int) $user->clinic_id, $ids, true)) {
                $ids[] = (int) $user->clinic_id;
            }
            $clinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        // We'll attribute consultations to the logged-in clinician via consultations.doctor_id
        // and use this datetime expression for all date filters/grouping.
        $dateExprSql = 'COALESCE(consultations.consultation_date, consultations.created_at)';

        // Base (ALL-TIME) – filtered by doctor + clinics
        $baseAll = DB::table('consultations')
            ->where('consultations.doctor_id', $user->id)
            ->when(! empty($clinicIds), fn($q) => $q->whereIn('consultations.clinic_id', $clinicIds));

        // Base (TODAY)
        $baseDay = (clone $baseAll)
            ->whereBetween(DB::raw($dateExprSql), [$startOfDay, $endOfDay]);

        // 1) Consultations TODAY (single number)
        $totalToday = (clone $baseDay)->count();

        // 2) Consultations – LAST 7 DAYS (bar series)
        $last7Raw = (clone $baseAll)
            ->whereBetween(DB::raw($dateExprSql), [$last7Start, $last7End])
            ->selectRaw("DATE($dateExprSql) AS d, COUNT(*) AS c")
            ->groupBy('d')
            ->orderBy('d')
            ->get()
            ->keyBy('d');

        $last7 = [];
        for ($i = 6; $i >= 0; $i--) {
            $day     = (clone $anchor)->subDays($i);
            $date    = $day->toDateString();
            $label   = $day->format('D d M'); // e.g., "Mon 23 Sep"
            $count   = (int) ($last7Raw[$date]->c ?? 0);
            $last7[] = ['date' => $date, 'label' => $label, 'count' => $count];
        }

        // 3) Referrals & Sick Leave – ALL TIME (for this doctor, scoped to clinics)
        $ref = (clone $baseAll)
            ->selectRaw('SUM(CASE WHEN consultations.refer_for_radiology=1 THEN 1 ELSE 0 END) AS radiology')
            ->selectRaw('SUM(CASE WHEN consultations.refer_for_lab=1 THEN 1 ELSE 0 END) AS lab')
            ->selectRaw('SUM(CASE WHEN consultations.refer_to_specialist=1 THEN 1 ELSE 0 END) AS specialist')
            ->selectRaw('SUM(CASE WHEN consultations.refer_to_casualty=1 THEN 1 ELSE 0 END) AS casualty')
            ->selectRaw('SUM(CASE WHEN consultations.refer_out_patient=1 THEN 1 ELSE 0 END) AS out_patient')
            ->selectRaw('SUM(CASE WHEN consultations.sick_leave=1 THEN 1 ELSE 0 END) AS sick_leave')
            ->first();

        $referrals = [
            'radiology'   => (int) ($ref->radiology ?? 0),
            'lab'         => (int) ($ref->lab ?? 0),
            'specialist'  => (int) ($ref->specialist ?? 0),
            'casualty'    => (int) ($ref->casualty ?? 0),
            'out_patient' => (int) ($ref->out_patient ?? 0),
            'sick_leave'  => (int) ($ref->sick_leave ?? 0),
        ];
        $referrals['total_referrals'] =
            $referrals['radiology'] +
            $referrals['lab'] +
            $referrals['specialist'] +
            $referrals['casualty'] +
            $referrals['out_patient'];

        // 4) Injury On Duty – ALL TIME
        $iodCount = (clone $baseAll)->where('consultations.injury_on_duty', 1)->count();

        // 5) Per-clinic counts – TODAY
        $perClinic = (clone $baseDay)
            ->join('clinics', 'clinics.id', '=', 'consultations.clinic_id')
            ->select('consultations.clinic_id', 'clinics.name as clinic_name', DB::raw('COUNT(*) AS count'))
            ->groupBy('consultations.clinic_id', 'clinics.name')
            ->orderByDesc('count')
            ->get()
            ->map(fn($r) => [
                'clinic_id'   => (int) $r->clinic_id,
                'clinic_name' => (string) $r->clinic_name,
                'count'       => (int) $r->count,
            ])
            ->values()
            ->all();

        // 6) Top diagnoses – ALL TIME (split on , ; | / newline). Limit 15
        $diagnosesRaw = (clone $baseAll)
            ->whereNotNull('consultations.diagnosis')
            ->pluck('consultations.diagnosis')
            ->all();

        $diagCounts = [];
        foreach ($diagnosesRaw as $txt) {
            $pieces = preg_split('/[,\;\|\n\/]+/u', (string) $txt);
            foreach ($pieces as $p) {
                $name = trim($p);
                if ($name === '' || mb_strlen($name) < 2) {
                    continue;
                }

                $norm              = mb_convert_case($name, MB_CASE_TITLE, 'UTF-8');
                $diagCounts[$norm] = ($diagCounts[$norm] ?? 0) + 1;
            }
        }
        arsort($diagCounts);
        $diagnoses = [];
        foreach (array_slice($diagCounts, 0, 15, true) as $k => $v) {
            $diagnoses[] = ['name' => $k, 'count' => (int) $v];
        }

        // 7) Extras (TODAY): payment methods (triages) + vitals averages
        $paymentMethods = DB::table('consultations')
            ->leftJoin('triages', 'triages.consultation_id', '=', 'consultations.id')
            ->where('consultations.doctor_id', $user->id)
            ->when(! empty($clinicIds), fn($q) => $q->whereIn('consultations.clinic_id', $clinicIds))
            ->whereBetween(DB::raw($dateExprSql), [$startOfDay, $endOfDay])
            ->select('triages.payment_method', DB::raw('COUNT(*) AS count'))
            ->groupBy('triages.payment_method')
            ->orderByDesc('count')
            ->get()
            ->map(fn($r) => [
                'payment_method' => (string) ($r->payment_method ?? 'N/A'),
                'count'          => (int) $r->count,
            ])
            ->values()
            ->all();

        $avgVitals = [
            'temperature' => (float) ((clone $baseDay)->avg('consultations.temperature') ?? 0),
            'systolic'    => (float) ((clone $baseDay)->avg('consultations.systolic') ?? 0),
            'diastolic'   => (float) ((clone $baseDay)->avg('consultations.diastolic') ?? 0),
        ];

        // Clinics for badges
        $allowedClinics = DB::table('clinics')
            ->whereIn('clinics.id', $clinicIds ?: [-1])
            ->select('clinics.id', 'clinics.name')
            ->orderBy('name')
            ->get();

        return Inertia::render('Nurse/Reports', [
            'day'            => $startOfDay->toDateString(),
            'nurseName'      => $user->name,
            'totalToday'     => $totalToday,
            'last7'          => $last7,          // [{date,label,count}]
            'referrals'      => $referrals,      // ALL TIME
            'iodCount'       => $iodCount,       // ALL TIME
            'perClinic'      => $perClinic,      // TODAY
            'diagnoses'      => $diagnoses,      // ALL TIME
            'paymentMethods' => $paymentMethods, // TODAY
            'avgVitals'      => $avgVitals,      // TODAY
            'allowedClinics' => $allowedClinics,
        ]);
    }

    public function stats(Request $request)
    {
        $user = Auth::user();
        abort_if(! $user, 401);

        $role = $user->role?->value ?? (string) $user->role;
        abort_if(! in_array($role, ['nurse', 'doctor', 'superadmin']), 403);

        // Day (defaults to today)
        $day   = $request->filled('date') ? Carbon::parse($request->string('date')) : now();
        $start = (clone $day)->startOfDay();
        $end   = (clone $day)->endOfDay();

        // Clinics this user can access (primary + pivot)
        if (method_exists($user, 'accessibleClinicIds')) {
            $clinicIds = $user->accessibleClinicIds();
        } else {
            $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            if (! empty($user->clinic_id) && ! in_array((int) $user->clinic_id, $ids, true)) {
                $ids[] = (int) $user->clinic_id;
            }
            $clinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        // doctor-based stats
        $dateCol = DB::raw('COALESCE(consultations.consultation_date, consultations.created_at)');
        $base    = DB::table('consultations')
            ->where('consultations.doctor_id', $user->id)
            ->when(! empty($clinicIds), fn($q) => $q->whereIn('consultations.clinic_id', $clinicIds));

        // Today
        $totalToday = (clone $base)->whereBetween($dateCol, [$start, $end])->count();

        // Last 7 days (ending on selected day)
        $last7 = [];
        for ($i = 6; $i >= 0; $i--) {
            $d       = (clone $day)->subDays($i);
            $ds      = (clone $d)->startOfDay();
            $de      = (clone $d)->endOfDay();
            $cnt     = (clone $base)->whereBetween($dateCol, [$ds, $de])->count();
            $last7[] = [
                'date'  => $d->toDateString(),
                'label' => $d->format('D j'),
                'count' => (int) $cnt,
            ];
        }

        // Today: per-clinic counts
        $perClinic = (clone $base)
            ->join('clinics', 'clinics.id', '=', 'consultations.clinic_id')
            ->whereBetween($dateCol, [$start, $end])
            ->select('consultations.clinic_id', 'clinics.name as clinic_name', DB::raw('COUNT(*) AS count'))
            ->groupBy('consultations.clinic_id', 'clinics.name')
            ->orderByDesc('count')
            ->get()
            ->map(fn($r) => [
                'clinic_id'   => (int) $r->clinic_id,
                'clinic_name' => (string) $r->clinic_name,
                'count'       => (int) $r->count,
            ])
            ->values()
            ->all();

        // All-time: referrals + sick leave
        $ref = (clone $base)
            ->selectRaw('SUM(consultations.refer_for_radiology = 1) AS radiology')
            ->selectRaw('SUM(consultations.refer_for_lab = 1) AS lab')
            ->selectRaw('SUM(consultations.refer_to_specialist = 1) AS specialist')
            ->selectRaw('SUM(consultations.refer_to_casualty = 1) AS casualty')
            ->selectRaw('SUM(consultations.refer_out_patient = 1) AS out_patient')
            ->selectRaw('SUM(consultations.sick_leave = 1) AS sick_leave')
            ->first();

        $referrals = [
            'radiology'   => (int) ($ref->radiology ?? 0),
            'lab'         => (int) ($ref->lab ?? 0),
            'specialist'  => (int) ($ref->specialist ?? 0),
            'casualty'    => (int) ($ref->casualty ?? 0),
            'out_patient' => (int) ($ref->out_patient ?? 0),
            'sick_leave'  => (int) ($ref->sick_leave ?? 0),
        ];
        $referrals['total_referrals'] = $referrals['radiology'] + $referrals['lab'] + $referrals['specialist'] + $referrals['casualty'] + $referrals['out_patient'];

        // All-time: Injury on duty
        $iodCount = (clone $base)->where('consultations.injury_on_duty', 1)->count();

        // All-time: diagnoses (top 15)
        $diagnosesRaw = (clone $base)->whereNotNull('consultations.diagnosis')->pluck('consultations.diagnosis')->all();
        $diagCounts   = [];
        foreach ($diagnosesRaw as $txt) {
            foreach (preg_split('/[,\;\|\n\/]+/u', (string) $txt) as $p) {
                $name = trim($p);
                if ($name === '' || mb_strlen($name) < 2) {
                    continue;
                }

                $norm              = mb_convert_case($name, MB_CASE_TITLE, 'UTF-8');
                $diagCounts[$norm] = ($diagCounts[$norm] ?? 0) + 1;
            }
        }
        arsort($diagCounts);
        $diagnoses = [];
        foreach (array_slice($diagCounts, 0, 15, true) as $k => $v) {
            $diagnoses[] = ['name' => $k, 'count' => (int) $v];
        }

        // Today: payment methods (via triages)
        $paymentMethods = (clone $base)
            ->join('triages', 'triages.consultation_id', '=', 'consultations.id')
            ->whereBetween($dateCol, [$start, $end])
            ->select('triages.payment_method', DB::raw('COUNT(*) AS count'))
            ->groupBy('triages.payment_method')
            ->orderByDesc('count')
            ->get()
            ->map(fn($r) => [
                'payment_method' => (string) ($r->payment_method ?? 'N/A'),
                'count'          => (int) $r->count,
            ])
            ->values()
            ->all();

        // Today: averages for vitals
        $avgToday  = (clone $base)->whereBetween($dateCol, [$start, $end]);
        $avgVitals = [
            'temperature' => (float) ($avgToday->avg('consultations.temperature') ?? 0),
            'systolic'    => (float) ($avgToday->avg('consultations.systolic') ?? 0),
            'diastolic'   => (float) ($avgToday->avg('consultations.diastolic') ?? 0),
        ];

        // Allowed clinics (badges)
        $allowedClinics = DB::table('clinics')
            ->whereIn('clinics.id', $clinicIds ?: [-1])
            ->select('clinics.id', 'clinics.name')
            ->orderBy('name')
            ->get();

        return response()->json([
            'day'            => $start->toDateString(),
            'nurseName'      => $user->name,
            'totalToday'     => $totalToday,
            'last7'          => $last7,
            'referrals'      => $referrals,
            'iodCount'       => $iodCount,
            'perClinic'      => $perClinic,
            'diagnoses'      => $diagnoses,
            'paymentMethods' => $paymentMethods,
            'avgVitals'      => $avgVitals,
            'allowedClinics' => $allowedClinics,
        ]);
    }
}
