<?php
namespace App\Http\Controllers;

use App\Models\ClinicMedicationStock;
use App\Models\Log;
use App\Models\NurseMedicationStock;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class NurseMedicationStockController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    public function nurseMedicationStock(User $user)
    {
        $stocks = NurseMedicationStock::with([
            'nurse',                        // Nurse details
            'clinic',                       // Clinic details
            'medicationBatch.medication',   // Medication details
            'medicationBatch.supplier',     // Supplier details
            'medicationBatch.centralStore', // Central store info
        ])
            ->where('nurse_id', $user->id) // ✅ filter by user id
            ->when(request('search'), function ($query, $search) {
                $query->whereHas('medicationBatch.medication', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->when(request('search_user'), function ($query, $searchUser) {
                $query->whereHas('nurse', function ($q) use ($searchUser) {
                    $q->where('name', 'like', "%{$searchUser}%");
                });
            })
            ->paginate(10)
            ->withQueryString(); // keep filters in pagination links

        return Inertia::render('Clinics/MedicationStocks/NurseMedicationStock', [
            'user'    => $user->only(['id', 'name', 'clinic_id']), // pass clean object
            'stocks'  => $stocks,
            'filters' => request()->only(['search', 'search_user']),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    public function handleTransferToNurse(Request $request, ClinicMedicationStock $stock)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1',
            'nurse_id' => 'required|exists:users,id',
        ]);

        $transferQuantity = $request->input('quantity');
        $nurseId          = $request->input('nurse_id');

        if ($transferQuantity > $stock->quantity) {
            return response()->json([
                'message' => 'Transfer quantity exceeds available stock.',
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Store old values for clinic stock log
            $oldClinicValues = $stock->getOriginal();

            // Decrease quantity on ClinicMedicationStock
            $stock->quantity -= $transferQuantity;
            $stock->save();

            // Store new values for clinic stock log
            $newClinicValues = $stock->getAttributes();

            // Find existing nurse stock for same medication batch and clinic
            $nurseStock = NurseMedicationStock::where('nurse_id', $nurseId)
                ->where('clinic_id', $stock->clinic_id)
                ->where('medication_batch_id', $stock->medication_batch_id)
                ->first();

            if ($nurseStock) {
                $oldNurseValues = $nurseStock->getOriginal();

                // Update existing nurse stock quantity
                $nurseStock->quantity += $transferQuantity;
                $nurseStock->save();

                $newNurseValues = $nurseStock->getAttributes();
            } else {
                $oldNurseValues = null;

                // Create new nurse stock record
                $nurseStock = NurseMedicationStock::create([
                    'nurse_id'            => $nurseId,
                    'clinic_id'           => $stock->clinic_id,
                    'medication_batch_id' => $stock->medication_batch_id,
                    'quantity'            => $transferQuantity,
                ]);

                $newNurseValues = $nurseStock->getAttributes();
            }

            // Log the clinic stock update
            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'updated',
                'loggable_type' => ClinicMedicationStock::class,
                'loggable_id'   => $stock->id,
                'description'   => 'Transferred stock quantity to nurse',
                'old_values'    => $oldClinicValues,
                'new_values'    => $newClinicValues,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->userAgent(),
            ]);

            // Log the nurse stock update/creation
            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => $oldNurseValues ? 'updated' : 'created',
                'loggable_type' => NurseMedicationStock::class,
                'loggable_id'   => $nurseStock->id,
                'description'   => 'Received stock quantity from clinic transfer',
                'old_values'    => $oldNurseValues,
                'new_values'    => $newNurseValues,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->userAgent(),
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Stock transferred successfully.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Failed to transfer stock.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function updateNurseMedQuantity(Request $request, NurseMedicationStock $nurseMedicationStock)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:0',
        ]);

        // Store old values for logging
        $oldValues = $nurseMedicationStock->getOriginal();

        // Update stock
        $nurseMedicationStock->update($validated);

        // Create log entry
        Log::create([
            'user_id'       => Auth::id(),
            'user_name'     => Auth::user()->name ?? null,
            'action'        => 'updated',
            'loggable_type' => NurseMedicationStock::class,
            'loggable_id'   => $nurseMedicationStock->id,
            'description'   => 'Updated nurse medication stock quantity',
            'old_values'    => $oldValues,
            'new_values'    => $nurseMedicationStock->getAttributes(),
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->userAgent(),
        ]);

        return response()->json([
            'message' => 'Stock updated successfully',
            'stock'   => $nurseMedicationStock,
        ], 200);
    }

    public function returnToClinic(Request $request, NurseMedicationStock $nurseMedicationStock)
    {
        $validated = $request->validate([
            'quantity'       => 'required|integer|min:1',
            'delete_if_zero' => 'sometimes|boolean',
            'reason'         => 'sometimes|nullable|string|max:500', // optional free-text reason
        ]);

        $qty          = (int) $validated['quantity'];
        $deleteIfZero = (bool) ($validated['delete_if_zero'] ?? false);

        DB::beginTransaction();

        try {
            // 1) Lock the nurse stock row (fresh copy under transaction)
            $nurseStock = NurseMedicationStock::query()
                ->whereKey($nurseMedicationStock->id)
                ->lockForUpdate()
                ->firstOrFail();

            if ($qty > $nurseStock->quantity) {
                DB::rollBack();
                return response()->json([
                    'message' => 'Return quantity exceeds nurse stock available.',
                ], 422);
            }

            // 2) Find (and lock) the corresponding clinic stock for the same batch + clinic
            $clinicStock = ClinicMedicationStock::query()
                ->where('clinic_id', $nurseStock->clinic_id)
                ->where('medication_batch_id', $nurseStock->medication_batch_id)
                ->lockForUpdate()
                ->first();

            // Create if missing (edge case)
            if (! $clinicStock) {
                $clinicStock = ClinicMedicationStock::create([
                    'clinic_id'           => $nurseStock->clinic_id,
                    'medication_batch_id' => $nurseStock->medication_batch_id,
                    'quantity'            => 0,
                ]);
                // Lock the newly created row (not strictly necessary, but consistent)
                $clinicStock = ClinicMedicationStock::query()
                    ->whereKey($clinicStock->id)
                    ->lockForUpdate()
                    ->first();
            }

            // 3) Snapshot old values for logs
            $oldNurseValues  = $nurseStock->getOriginal();
            $oldClinicValues = $clinicStock->getOriginal();

            // 4) Apply the return movement
            $nurseStock->quantity -= $qty;
            $nurseStock->save();

            $clinicStock->quantity += $qty;
            $clinicStock->save();

            // 5) Optionally delete nurse stock row if empty
            if ($deleteIfZero && $nurseStock->quantity === 0) {
                $nurseStock->delete();
            }

            // 6) Logs
            $descSuffix = isset($validated['reason']) && $validated['reason'] !== null
            ? ' Reason: ' . $validated['reason']
            : '';

            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'updated',
                'loggable_type' => NurseMedicationStock::class,
                'loggable_id'   => $nurseStock->id,
                'description'   => "Returned {$qty} unit(s) from nurse stock to clinic." . $descSuffix,
                'old_values'    => $oldNurseValues,
                'new_values'    => $nurseStock->fresh()?->getAttributes(), // if deleted, this may be null
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->userAgent(),
            ]);

            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'updated',
                'loggable_type' => ClinicMedicationStock::class,
                'loggable_id'   => $clinicStock->id,
                'description'   => "Received {$qty} unit(s) returned from nurse stock." . $descSuffix,
                'old_values'    => $oldClinicValues,
                'new_values'    => $clinicStock->getAttributes(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->userAgent(),
            ]);

            DB::commit();

            return response()->json([
                'message'      => 'Medication returned to clinic successfully.',
                'nurse_stock'  => $nurseStock->exists ? $nurseStock->only(['id', 'quantity', 'nurse_id', 'clinic_id', 'medication_batch_id']) : null,
                'clinic_stock' => $clinicStock->only(['id', 'quantity', 'clinic_id', 'medication_batch_id']),
            ], 200);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to return medication.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(NurseMedicationStock $nurseMedicationStock)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(NurseMedicationStock $nurseMedicationStock)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, NurseMedicationStock $nurseMedicationStock)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(NurseMedicationStock $nurseMedicationStock)
    {
        $nurseMedicationStock->delete();
        return redirect()->back()->with('success', 'Stock entry deleted');
    }

}
