<?php

namespace App\Http\Controllers;

use App\Models\Dispensation;
use App\Models\NurseMedicationStock;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class NurseReportsController extends Controller
{
    /**
     * Nurse's own stock take (Excel)
     */
    public function nurseMedicationStockTake(Request $request)
    {
        $user = auth()->user();

        // role could be enum-cast or string
        $roleVal = is_object($user->role) ? ($user->role->value ?? null) : ($user->role ?? null);
        abort_if($roleVal !== 'nurse', 403, 'Only nurses can download this stock sheet.');

        // Load clinic from the user model
        $user->load('clinic:id,name,address,city,email_address');
        $clinic = $user->clinic;

        $search      = $request->query('search');      // medication name / batch number
        $expiryFrom  = $request->query('expiry_from'); // YYYY-MM-DD
        $expiryTo    = $request->query('expiry_to');   // YYYY-MM-DD
        $includeZero = $request->boolean('include_zero', false);

        // Build dataset: strictly this nurse + (optionally) their clinic_id if present
        $query = NurseMedicationStock::query()
            ->with([
                'clinic:id,name,address,city,email_address',
                'medicationBatch.medication:id,name,dosage,form,unit',
            ])
            ->where('nurse_id', $user->id)
            ->when($clinic?->id, fn ($q) => $q->where('clinic_id', $clinic->id))
            ->when(!$includeZero, fn ($q) => $q->where('quantity', '>', 0))
            ->when($search, function ($q) use ($search) {
                $q->whereHas('medicationBatch.medication', fn ($m) =>
                    $m->where('name', 'like', "%{$search}%")
                )->orWhereHas('medicationBatch', fn ($b) =>
                    $b->where('batch_number', 'like', "%{$search}%")
                );
            })
            ->when($expiryFrom || $expiryTo, function ($q) use ($expiryFrom, $expiryTo) {
                if ($expiryFrom && $expiryTo) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereBetween('expiry_date', [$expiryFrom, $expiryTo])
                    );
                } elseif ($expiryFrom) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '>=', $expiryFrom)
                    );
                } else {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '<=', $expiryTo)
                    );
                }
            });

        $stocks = $query->get()
            ->sortBy(fn ($s) => sprintf(
                '%s|%s',
                mb_strtolower(optional(optional($s->medicationBatch)->medication)->name ?? ''),
                optional($s->medicationBatch)->expiry_date ?? '9999-12-31'
            ))
            ->values();

        $generatedAt = now()->format('Y-m-d H:i:s');

        $filters = [];
        if ($search)      $filters[] = "Search: {$search}";
        if ($expiryFrom)  $filters[] = "Expiry From: {$expiryFrom}";
        if ($expiryTo)    $filters[] = "Expiry To: {$expiryTo}";
        if ($includeZero) $filters[] = "Include Zero: yes";
        $filtersLine = implode(' | ', $filters);

        $clinicLine = $clinic
            ? trim($clinic->name . ($clinic->city ? " ({$clinic->city})" : ''))
            : '—';

        $fileName = sprintf(
            'Nurse_Stock_Take_%s_%s.xlsx',
            preg_replace('/\s+/', '_', $user->name ?? 'Nurse'),
            now()->format('Ymd_His')
        );

        return response()->streamDownload(function () use ($stocks, $user, $clinicLine, $generatedAt, $filtersLine) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // Sheet name
            $sheetName = $this->safeSheetName('Nurse Stock Take');
            $writer->getCurrentSheet()->setName($sheetName);

            // Header block
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['NURSE MEDICATION STOCK TAKE'],
                ['Generated At', $generatedAt],
                ['Nurse', $user->name . ($user->email ? " ({$user->email})" : '')],
                ['Clinic', $clinicLine],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);

            // Table header
            $headers = [
                'Medication',
                'Dosage',
                'Form',
                'Unit',
                'Batch #',
                'Expiry Date',
                'Manufacture Date',
                'Received Date',
                'Quantity',
            ];
            $writer->addRow(Row::fromValues($headers, $headerStyle));

            // Column widths (optional)
            $opts = $writer->getOptions();
            $opts->setColumnWidth(36, 1); // Medication
            $opts->setColumnWidth(14, 5); // Batch #
            $opts->setColumnWidth(14, 6); // Expiry

            // Rows
            $totalQty = 0;
            foreach ($stocks as $s) {
                $batch = $s->medicationBatch;
                $med   = optional($batch)->medication;

                $row = [
                    $med->name ?? '',
                    $med->dosage ?? '',
                    $med->form ?? '',
                    $med->unit ?? '',
                    $batch->batch_number ?? '',
                    optional($batch->expiry_date)->format('Y-m-d') ?? (is_string($batch->expiry_date ?? null) ? $batch->expiry_date : ''),
                    optional($batch->manufacture_date)->format('Y-m-d') ?? (is_string($batch->manufacture_date ?? null) ? $batch->manufacture_date : ''),
                    optional($batch->received_date)->format('Y-m-d') ?? (is_string($batch->received_date ?? null) ? $batch->received_date : ''),
                    (int) $s->quantity,
                ];

                $totalQty += (int) $s->quantity;
                $writer->addRow(Row::fromValues($row, $wrapStyle));
            }

            // Footer totals
            $writer->addRow(Row::fromValues([]));
            $writer->addRow(Row::fromValues(['Total Lines', $stocks->count()]));
            $writer->addRow(Row::fromValues(['Total Quantity', $totalQty]));

            // Medication summary
            if ($stocks->isNotEmpty()) {
                $writer->addRow(Row::fromValues([]));
                $writer->addRow(Row::fromValues(['Medication Summary'], $titleStyle));
                $writer->addRow(Row::fromValues(['Medication', 'Total Quantity'], $headerStyle));

                $byMed = $stocks
                    ->groupBy(fn ($s) => optional(optional($s->medicationBatch)->medication)->name ?? '(Unknown)')
                    ->map(fn ($group) => $group->sum('quantity'))
                    ->sortKeys();

                foreach ($byMed as $medName => $qty) {
                    $writer->addRow(Row::fromValues([$medName, (int) $qty]));
                }
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /**
     * Dispensations Report (Excel)
     * - Includes Diagnosis & ✓ if sick_leave on the linked consultation
     * - If dispensed_by is omitted, creates one sheet per dispenser + an "All Dispensations" sheet
     */
    public function dispensationsReport(Request $request)
    {
        $search      = $request->query('search');
        $startDate   = $request->query('start_date');
        $endDate     = $request->query('end_date');
        $patientId   = $request->query('patient_id');
        $dispensedBy = $request->query('dispensed_by'); // user id

        $query = Dispensation::query()
            ->with([
                'patient:id,first_name,surname',
                'dispensedBy:id,name',
                'consultation:id,diagnosis,sick_leave',
                'clinicMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
                'nurseMedicationStock.medicationBatch.medication:id,name,dosage,form,unit',
                'clinicMedicationStock.medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
                'nurseMedicationStock.medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
            ])
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('notes', 'like', "%{$search}%")
                        ->orWhereHas('patient', fn ($p) =>
                            $p->where('first_name', 'like', "%{$search}%")
                              ->orWhere('surname', 'like', "%{$search}%")
                        )
                        ->orWhereHas('clinicMedicationStock.medicationBatch', fn ($b) =>
                            $b->where('batch_number', 'like', "%{$search}%")
                        )
                        ->orWhereHas('nurseMedicationStock.medicationBatch', fn ($b) =>
                            $b->where('batch_number', 'like', "%{$search}%")
                        )
                        ->orWhereHas('clinicMedicationStock.medicationBatch.medication', fn ($m) =>
                            $m->where('name', 'like', "%{$search}%")
                        )
                        ->orWhereHas('nurseMedicationStock.medicationBatch.medication', fn ($m) =>
                            $m->where('name', 'like', "%{$search}%")
                        );
                });
            })
            ->when($startDate && $endDate, fn ($q) =>
                $q->whereBetween('dispensed_at', [$startDate.' 00:00:00', $endDate.' 23:59:59'])
            )
            ->when($startDate && !$endDate, fn ($q) =>
                $q->where('dispensed_at', '>=', $startDate.' 00:00:00')
            )
            ->when(!$startDate && $endDate, fn ($q) =>
                $q->where('dispensed_at', '<=', $endDate.' 23:59:59')
            )
            ->when($patientId, fn ($q) => $q->where('patient_id', $patientId))
            ->when($dispensedBy, fn ($q) => $q->where('dispensed_by', $dispensedBy))
            ->orderBy('dispensed_at');

        $rows = $query->get();

        // Build filters line to show in the header
        $generatedAt = now()->format('Y-m-d H:i:s');
        $filters     = [];
        if ($search)     $filters[] = "Search: {$search}";
        if ($startDate)  $filters[] = "From: {$startDate}";
        if ($endDate)    $filters[] = "To: {$endDate}";
        if ($patientId)  $filters[] = "Patient ID: {$patientId}";
        if ($dispensedBy)$filters[] = "Dispensed By: {$dispensedBy}";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Dispensations_%s.xlsx', now()->format('Ymd_His'));

        return response()->streamDownload(function () use ($rows, $generatedAt, $filtersLine, $dispensedBy) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // Group rows by dispenser (name) if we need multiple sheets
            $grouped = $rows->groupBy(fn ($d) => optional($d->dispensedBy)->name ?? 'Unknown');

            if ($dispensedBy) {
                // Single sheet only for the filtered user
                $writer->getCurrentSheet()->setName($this->safeSheetName('Dispensations'));
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['DISPENSATION REPORT'],
                    ['Generated At', $generatedAt],
                    $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                ]);
                $this->writeDispensationsTable($writer, $rows, $headerStyle, $wrapStyle);
                $this->writeDispensationsTotals($writer, $rows);
            } else {
                // Overall sheet first
                $writer->getCurrentSheet()->setName($this->safeSheetName('All Dispensations'));
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['DISPENSATION REPORT – ALL'],
                    ['Generated At', $generatedAt],
                    $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                ]);
                $this->writeDispensationsTable($writer, $rows, $headerStyle, $wrapStyle);
                $this->writeDispensationsTotals($writer, $rows);

                // One sheet per dispenser
                $used = ['All Dispensations' => true];
                foreach ($grouped as $dispenser => $dataset) {
                    $writer->addNewSheetAndMakeItCurrent();
                    $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($dispenser));
                    $writer->getCurrentSheet()->setName($sheetName);

                    $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                        ["DISPENSATION REPORT — {$dispenser}"],
                        ['Generated At', $generatedAt],
                        $filtersLine !== '' ? ['Filters', $filtersLine] : null,
                    ]);
                    $this->writeDispensationsTable($writer, $dataset, $headerStyle, $wrapStyle);
                    $this->writeDispensationsTotals($writer, $dataset);
                }
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /** Render the dispensations table. Includes Diagnosis + ✓ if sick_leave; widened columns for patient & dosage. */
    private function writeDispensationsTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle
    ): void {
        $headers = [
            'Date/Time',
            'Patient FirstName',
            'Patient Surname',
            'Medication',
            'Quantity',
            'Dosage/Route',
            'Dispensed By',
            'Diagnosis',
            'Sick Leave',
            'Dispense Notes',
        ];
        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Widen specific columns
        // (1-based indexes): 1=Date/Time, 2=FirstName, 3=Surname, 4=Medication, 5=Quantity, 6=Dosage, 7=By, 8=Diagnosis, 9=Sick, 10=Notes
        $opts = $writer->getOptions();
        $opts->setColumnWidth(22, 1); // Date/Time
        $opts->setColumnWidth(26, 2); // Patient FirstName (wider)
        $opts->setColumnWidth(26, 3); // Patient Surname  (wider)
        $opts->setColumnWidth(28, 4); // Medication
        $opts->setColumnWidth(28, 6); // Dosage/Route (wider)
        $opts->setColumnWidth(36, 8); // Diagnosis
        $opts->setColumnWidth(36, 10); // Dispense Notes

        foreach ($dataset as $d) {
            // Resolve medication name from either clinic or nurse stock path
            $medName   = $this->resolveMedicationName($d);
            $patient   = $d->patient;
            $dispenser = $d->dispensedBy;

            $diagnosis = optional($d->consultation)->diagnosis ?? '';
            $sickLeave = optional($d->consultation)->sick_leave ? '✓' : '';

            // Compact dosage/route string
            $dose = trim(collect([
                $d->route ?: null,
                $d->frequency ? "Freq: {$d->frequency}" : null,
                $d->duration ? "Dur: {$d->duration}" : null,
            ])->filter()->implode(' • '));

            $row = [
                optional($d->dispensed_at)->format('Y-m-d H:i') ?? (optional($d->created_at)->format('Y-m-d H:i') ?? ''),
                $patient->first_name ?? '',
                $patient->surname ?? '',
                $medName,
                (int) ($d->quantity ?? 0),
                $dose,
                $dispenser->name ?? '',
                $diagnosis,
                $sickLeave,
                (string) ($d->notes ?? ''),
            ];

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    private function writeDispensationsTotals(XLSXWriter $writer, $dataset): void
    {
        $writer->addRow(Row::fromValues([]));
        $writer->addRow(Row::fromValues(['Total Dispensations', $dataset->count()]));
        $writer->addRow(Row::fromValues(['Total Quantity', (int) $dataset->sum('quantity')]));
    }

    /** Try eager paths; fallback between clinic/nurse stock. */
    private function resolveMedicationName($dispensation): string
    {
        $paths = [
            'clinicMedicationStock.medicationBatch.medication.name',
            'nurseMedicationStock.medicationBatch.medication.name',
        ];
        foreach ($paths as $path) {
            $val = data_get($dispensation, $path);
            if (!empty($val)) {
                return (string) $val;
            }
        }
        return '';
    }

    /* --------------------- shared helpers --------------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') {
            $name = 'Sheet';
        }
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) {
                continue;
            }
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                // label/value
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        // spacer
        $writer->addRow(Row::fromValues([]));
    }
}
