<?php

namespace App\Http\Controllers;

use App\Models\NurseMedicationStock;
use App\Models\User;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class NurseStockTakeReportController extends Controller
{
    /**
     * Export a Nurse Stock Take workbook with one sheet per user.
     *
     * Query params (all optional unless noted):
     * - search:        string, matches medication name or batch number
     * - expiry_from:   YYYY-MM-DD
     * - expiry_to:     YYYY-MM-DD
     * - include_zero:  bool (default false) include zero-qty rows
     * - clinic_id:     int, limit to a single clinic
     * - nurse_id:      int, only this user’s sheet (still uses the same layout)
     * - include_empty: bool (default false) also include users who have no stock rows
     */
    public function allUsersStockTake(Request $request)
    {
        $search       = $request->query('search');
        $expiryFrom   = $request->query('expiry_from');
        $expiryTo     = $request->query('expiry_to');
        $includeZero  = $request->boolean('include_zero', false);
        $clinicId     = $request->query('clinic_id');
        $nurseId      = $request->query('nurse_id');
        $includeEmpty = $request->boolean('include_empty', false);

        // Base stock query
        $query = NurseMedicationStock::query()
            ->with([
                'nurse:id,name,email',
                'clinic:id,name,city',
                'medicationBatch.medication:id,name,dosage,form,unit',
                'medicationBatch:id,batch_number,expiry_date,manufacture_date,received_date,medication_id',
            ])
            ->when(!$includeZero, fn ($q) => $q->where('quantity', '>', 0))
            ->when($clinicId, fn ($q) => $q->where('clinic_id', $clinicId))
            ->when($nurseId, fn ($q) => $q->where('nurse_id', $nurseId))
            ->when($search, function ($q) use ($search) {
                $q->whereHas('medicationBatch.medication', fn ($m) =>
                    $m->where('name', 'like', "%{$search}%")
                )->orWhereHas('medicationBatch', fn ($b) =>
                    $b->where('batch_number', 'like', "%{$search}%")
                );
            })
            ->when($expiryFrom || $expiryTo, function ($q) use ($expiryFrom, $expiryTo) {
                if ($expiryFrom && $expiryTo) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereBetween('expiry_date', [$expiryFrom, $expiryTo])
                    );
                } elseif ($expiryFrom) {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '>=', $expiryFrom)
                    );
                } else {
                    $q->whereHas('medicationBatch', fn ($b) =>
                        $b->whereDate('expiry_date', '<=', $expiryTo)
                    );
                }
            });

        $rows = $query->get()
            ->sortBy(fn ($s) => sprintf(
                '%s|%s|%s',
                mb_strtolower(optional($s->nurse)->name ?? ('User#' . $s->nurse_id)),
                mb_strtolower(optional(optional($s->medicationBatch)->medication)->name ?? ''),
                optional($s->medicationBatch)->expiry_date ?? '9999-12-31'
            ))
            ->values();

        // Build list of users to produce sheets for:
        //  - If include_empty: every user (or you can scope to active staff if you have such a flag)
        //  - Else: only users that appear in $rows
        $usersToSheet = collect();

        if ($includeEmpty) {
            $usersToSheet = User::query()
                ->when($nurseId, fn ($q) => $q->where('id', $nurseId))
                ->get(['id', 'name', 'email']);
        } else {
            $userIds = $rows->pluck('nurse_id')->filter()->unique()->values();
            if ($userIds->isNotEmpty()) {
                $usersToSheet = User::whereIn('id', $userIds)->get(['id', 'name', 'email']);
            }
        }

        // Map user id -> display label and fallbacks
        $userDisplay = fn ($u) => trim(($u->name ?? ('User#' . $u->id)) . ($u->email ? " ({$u->email})" : ''));

        // Filters line
        $generatedAt = now()->format('Y-m-d H:i:s');
        $filters     = [];
        if ($search)      $filters[] = "Search: {$search}";
        if ($expiryFrom)  $filters[] = "Expiry From: {$expiryFrom}";
        if ($expiryTo)    $filters[] = "Expiry To: {$expiryTo}";
        if ($clinicId)    $filters[] = "Clinic ID: {$clinicId}";
        if ($nurseId)     $filters[] = "User ID: {$nurseId}";
        if ($includeZero) $filters[] = "Include Zero: yes";
        if ($includeEmpty)$filters[] = "Include Empty Users: yes";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf(
            $nurseId
                ? 'Nurse_Stock_Take_%s_%s.xlsx'
                : 'Nurse_Stock_Take_All_Users_%s.xlsx',
            $nurseId
                ? preg_replace('/\s+/', '_', optional($usersToSheet->first())->name ?? ('User_' . $nurseId))
                : now()->format('Ymd_His'),
            $nurseId ? now()->format('Ymd_His') : ''
        );

        return response()->streamDownload(function () use ($rows, $usersToSheet, $userDisplay, $generatedAt, $filtersLine) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            /* ========== Overview sheet ========== */
            $writer->getCurrentSheet()->setName($this->safeSheetName('All Nurses — Summary'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['NURSE MEDICATION STOCK TAKE — ALL USERS'],
                ['Generated At', $generatedAt],
                $filtersLine !== '' ? ['Filters', $filtersLine] : null,
            ]);

            $writer->addRow(Row::fromValues(['User', 'Total Lines', 'Total Quantity', 'Clinics (distinct)'], $headerStyle));

            $overallLines = 0;
            $overallQty   = 0;

            // Pre-group by nurse_id for efficiency
            $byUser = $rows->groupBy('nurse_id');

            foreach ($usersToSheet as $u) {
                $set     = $byUser->get($u->id, collect());
                $lines   = $set->count();
                $qty     = (int) $set->sum('quantity');
                $clinics = $set->pluck('clinic.name')->filter()->unique()->values()->count();

                $overallLines += $lines;
                $overallQty   += $qty;

                $writer->addRow(Row::fromValues([
                    $userDisplay($u),
                    $lines,
                    $qty,
                    $clinics,
                ]));
            }

            // Totals
            $writer->addRow(Row::fromValues([]));
            $writer->addRow(Row::fromValues(['ALL USERS LINES', $overallLines]));
            $writer->addRow(Row::fromValues(['ALL USERS QUANTITY', $overallQty]));

            /* ========== One sheet per user ========== */
            $used = ['All Nurses — Summary' => true];

            foreach ($usersToSheet as $u) {
                $writer->addNewSheetAndMakeItCurrent();
                $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($u->name ?: ('User ' . $u->id)));
                $writer->getCurrentSheet()->setName($sheetName);

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['NURSE MEDICATION STOCK TAKE'],
                    ['User', $userDisplay($u)],
                    ['Generated At', $generatedAt],
                ]);

                // Table header (includes Clinic column since a user may have multiple clinic rows)
                $headers = [
                    'Medication',
                    'Dosage',
                    'Form',
                    'Unit',
                    'Batch #',
                    'Expiry Date',
                    'Manufacture Date',
                    'Received Date',
                    'Quantity',
                    'Clinic',
                ];
                $writer->addRow(Row::fromValues($headers, $headerStyle));

                // Column widths
                $opts = $writer->getOptions();
                $opts->setColumnWidth(36, 1);  // Medication
                $opts->setColumnWidth(14, 5);  // Batch #
                $opts->setColumnWidth(14, 6);  // Expiry
                $opts->setColumnWidth(24, 10); // Clinic

                $userRows  = $byUser->get($u->id, collect());
                $totalQty  = 0;

                foreach ($userRows as $s) {
                    // use data_get/optional for full null safety
                    $batch   = $s->medicationBatch; // may be null
                    $row = [
                        // Medication fields
                        data_get($batch, 'medication.name', ''),
                        data_get($batch, 'medication.dosage', ''),
                        data_get($batch, 'medication.form', ''),
                        data_get($batch, 'medication.unit', ''),
                        // Batch number
                        data_get($batch, 'batch_number', ''),
                        // Dates — safe formatter handles Carbon|string|null
                        $this->fmtDateCell(data_get($batch, 'expiry_date')),
                        $this->fmtDateCell(data_get($batch, 'manufacture_date')),
                        $this->fmtDateCell(data_get($batch, 'received_date')),
                        // Quantity & clinic
                        (int) ($s->quantity ?? 0),
                        $this->clinicLabel($s),
                    ];

                    $totalQty += (int) ($s->quantity ?? 0);
                    $writer->addRow(Row::fromValues($row, $wrapStyle));
                }

                // Footer totals
                $writer->addRow(Row::fromValues([]));
                $writer->addRow(Row::fromValues(['Total Lines', $userRows->count()]));
                $writer->addRow(Row::fromValues(['Total Quantity', $totalQty]));

                // Medication summary for this user
                if ($userRows->isNotEmpty()) {
                    $writer->addRow(Row::fromValues([]));
                    $writer->addRow(Row::fromValues(['Medication Summary'], $titleStyle));
                    $writer->addRow(Row::fromValues(['Medication', 'Total Quantity'], $headerStyle));

                    $byMed = $userRows
                        ->groupBy(fn ($s) => data_get($s, 'medicationBatch.medication.name', '(Unknown)'))
                        ->map(fn ($group) => $group->sum('quantity'))
                        ->sortKeys();

                    foreach ($byMed as $medName => $qty) {
                        $writer->addRow(Row::fromValues([$medName, (int) $qty]));
                    }
                }
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /* ---------------- helpers ---------------- */

    private function clinicLabel($stock): string
    {
        $c = $stock->clinic;
        if (! $c) return '';
        $name = (string) ($c->name ?? '');
        $city = (string) ($c->city ?? '');
        return trim($name . ($city ? " ({$city})" : ''));
    }

    /** Format Carbon|string|null into Y-m-d or empty string. */
    private function fmtDateCell($val): string
    {
        if ($val instanceof \DateTimeInterface) {
            return $val->format('Y-m-d');
        }
        if (is_string($val)) {
            return $val; // assume already Y-m-d-ish
        }
        return '';
    }

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
