<?php
namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Consultation;
use App\Models\Log;
use App\Models\Patient;
use App\Models\Triage;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log as LaravelLog;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class PatientController extends Controller
{
    

    public function index(Request $request)
    {
        $search                = (string) $request->input('search', '');
        $gender                = $request->input('gender');
        $companyId             = $request->input('company_id');
        $maritalStatus         = $request->input('marital_status');
        $ageMin                = $request->input('age_min');
        $ageMax                = $request->input('age_max');
        $isChronicPatientInput = $request->input('is_chronic_patient'); // NEW filter

        // ── Normalize search forms (handled case/spacing/hyphens) ──────────────
        $searchTrim  = trim(preg_replace('/\s+/u', ' ', $search));
        $normIdEq    = preg_replace('/[\s-]+/', '', mb_strtolower($searchTrim)); // for id_number (case-insensitive)
        $normEmpEq   = preg_replace('/[\s-]+/', '', mb_strtoupper($searchTrim)); // for employee_number (often letters)
        $normIdLike  = '%' . $normIdEq . '%';
        $normEmpLike = '%' . $normEmpEq . '%';

        // Tokenize for full-name intent ("first last" or "last first" etc.)
        $tokens = array_values(array_filter(explode(' ', $searchTrim), fn($t) => $t !== ''));

        $patients = Patient::query()
            ->with(['company:id,name'])
            ->addSelect('patients.*')
            ->withCount('dependents')
            ->addSelect([
                'latest_consultation_id' => Consultation::select('id')
                    ->whereColumn('consultations.patient_id', 'patients.id')
                    ->orderByDesc('consultation_date')
                    ->limit(1),
            ])
            ->whereNull('parent_patient_id')

        // ── ROBUST SEARCH (normalized exacts + full-name + normalized partials) ──
            ->when($searchTrim !== '', function ($query) use ($searchTrim, $tokens, $normIdEq, $normEmpEq, $normIdLike, $normEmpLike) {
                $query->where(function ($q) use ($searchTrim, $tokens, $normIdEq, $normEmpEq, $normIdLike, $normEmpLike) {
                    // 1) Exact National ID (ignoring hyphens/spaces)
                    $q->whereRaw("LOWER(REPLACE(REPLACE(id_number, ' ', ''), '-', '')) = ?", [$normIdEq])

                    // 2) OR Exact Employee Number (ignoring hyphens/spaces; case-insensitive)
                        ->orWhereRaw("UPPER(REPLACE(REPLACE(employee_number, ' ', ''), '-', '')) = ?", [$normEmpEq])

                    // 3) OR Full-name intent ("first last" or "last first"; supports 2–3 tokens)
                        ->orWhere(function ($qq) use ($tokens) {
                            if (count($tokens) >= 2) {
                                [$a, $b, $c] = [$tokens[0] ?? null, $tokens[1] ?? null, $tokens[2] ?? null];

                                $qq->where(function ($qqq) use ($a, $b, $c) {
                                    // first ... last
                                    $qqq->where('first_name', 'like', "%{$a}%")
                                        ->when($c,
                                            fn($qqqq) => $qqqq->where('surname', 'like', "%{$c}%"),
                                            fn($qqqq) => $qqqq->where('surname', 'like', "%{$b}%")
                                        );
                                })
                                    ->orWhere(function ($qqq) use ($a, $b, $c) {
                                        // last ... first
                                        $qqq->where('surname', 'like', "%{$a}%")
                                            ->when($c,
                                                fn($qqqq) => $qqqq->where('first_name', 'like', "%{$c}%"),
                                                fn($qqqq) => $qqqq->where('first_name', 'like', "%{$b}%")
                                            );
                                    })
                                    ->orWhereRaw(
                                        "CONCAT_WS(' ', first_name, middle_name, surname) LIKE ?",
                                        ['%' . implode(' ', array_filter([$a, $b, $c])) . '%']
                                    );
                            }
                        })

                    // 4) OR Broad fallback (partials; includes normalized partials)
                        ->orWhere('first_name', 'like', "%{$searchTrim}%")
                        ->orWhere('middle_name', 'like', "%{$searchTrim}%")
                        ->orWhere('surname', 'like', "%{$searchTrim}%")
                        ->orWhere('employee_number', 'like', "%{$searchTrim}%")
                        ->orWhere('id_number', 'like', "%{$searchTrim}%")
                        ->orWhereRaw("CONCAT_WS(' ', first_name, middle_name, surname) LIKE ?", ["%{$searchTrim}%"])
                    // normalized partials, so "63-2304" or "632304" both work
                        ->orWhereRaw("LOWER(REPLACE(REPLACE(id_number, ' ', ''), '-', '')) LIKE ?", [$normIdLike])
                        ->orWhereRaw("UPPER(REPLACE(REPLACE(employee_number, ' ', ''), '-', '')) LIKE ?", [$normEmpLike]);
                });
            })

        // Other filters (unchanged)
            ->when($gender, fn($q) => $q->where('gender', $gender))
            ->when($companyId, fn($q) => $q->where('company_id', $companyId))
            ->when($maritalStatus, fn($q) => $q->where('marital_status', $maritalStatus))
            ->when(($ageMin || $ageMax), function ($q) use ($ageMin, $ageMax) {
                $now = Carbon::now();
                if ($ageMin) {
                    $q->whereDate('date_of_birth', '<=', $now->copy()->subYears($ageMin));
                }
                if ($ageMax) {
                    $q->whereDate('date_of_birth', '>=', $now->copy()->subYears($ageMax + 1)->addDay());
                }
            })

        // NEW: chronic filter (accepts 1/0, true/false, 'true'/'false')
            ->when($isChronicPatientInput !== null && $isChronicPatientInput !== '', function ($q) use ($isChronicPatientInput) {
                $bool = filter_var($isChronicPatientInput, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                if ($bool !== null) {
                    $q->where('is_chronic_patient', $bool);
                }
            })

            ->orderByDesc('updated_at')
            ->orderByDesc('created_at')
            ->paginate(15)
            ->withQueryString();

        // ── Stats (unchanged + chronic) ─────────────────────────────────────────
        $employeesTotal  = Patient::whereNull('parent_patient_id')->count();
        $dependentsTotal = Patient::whereNotNull('parent_patient_id')->count();
        $chronicTotal    = Patient::whereNull('parent_patient_id')->where('is_chronic_patient', true)->count(); // NEW stat

        $genderCounts = Patient::whereNull('parent_patient_id')
            ->select('gender', DB::raw('COUNT(*) as count'))
            ->groupBy('gender')
            ->pluck('count', 'gender');

        $male    = (int) ($genderCounts['Male'] ?? 0);
        $female  = (int) ($genderCounts['Female'] ?? 0);
        $unknown = max(0, $employeesTotal - ($male + $female));

        return inertia('Patients/Index', [
            'patients'  => $patients,
            'filters'   => $request->only('search', 'gender', 'company_id', 'marital_status', 'age_min', 'age_max', 'is_chronic_patient'), // include new filter
            'companies' => Company::select('id', 'name')->get(),
            'stats'     => [
                'employees_total'  => $employeesTotal,
                'dependents_total' => $dependentsTotal,
                'chronic_total'    => $chronicTotal, // NEW
                'gender'           => [
                    'male'    => $male,
                    'female'  => $female,
                    'unknown' => $unknown,
                ],
            ],
        ]);
    }

    public function stats()
    {
        $employeeCount  = Patient::whereNull('parent_patient_id')->count();
        $dependentCount = Patient::whereNotNull('parent_patient_id')->count();

        return response()->json([
            'employees'  => $employeeCount,
            'dependents' => $dependentCount,
            'total'      => $employeeCount + $dependentCount,
        ]);
    }

    public function dependentsIndex(Request $request)
    {
        $search                = trim((string) $request->input('search', ''));
        $gender                = $request->input('gender');
        $relationship          = $request->input('relationship');
        $ageMin                = $request->input('ageMin');
        $ageMax                = $request->input('ageMax');
        $isChronicPatientInput = $request->input('is_chronic_patient'); // NEW filter

        $dependents = Patient::query()
            ->with('parent')
            ->select('patients.*')
            ->addSelect([
                'latest_consultation_id' => Consultation::select('id')
                    ->whereColumn('consultations.patient_id', 'patients.id')
                    ->orderByDesc('consultation_date')
                    ->limit(1),
            ])
            ->whereNotNull('parent_patient_id')
            ->when($search !== '', function ($query) use ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('first_name', 'like', "%{$search}%")
                        ->orWhere('middle_name', 'like', "%{$search}%")
                        ->orWhere('surname', 'like', "%{$search}%")
                        ->orWhereHas('parent', function ($sub) use ($search) {
                            $sub->where('first_name', 'like', "%{$search}%")
                                ->orWhere('middle_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%");
                        });
                });
            })
            ->when($gender, fn($q) => $q->where('gender', $gender))
            ->when($relationship, fn($q) => $q->where('relationship', $relationship))
            ->when(($ageMin || $ageMax), function ($q) use ($ageMin, $ageMax) {
                if ($ageMin) {
                    $q->where('date_of_birth', '<=', Carbon::now()->subYears((int) $ageMin));
                }
                if ($ageMax) {
                    $q->where('date_of_birth', '>=', Carbon::now()->subYears(((int) $ageMax) + 1)->addDay());
                }
            })
        // NEW: chronic filter
            ->when($isChronicPatientInput !== null && $isChronicPatientInput !== '', function ($q) use ($isChronicPatientInput) {
                $bool = filter_var($isChronicPatientInput, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                if ($bool !== null) {
                    $q->where('is_chronic_patient', $bool);
                }
            })
            ->orderByDesc('updated_at')
            ->orderByDesc('created_at')
            ->paginate(15)
            ->withQueryString();

        return inertia('Dependents/Index', [
            'dependents' => $dependents,
            'filters'    => $request->only('search', 'gender', 'relationship', 'ageMin', 'ageMax', 'is_chronic_patient'), // include new filter
        ]);
    }

    public function show($id)
    {
        $patient = Patient::with([
            'company', 'dependents', 'parent',
            'latestConsultation.doctor', 'latestConsultation.clinic', 'latestConsultation.patient',
        ])->findOrFail($id);

        return inertia('Patients/Show', [
            'patient' => $patient,
        ]);
    }

    public function showDependent($id)
    {
        $dependent = Patient::with([
            'parent', 'company',
            'latestConsultation.doctor', 'latestConsultation.clinic', 'latestConsultation.patient',
        ])->findOrFail($id);

        if (! $dependent->parent_patient_id) {
            abort(404);
        }

        return inertia('Dependents/Show', [
            'dependent' => $dependent,
        ]);
    }

    public function edit($id)
    {
        $patient = Patient::with(['company', 'parent'])->findOrFail($id);

        return inertia('Patients/Edit', [
            'patient'        => $patient,
            'companies'      => Company::all(),
            'relationships'  => [
                'Father', 'Mother', 'Brother', 'Sister', 'Son', 'Daughter', 'Spouse',
                'Uncle', 'Aunt', 'Cousin', 'Extended Family', 'Other',
            ],
            // from session (set by middleware)
            'missing_fields' => session('missing_fields', []),
            'flash'          => [
                'warning' => session('warning'),
            ],
        ]);
    }

    public function editDependent($id)
    {
        $dependent = Patient::with(['parent', 'company'])->findOrFail($id);

        if (! $dependent->parent_patient_id) {
            return redirect()->route('patients.edit', $dependent->id);
        }

        return inertia('Dependents/Edit', [
            'dependent'     => $dependent,
            'parents'       => Patient::whereNull('parent_patient_id')->get(),
            'relationships' => ['Spouse', 'Child', 'Parent', 'Sibling', 'Other'],
            'companies'     => Company::all(),
        ]);
    }

    public function create()
    {
        return inertia('Patients/Create', [
            'companies' => Company::all(),
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name'                 => 'required|string|max:255',
            'middle_name'                => 'nullable|string|max:255',
            'surname'                    => 'required|string|max:255',
            'employee_number'            => [
                'nullable', 'string', 'max:255',
                Rule::unique('patients', 'employee_number')
                    ->where(function ($q) use ($request) {
                        $companyId = $request->input('company_id');
                        return $q->where('company_id', $companyId ?: null);
                    }),
            ],
            'company_id'                 => 'nullable|exists:companies,id',
            'date_of_birth'              => 'nullable|date',
            'email'                      => 'nullable|email|max:255',
            'gender'                     => 'nullable|in:Male,Female,Other',
            'id_number'                  => 'nullable|string|max:255|unique:patients,id_number',
            'medical_aid_number'         => 'nullable|string|max:255',
            'medical_aid_provider'       => 'nullable|string|max:255',
            'phone'                      => 'required|string|max:20',
            'emergency_contact_name'     => 'nullable|string|max:255',
            'emergency_contact_relation' => 'nullable|string|max:255',
            'emergency_contact_phone'    => 'nullable|string|max:20',
            'allergies'                  => 'nullable|string',
            'is_smoker'                  => 'nullable|boolean',
            'occupation'                 => 'nullable|string|max:255',
            'home_address'               => 'nullable|string|max:1000',
            'work_area'                  => 'nullable|string|max:255',
            'suburb'                     => 'nullable|string|max:255',
            'marital_status'             => 'nullable|string|max:255',
            'is_chronic_patient'         => 'nullable|boolean', // NEW
        ]);

        $normalizeName = function (?string $v): ?string {
            if ($v === null) {
                return null;
            }
            $v = preg_replace('/\s+/u', ' ', trim($v));
            return Str::title($v);
        };

        $validator->after(function ($v) use ($request, $normalizeName) {
            $first     = $normalizeName($request->input('first_name'));
            $last      = $normalizeName($request->input('surname'));
            $dobRaw    = $request->input('date_of_birth');
            $phone     = trim((string) $request->input('phone', ''));
            $idno      = trim((string) $request->input('id_number', ''));
            $empNo     = trim((string) $request->input('employee_number', ''));
            $companyId = $request->input('company_id');

            $query = Patient::query();

            if ($idno !== '') {
                $query->orWhere('id_number', $idno);
            }

            if ($empNo !== '') {
                $query->orWhere(function ($q) use ($empNo, $companyId) {
                    $q->where('employee_number', $empNo)
                        ->where(function ($qq) use ($companyId) {
                            if ($companyId !== null && $companyId !== '') {
                                $qq->where('company_id', $companyId);
                            } else {
                                $qq->whereNull('company_id');
                            }
                        });
                });
            }

            if ($first && $last) {
                $query->orWhere(function ($q) use ($first, $last, $dobRaw, $phone) {
                    $q->where('first_name', $first)
                        ->where('surname', $last)
                        ->when(! empty($dobRaw), function ($qq) use ($dobRaw) {
                            try {
                                $dob = Carbon::parse($dobRaw)->toDateString();
                                $qq->whereDate('date_of_birth', $dob);
                            } catch (\Throwable $e) {}
                        }, function ($qq) use ($phone) {
                            if ($phone !== '') {
                                $qq->where('phone', $phone);
                            }
                        });
                });
            }

            $matches = $query
                ->select(['id', 'first_name', 'surname', 'date_of_birth', 'id_number', 'employee_number', 'phone', 'company_id'])
                ->limit(5)
                ->get();

            if ($matches->count() > 0) {
                $v->errors()->add(
                    'first_name',
                    'Potential duplicate detected. A patient with the same identifiers already exists.'
                );

                $payload = $matches->map(function ($p) {
                    return [
                        'id'              => $p->id,
                        'full_name'       => trim($p->first_name . ' ' . $p->surname),
                        'date_of_birth'   => $p->date_of_birth ? $p->date_of_birth->format('Y-m-d') : null,
                        'id_number'       => $p->id_number,
                        'employee_number' => $p->employee_number,
                        'phone'           => $p->phone,
                        'company_id'      => $p->company_id,
                    ];
                })->values()->all();

                $v->errors()->add('duplicate_candidates', json_encode($payload));
            }
        });

        $validated = $validator->validate();

        $validated['first_name']  = $normalizeName($validated['first_name']);
        $validated['middle_name'] = $normalizeName($validated['middle_name'] ?? null);
        $validated['surname']     = $normalizeName($validated['surname']);

        $patient = Patient::create($validated);

        $redirectToDependent = $request->boolean('redirect_to_dependent');

        if ($redirectToDependent) {
            return redirect()
                ->route('dependents.create', ['parent_id' => $patient->id])
                ->with('success', 'Patient created. Proceed to add dependent.');
        }

        return redirect()
            ->route('triages.create', ['patient_id' => $patient->id])
            ->with('success', 'Patient created successfully. Proceed to create triage.');
    }

    public function createDependent(Request $request)
    {
        $parents = Patient::with(['company:id,name'])
            ->whereNull('parent_patient_id')
            ->select(
                'id', 'first_name', 'surname', 'company_id',
                'home_address', 'work_area', 'suburb', 'phone',
                'emergency_contact_name', 'emergency_contact_relation', 'emergency_contact_phone'
            )
            ->orderBy('surname')
            ->get()
            ->map(function ($p) {
                return [
                    'id'                         => $p->id,
                    'first_name'                 => $p->first_name,
                    'surname'                    => $p->surname,
                    'company'                    => $p->company ? [
                        'id'   => $p->company->id,
                        'name' => $p->company->name,
                    ] : null,
                    'home_address'               => $p->home_address,
                    'work_area'                  => $p->work_area,
                    'suburb'                     => $p->suburb,
                    'phone'                      => $p->phone,
                    'emergency_contact_name'     => $p->emergency_contact_name,
                    'emergency_contact_relation' => $p->emergency_contact_relation,
                    'emergency_contact_phone'    => $p->emergency_contact_phone,
                ];
            });

        return inertia('Dependents/Create', [
            'parents'       => $parents,
            'relationships' => [
                'Spouse', 'Domestic Partner', 'Fiancé(e)', 'Ex-Spouse',
                'Biological Child', 'Stepchild', 'Adopted Child', 'Foster Child', 'Ward / Legal Ward', 'Grandchild',
                'Parent', 'Step-Parent', 'Parent-in-law', 'Grandparent',
                'Brother', 'Sister', 'Half-Sibling', 'Step-Sibling', 'Sibling-in-law',
                'Aunt/Uncle', 'Niece/Nephew', 'Cousin',
                'Guardian', 'Caregiver', 'Other',
            ],
            'parent_id'     => $request->query('parent_id'),
        ]);
    }

    public function storeDependent(Request $request)
    {
        $validated = $request->validate([
            'first_name'                 => 'required|string|max:255',
            'middle_name'                => 'nullable|string|max:255',
            'surname'                    => 'required|string|max:255',
            'parent_patient_id'          => 'required|exists:patients,id',
            'relationship'               => 'required|string|max:255',
            'date_of_birth'              => 'nullable|date',
            'email'                      => 'nullable|email|max:255',
            'gender'                     => 'nullable|in:Male,Female,Other',
            'id_number'                  => 'nullable|string|max:255',
            'medical_aid_number'         => 'nullable|string|max:255',
            'medical_aid_provider'       => 'nullable|string|max:255',
            'phone'                      => 'required|string|max:20',
            'emergency_contact_name'     => 'nullable|string|max:255',
            'emergency_contact_relation' => 'nullable|string|max:255',
            'emergency_contact_phone'    => 'nullable|string|max:20',
            'allergies'                  => 'nullable|string',
            'is_smoker'                  => 'nullable|boolean',
            'occupation'                 => 'nullable|string|max:255',
            'home_address'               => 'nullable|string|max:1000',
            'work_area'                  => 'nullable|string|max:255',
            'suburb'                     => 'nullable|string|max:255',
            'marital_status'             => 'nullable|string|max:255',
            'is_chronic_patient'         => 'nullable|boolean', // NEW
        ]);

        $normalize = function (?string $v): ?string {
            if ($v === null) {
                return null;
            }
            $v = preg_replace('/\s+/u', ' ', trim($v));
            return Str::title($v);
        };
        $validated['first_name']  = $normalize($validated['first_name']);
        $validated['middle_name'] = $normalize($validated['middle_name'] ?? null);
        $validated['surname']     = $normalize($validated['surname']);

        $dependent = Patient::create($validated);

        return redirect()
            ->route('triages.create', ['patient_id' => $dependent->id])
            ->with('success', 'Dependent created successfully. Proceed to create triage.');
    }

    public function update(Request $request, $id)
    {
        $patient = Patient::findOrFail($id);

        $validated = $request->validate([
            'first_name'                 => 'required|string|max:255',
            'middle_name'                => 'nullable|string|max:255',
            'surname'                    => 'required|string|max:255',
            'employee_number'            => 'nullable|string|max:255',
            'company_id'                 => 'nullable|exists:companies,id',
            'date_of_birth'              => 'nullable|date',
            'email'                      => 'nullable|email|max:255',
            'gender'                     => 'nullable|in:Male,Female,Other',
            'id_number'                  => 'nullable|string|max:255',
            'medical_aid_number'         => 'nullable|string|max:255',
            'medical_aid_provider'       => 'nullable|string|max:255',
            'phone'                      => 'required|string|max:20',
            'emergency_contact_name'     => 'nullable|string|max:255',
            'emergency_contact_relation' => 'nullable|string|max:255',
            'emergency_contact_phone'    => 'nullable|string|max:20',
            'allergies'                  => 'nullable|string',
            'is_smoker'                  => 'nullable|boolean',
            'occupation'                 => 'nullable|string|max:255',
            'home_address'               => 'nullable|string|max:1000',
            'work_area'                  => 'nullable|string|max:255',
            'suburb'                     => 'nullable|string|max:255',
            'marital_status'             => 'nullable|string|max:255',
            'is_chronic_patient'         => 'nullable|boolean', // NEW
        ]);

        $patient->update($validated);

        $return = $request->query('return');
        if ($return && filter_var($return, FILTER_VALIDATE_URL)) {
            return redirect($return)->with('success', 'Patient updated successfully.');
        }

        return redirect()->route('patients.show', $patient->id)
            ->with('success', 'Patient updated successfully.');
    }

    public function updateDependent(Request $request, $id)
    {
        $dependent = Patient::findOrFail($id);

        $validated = $request->validate([
            'first_name'                 => 'required|string|max:255',
            'middle_name'                => 'nullable|string|max:255',
            'surname'                    => 'required|string|max:255',
            'parent_patient_id'          => 'required|exists:patients,id',
            'relationship'               => 'required|string|max:255',
            'date_of_birth'              => 'nullable|date',
            'email'                      => 'nullable|email|max:255',
            'gender'                     => 'nullable|in:Male,Female,Other',
            'id_number'                  => 'nullable|string|max:255',
            'medical_aid_number'         => 'nullable|string|max:255',
            'medical_aid_provider'       => 'nullable|string|max:255',
            'phone'                      => 'required|string|max:20',
            'emergency_contact_name'     => 'nullable|string|max:255',
            'emergency_contact_relation' => 'nullable|string|max:255',
            'emergency_contact_phone'    => 'nullable|string|max:20',
            'allergies'                  => 'nullable|string',
            'is_smoker'                  => 'nullable|boolean',
            'occupation'                 => 'nullable|string|max:255',
            'home_address'               => 'nullable|string|max:1000',
            'work_area'                  => 'nullable|string|max:255',
            'suburb'                     => 'nullable|string|max:255',
            'marital_status'             => 'nullable|string|max:255',
            'is_chronic_patient'         => 'nullable|boolean', // NEW
        ]);

        $dependent->update($validated);

        return redirect()->route('dependents.show', $dependent->id)
            ->with('success', 'Dependent updated successfully.');
    }

    public function searchPage()
    {
        $user = Auth::user();

        $triages = Triage::with(['patient', 'nurse', 'consultor'])
            ->where('clinic_id', $user->clinic_id)
            ->where('is_active', true)
            ->whereDate('created_at', today())
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        return inertia('Patients/SearchPage', [
            'triages' => $triages,
        ]);
    }

    public function currentDayTriages()
    {
        $user = Auth::user();

        $triages = Triage::with(['patient', 'nurse', 'consultor'])
            ->where('clinic_id', $user->clinic_id)
            ->where('is_active', true)
            ->whereDate('created_at', today())
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        return response()->json([
            'triages' => $triages,
        ]);
    }

    // public function search(Request $request)
    // {
    //     $search = $request->input('query');

    //     $patients = Patient::with(['company', 'parent', 'dependents'])
    //         ->where(function ($q) use ($search) {
    //             $q->where('first_name', 'like', "%{$search}%")
    //                 ->orWhere('surname', 'like', "%{$search}%")
    //                 ->orWhere('id_number', 'like', "%{$search}%");
    //         })
    //         ->limit(40)
    //         ->get();

    //     return response()->json($patients);
    // }

    public function search(Request $request)
    {
        $raw   = (string) $request->input('query', '');
        $query = trim(preg_replace('/\s+/u', ' ', $raw));

        if ($query === '') {
            return response()->json([]);
        }

        // Eager-loads used everywhere
        $with = ['company', 'parent', 'dependents'];

        // Normalized forms (remove spaces & hyphens; unify case)
        $normalizedId   = preg_replace('/[\s-]+/', '', mb_strtolower($query));
        $normalizedEmp  = preg_replace('/[\s-]+/', '', mb_strtoupper($query)); // employee numbers often have letters
        $normalizedLike = '%' . $normalizedEmp . '%';

        // --- 1) Exact National ID (ignoring hyphens/spaces) --------------------
        $byNatId = \App\Models\Patient::with($with)
            ->whereRaw("LOWER(REPLACE(REPLACE(id_number, ' ', ''), '-', '')) = ?", [$normalizedId])
            ->limit(10)
            ->get();

        if ($byNatId->isNotEmpty()) {
            return response()->json($byNatId);
        }

        // --- 2) Exact Employee Number (ignoring hyphens/spaces; case-insensitive) ----
        $byEmpExact = \App\Models\Patient::with($with)
            ->whereRaw("UPPER(REPLACE(REPLACE(employee_number, ' ', ''), '-', '')) = ?", [$normalizedEmp])
            ->limit(10)
            ->get();

        if ($byEmpExact->isNotEmpty()) {
            return response()->json($byEmpExact);
        }

        // --- 3) Full-name intent ("first last" or "last first"; supports 2–3 tokens) --
        $tokens = array_values(array_filter(explode(' ', $query), fn($t) => $t !== ''));
        if (count($tokens) >= 2) {
            [$a, $b, $c] = [$tokens[0] ?? null, $tokens[1] ?? null, $tokens[2] ?? null];

            $byFullName = \App\Models\Patient::with($with)
                ->where(function ($q) use ($a, $b, $c) {
                    // first ... last
                    $q->where(function ($qq) use ($a, $b, $c) {
                        $qq->where('first_name', 'like', "%{$a}%")
                            ->when($c,
                                fn($qqq) => $qqq->where('surname', 'like', "%{$c}%"),
                                fn($qqq) => $qqq->where('surname', 'like', "%{$b}%")
                            );
                    })
                    // last ... first (reverse order often typed)
                        ->orWhere(function ($qq) use ($a, $b, $c) {
                            $qq->where('surname', 'like', "%{$a}%")
                                ->when($c,
                                    fn($qqq) => $qqq->where('first_name', 'like', "%{$c}%"),
                                    fn($qqq) => $qqq->where('first_name', 'like', "%{$b}%")
                                );
                        })
                    // concatenated contains
                        ->orWhereRaw(
                            "CONCAT_WS(' ', first_name, middle_name, surname) LIKE ?",
                            ['%' . $a . ' ' . $b . ($c ? ' ' . $c : '') . '%']
                        );
                })
                ->orderByDesc('updated_at')
                ->limit(10)
                ->get();

            if ($byFullName->isNotEmpty()) {
                return response()->json($byFullName);
            }
        }

        // --- 4) Broad fallback (partials; includes normalized partials) ---------------
        $fallback = \App\Models\Patient::with($with)
            ->where(function ($q) use ($query, $normalizedId, $normalizedEmp, $normalizedLike) {
                $q->where('first_name', 'like', "%{$query}%")
                    ->orWhere('middle_name', 'like', "%{$query}%")
                    ->orWhere('surname', 'like', "%{$query}%")
                    ->orWhere('employee_number', 'like', "%{$query}%")
                    ->orWhere('id_number', 'like', "%{$query}%")
                    ->orWhereRaw("CONCAT_WS(' ', first_name, middle_name, surname) LIKE ?", ["%{$query}%"])
                // normalized partials (so "63-2304" or "632304" both work)
                    ->orWhereRaw("LOWER(REPLACE(REPLACE(id_number, ' ', ''), '-', '')) LIKE ?", ['%' . $normalizedId . '%'])
                    ->orWhereRaw("UPPER(REPLACE(REPLACE(employee_number, ' ', ''), '-', '')) LIKE ?", [$normalizedLike]);
            })
            ->orderByDesc('updated_at')
            ->limit(10)
            ->get();

        return response()->json($fallback);
    }

    public function destroy(string $id)
    {
        $patient = Patient::with(['dependents'])->findOrFail($id);

        DB::transaction(function () use ($patient) {
            $this->deletePatientDeep($patient);
        });

        $redirectRoute = $patient->parent_patient_id ? 'dependents.index' : 'patients.index';

        return redirect()->route($redirectRoute)
            ->with('success', 'Patient and all related records were deleted successfully.');
    }

    /**
     * Recursive deep delete with audit log.
     */
    protected function deletePatientDeep(Patient $patient): void
    {
        try {
            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'deleted',
                'description'   => 'Deleted patient (deep) ID: ' . $patient->id,
                'loggable_type' => Patient::class,
                'loggable_id'   => $patient->id,
                'old_values'    => $patient->toArray(),
                'new_values'    => null,
                'ip_address'    => request()->ip(),
                'user_agent'    => request()->header('User-Agent'),
            ]);
        } catch (\Throwable $e) {
            LaravelLog::warning('Audit log failed during deep delete', [
                'patient_id' => $patient->id,
                'error'      => $e->getMessage(),
            ]);
        }

        $patient->loadMissing('dependents');
        foreach ($patient->dependents as $dependent) {
            $this->deletePatientDeep($dependent);
        }

        Consultation::where('patient_id', $patient->id)->delete();
        Triage::where('patient_id', $patient->id)->delete();

        $patient->delete();
    }

    /**
     * BULK DELETE — superadmin only; 302 redirect with flash for Inertia; JSON for API callers.
     */
    public function bulkDestroy(Request $request)
    {
        $user      = $request->user();
        $isInertia = $request->header('X-Inertia') === 'true';

        $validated = $request->validate([
            'ids'   => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'distinct', 'exists:patients,id'],
        ]);

        $ids      = array_values($validated['ids']);
        $patients = Patient::with('company')->whereIn('id', $ids)->get();

        if ($patients->isEmpty()) {
            LaravelLog::info('bulkDestroy: no matching patients for provided IDs', ['ids' => $ids]);

            if ($isInertia) {
                return back()->with('info', 'No matching patients to delete.');
            }
            return response()->json([
                'success'        => true,
                'message'        => 'No matching patients to delete.',
                'deleted_ids'    => [],
                'affected'       => 0,
                'company_counts' => [],
            ]);
        }

        $companyCounts = [];
        foreach ($patients as $p) {
            $name                 = $p->company?->name ?? 'No Company';
            $companyCounts[$name] = ($companyCounts[$name] ?? 0) + 1;
        }

        DB::beginTransaction();
        try {
            Patient::with(['dependents'])
                ->whereIn('id', $ids)
                ->orderBy('id')
                ->chunkById(50, function ($chunk) {
                    foreach ($chunk as $patient) {
                        $this->deletePatientDeep($patient);
                    }
                });

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();

            LaravelLog::error('bulkDestroy failed', [
                'user_id' => optional($user)->id,
                'error'   => $e->getMessage(),
            ]);

            if ($isInertia) {
                return back()->with('error', 'Bulk delete failed: ' . $e->getMessage());
            }
            return response()->json([
                'success' => false,
                'message' => 'Bulk delete failed: ' . $e->getMessage(),
            ], 500);
        }

        $deletedIds = $patients->pluck('id')->values();

        LaravelLog::info('bulkDestroy success', [
            'user_id'        => optional($user)->id,
            'count'          => $deletedIds->count(),
            'ids'            => $deletedIds,
            'company_counts' => $companyCounts,
        ]);

        if ($isInertia) {
            return back()->with([
                'success'            => 'Selected patients deleted successfully.',
                'bulk_delete_result' => [
                    'affected'       => $deletedIds->count(),
                    'company_counts' => $companyCounts,
                    'deleted_ids'    => $deletedIds,
                ],
            ]);
        }

        return response()->json([
            'success'        => true,
            'message'        => 'Patients deleted successfully.',
            'deleted_ids'    => $deletedIds,
            'affected'       => $deletedIds->count(),
            'company_counts' => $companyCounts,
        ]);
    }
}
