<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

// Zip
use ZipArchive;

class PayrollReportsController extends Controller
{
    /**
     * Simple portal page (month/year pickers).
     */
    public function portal()
    {
        return Inertia::render('Reports/Portal', [
            'defaults' => [
                'year'  => now()->year,
                'month' => now()->month,
            ],
        ]);
    }

    /**
     * Payroll-style monthly report grouped into two sheets:
     *  - Employees (account holders, parent_patient_id is NULL)
     *  - Dependents (parent_patient_id is NOT NULL)
     *
     * ⚠️ Constrained to clinics the AUTHENTICATED user can access.
     */
    public function payrollMonthly(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month || $month < 1 || $month > 12, 422, 'Both valid year and month are required.');

        // --- Resolve accessible clinic IDs for this user ---
        if (method_exists($user, 'accessibleClinicIds')) {
            $clinicIds = array_map('intval', (array) $user->accessibleClinicIds());
        } else {
            $ids = [];
            if (method_exists($user, 'accessibleClinics')) {
                $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            }
            if (!empty($user->clinic_id)) {
                $ids[] = (int) $user->clinic_id;
            }
            $clinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        $clinicFilter = !empty($clinicIds) ? $clinicIds : [-1];

        $consultations = Consultation::query()
            ->with([
                'patient:id,first_name,middle_name,surname,employee_number,gender,company_id,parent_patient_id,date_of_birth,relationship',
                'patient.company:id,name',
                'triage:id,consultation_id,triage_level,bill_type,bill_class,payment_point,payment_method',
            ])
            ->whereYear('consultation_date', $year)
            ->whereMonth('consultation_date', $month)
            ->whereIn('clinic_id', $clinicFilter)
            ->orderBy('consultation_date')
            ->get();

        // Parent map for dependents
        $parentIds = $consultations->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentMap = collect();
        if ($parentIds->isNotEmpty()) {
            $parents = Patient::with('company:id,name')
                ->select('id','first_name','middle_name','surname','employee_number','company_id')
                ->whereIn('id', $parentIds)
                ->get()
                ->keyBy('id');
            $parentMap = $parents;
        }

        $generatedAt = now()->format('Y-m-d H:i:s');
        $periodLabel = Carbon::create($year, $month, 1)->isoFormat('MMMM YYYY');

        $fileName = sprintf('Payroll_Report_%d_%02d.xlsx', $year, $month);

        return response()->streamDownload(function () use ($consultations, $generatedAt, $periodLabel, $parentMap) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // Split into Employees vs Dependents
            $employees  = $consultations->filter(fn ($c) => optional($c->patient)->parent_patient_id === null)->values();
            $dependents = $consultations->filter(fn ($c) => optional($c->patient)->parent_patient_id !== null)->values();

            // --- Employees sheet ---
            $writer->getCurrentSheet()->setName($this->safeSheetName('Employees'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['PAYROLL REPORT — EMPLOYEES'],
                ['Period', $periodLabel],
                ['Generated At', $generatedAt],
            ]);
            $this->writePayrollTable($writer, $employees, $headerStyle, $wrapStyle, $parentMap, false);

            // --- Dependents sheet ---
            $writer->addNewSheetAndMakeItCurrent();
            $writer->getCurrentSheet()->setName($this->safeSheetName('Dependents'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['PAYROLL REPORT — DEPENDENTS'],
                ['Period', $periodLabel],
                ['Generated At', $generatedAt],
            ]);
            $this->writePayrollTable($writer, $dependents, $headerStyle, $wrapStyle, $parentMap, true);

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /**
     * DEPENDENTS ONLY grouped by the *parent account holder's company*,
     * with an extra sheet at the END listing all EMPLOYEES consulted.
     *
     * - One sheet per company (dependents columns).
     * - Final sheet "Employees" with employees columns.
     * - Same logic for defaults, casing, IOD tick, billed amount, etc.
     * - Constrained to clinics the authenticated user can access.
     */
    public function payrollMonthlyDependentsByCompany(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month || $month < 1 || $month > 12, 422, 'Both valid year and month are required.');

        // --- Resolve accessible clinic IDs for this user ---
        if (method_exists($user, 'accessibleClinicIds')) {
            $clinicIds = array_map('intval', (array) $user->accessibleClinicIds());
        } else {
            $ids = [];
            if (method_exists($user, 'accessibleClinics')) {
                $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            }
            if (!empty($user->clinic_id)) {
                $ids[] = (int) $user->clinic_id;
            }
            $clinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        $clinicFilter = !empty($clinicIds) ? $clinicIds : [-1];

        $consultations = Consultation::query()
            ->with([
                'patient:id,first_name,middle_name,surname,employee_number,gender,company_id,parent_patient_id,date_of_birth,relationship',
                'patient.company:id,name',
                'triage:id,consultation_id,triage_level,bill_type,bill_class,payment_point,payment_method',
            ])
            ->whereYear('consultation_date', $year)
            ->whereMonth('consultation_date', $month)
            ->whereIn('clinic_id', $clinicFilter)
            ->orderBy('consultation_date')
            ->get();

        // Split datasets
        $dependents = $consultations->filter(fn ($c) => optional($c->patient)->parent_patient_id !== null)->values();
        $employees  = $consultations->filter(fn ($c) => optional($c->patient)->parent_patient_id === null)->values();

        // Parent map (must include company)
        $parentIds = $dependents->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentMap = collect();
        if ($parentIds->isNotEmpty()) {
            $parents = Patient::with('company:id,name')
                ->select('id','first_name','middle_name','surname','employee_number','company_id')
                ->whereIn('id', $parentIds)
                ->get()
                ->keyBy('id');
            $parentMap = $parents;
        }

        // Group dependents by *parent* company name
        $grouped = $dependents->groupBy(function ($c) use ($parentMap) {
            $p = $c->patient;
            $holder = $p && $p->parent_patient_id ? ($parentMap->get($p->parent_patient_id) ?? null) : null;
            $companyName = optional($holder?->company)->name;
            $companyName = $companyName ? trim($companyName) : 'No Company';
            return $companyName;
        })->sortKeys();

        $generatedAt = now()->format('Y-m-d H:i:s');
        $periodLabel = Carbon::create($year, $month, 1)->isoFormat('MMMM YYYY');

        $fileName = sprintf('Payroll_Report_Dependents_ByCompany_%d_%02d.xlsx', $year, $month);

        return response()->streamDownload(function () use ($grouped, $employees, $generatedAt, $periodLabel, $parentMap) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            $isFirstSheet = true;

            // 1) Write dependents sheets grouped by company
            foreach ($grouped as $companyName => $rows) {
                if ($isFirstSheet) {
                    $isFirstSheet = false;
                } else {
                    $writer->addNewSheetAndMakeItCurrent();
                }

                $safeName = $this->safeSheetName($companyName);
                $writer->getCurrentSheet()->setName($safeName);

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['PAYROLL REPORT — DEPENDENTS'],
                    ['Company', $companyName],
                    ['Period', $periodLabel],
                    ['Generated At', $generatedAt],
                ]);

                $rows = $rows->sortBy([
                    ['consultation_date', 'asc'],
                    ['id', 'asc'],
                ])->values();

                $this->writePayrollTable($writer, $rows, $headerStyle, $wrapStyle, $parentMap, true);
            }

            // If there were no dependents, create a placeholder sheet
            if ($grouped->isEmpty()) {
                $writer->getCurrentSheet()->setName($this->safeSheetName('No Dependents'));
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['PAYROLL REPORT — DEPENDENTS'],
                    ['Notice', 'No dependent consultations found for the selected period / clinics.'],
                ]);
                $this->writePayrollTable($writer, collect(), $headerStyle, (new Style())->setShouldWrapText(true), collect(), true);
            }

            // 2) Append Employees sheet at the END (always present)
            $writer->addNewSheetAndMakeItCurrent();
            $writer->getCurrentSheet()->setName($this->safeSheetName('Employees'));

            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['PAYROLL REPORT — EMPLOYEES'],
                ['Period', $periodLabel],
                ['Generated At', $generatedAt],
            ]);

            // Sort employees for stable output
            $employeesSorted = $employees->sortBy([
                ['consultation_date', 'asc'],
                ['id', 'asc'],
            ])->values();

            $this->writePayrollTable($writer, $employeesSorted, $headerStyle, $wrapStyle, $parentMap, false);

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /**
     * NEW: Create SEPARATE XLSX files for each company (dependents of that company only),
     * then bundle them together into a single ZIP for download.
     *
     * - Each XLSX file includes a single sheet "Dependents" for that company.
     * - If there are no dependents at all, we still return a ZIP containing NO_DEPENDENTS.txt.
     * - Adds README.txt with period and timestamp.
     *
     * ⚠️ No frontend/route edits here. Wire this method in routes as you prefer.
     */
    public function payrollMonthlyDependentsZipByCompany(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month || $month < 1 || $month > 12, 422, 'Both valid year and month are required.');

        // --- Resolve accessible clinic IDs for this user ---
        if (method_exists($user, 'accessibleClinicIds')) {
            $clinicIds = array_map('intval', (array) $user->accessibleClinicIds());
        } else {
            $ids = [];
            if (method_exists($user, 'accessibleClinics')) {
                $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            }
            if (!empty($user->clinic_id)) {
                $ids[] = (int) $user->clinic_id;
            }
            $clinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        $clinicFilter = !empty($clinicIds) ? $clinicIds : [-1];

        $consultations = Consultation::query()
            ->with([
                'patient:id,first_name,middle_name,surname,employee_number,gender,company_id,parent_patient_id,date_of_birth,relationship',
                'patient.company:id,name',
                'triage:id,consultation_id,triage_level,bill_type,bill_class,payment_point,payment_method',
            ])
            ->whereYear('consultation_date', $year)
            ->whereMonth('consultation_date', $month)
            ->whereIn('clinic_id', $clinicFilter)
            ->orderBy('consultation_date')
            ->get();

        // Only dependents
        $dependents = $consultations->filter(fn ($c) => optional($c->patient)->parent_patient_id !== null)->values();

        // Parent map (must include company)
        $parentIds = $dependents->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentMap = collect();
        if ($parentIds->isNotEmpty()) {
            $parents = Patient::with('company:id,name')
                ->select('id','first_name','middle_name','surname','employee_number','company_id')
                ->whereIn('id', $parentIds)
                ->get()
                ->keyBy('id');
            $parentMap = $parents;
        }

        // Group dependents by *parent* company name
        $grouped = $dependents->groupBy(function ($c) use ($parentMap) {
            $p = $c->patient;
            $holder = $p && $p->parent_patient_id ? ($parentMap->get($p->parent_patient_id) ?? null) : null;
            $companyName = optional($holder?->company)->name;
            $companyName = $companyName ? trim($companyName) : 'No Company';
            return $companyName;
        })->sortKeys();

        $generatedAt = now()->format('Y-m-d H:i:s');
        $periodLabel = Carbon::create($year, $month, 1)->isoFormat('MMMM YYYY');
        $zipDownloadName = sprintf('Payroll_Dependents_Per_Company_%d_%02d.zip', $year, $month);

        return response()->streamDownload(function () use ($grouped, $generatedAt, $periodLabel, $parentMap) {
            $tmpZip = tempnam(sys_get_temp_dir(), 'zip_');
            $zip = new ZipArchive();
            $opened = $zip->open($tmpZip, ZipArchive::CREATE | ZipArchive::OVERWRITE);
            if ($opened !== true) {
                throw new \RuntimeException('Could not create ZIP archive');
            }

            // README
            $readme = "Payroll Dependents — One XLSX per company\n"
                ."Period: {$periodLabel}\n"
                ."Generated At: {$generatedAt}\n\n"
                ."Each workbook has a single sheet: Dependents.\n";
            $zip->addFromString('README.txt', $readme);

            $createdAny = false;
            $tempFiles = [];

            foreach ($grouped as $companyName => $rows) {
                $createdAny = true;

                // Create a temp XLSX for this company
                $tmpXlsx = tempnam(sys_get_temp_dir(), 'xlsx_');
                $tempFiles[] = $tmpXlsx;

                $writer = new XLSXWriter();
                $writer->openToFile($tmpXlsx);

                // Styles
                $titleStyle  = (new Style())->setFontBold();
                $labelStyle  = (new Style())->setFontBold();
                $headerStyle = (new Style())->setFontBold();
                $wrapStyle   = (new Style())->setShouldWrapText(true);

                // Single sheet
                $writer->getCurrentSheet()->setName($this->safeSheetName('Dependents'));

                // Header
                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ['PAYROLL REPORT — DEPENDENTS'],
                    ['Company', $companyName],
                    ['Period', $periodLabel],
                    ['Generated At', $generatedAt],
                ]);

                // Stable order inside the sheet
                $rows = $rows->sortBy([
                    ['consultation_date', 'asc'],
                    ['id', 'asc'],
                ])->values();

                // Table (dependents columns)
                $this->writePayrollTable($writer, $rows, $headerStyle, $wrapStyle, $parentMap, true);

                $writer->close();

                // Add file to ZIP
                $zipFileName = $this->safeFileName("Dependents_{$companyName}.xlsx");
                $zip->addFile($tmpXlsx, $zipFileName);
            }

            if (!$createdAny) {
                $zip->addFromString('NO_DEPENDENTS.txt', "No dependent consultations found for the selected period / clinics.\n");
            }

            $zip->close();

            // Stream the zip file
            $stream = fopen($tmpZip, 'rb');
            if ($stream === false) {
                // Cleanup temp files before throwing
                foreach ($tempFiles as $f) { @unlink($f); }
                @unlink($tmpZip);
                throw new \RuntimeException('Could not read ZIP for streaming');
            }

            fpassthru($stream);
            fclose($stream);

            // Cleanup temp files
            foreach ($tempFiles as $f) { @unlink($f); }
            @unlink($tmpZip);
        }, $zipDownloadName, [
            'Content-Type' => 'application/zip',
        ]);
    }

    /**
     * Render payroll rows table.
     * If $isDependents is true, customize columns as requested.
     */
    private function writePayrollTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle,
        $parentMap,
        bool $isDependents
    ): void {
        if ($isDependents) {
            // Dependents headers:
            $headers = [
                'Consultation Date',
                'First Name',
                'Middle Name',
                'Surname',
                'Gender',
                'Company',
                'Account Holder Full Name',
                'Account Holder Employee #',
                'Relationship',
                'Injury On Duty',
                'Triage Level',
                'Bill Type',
                'Bill Class',
                'Payment Method',
                'Billed Amount (USD)',
            ];
        } else {
            // Employees headers
            $headers = [
                'Consultation Date',
                'First Name',
                'Middle Name',
                'Surname',
                'Gender',
                'Company',
                'Employee Number',
                'Injury On Duty',
                'Triage Level',
                'Bill Type',
                'Bill Class',
                'Payment Method',
            ];
        }

        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Column widths (1-based indices)
        $opts = $writer->getOptions();
        $opts->setColumnWidth(20, 1); // Consultation Date
        $opts->setColumnWidth(18, 2); // First Name
        $opts->setColumnWidth(18, 3); // Middle Name
        $opts->setColumnWidth(18, 4); // Surname
        $opts->setColumnWidth(10, 5); // Gender
        $opts->setColumnWidth(26, 6); // Company

        if ($isDependents) {
            $opts->setColumnWidth(28, 7);
            $opts->setColumnWidth(20, 8);
            $opts->setColumnWidth(14, 9);
            $opts->setColumnWidth(12, 10);
            $opts->setColumnWidth(14, 11);
            $opts->setColumnWidth(20, 12);
            $opts->setColumnWidth(16, 13);
            $opts->setColumnWidth(16, 14);
            $opts->setColumnWidth(16, 15);
        } else {
            $opts->setColumnWidth(18, 7);
            $opts->setColumnWidth(12, 8);
            $opts->setColumnWidth(14, 9);
            $opts->setColumnWidth(20, 10);
            $opts->setColumnWidth(16, 11);
            $opts->setColumnWidth(16, 12);
        }

        foreach ($dataset as $c) {
            $p      = $c->patient;
            $triage = $c->triage;

            $isDependentRow = !empty($p->parent_patient_id);
            /** @var \App\Models\Patient|null $holder */
            $holder = $isDependentRow ? ($parentMap->get($p->parent_patient_id) ?? null) : null;

            // Company: dependents use holder's company; employees use patient's company
            $company = $isDependentRow
                ? (optional($holder?->company)->name ?? '')
                : (optional($p->company)->name ?? '');

            // Age at consultation date (for bill type default)
            $age = null;
            if (!empty($p->date_of_birth)) {
                try {
                    $dob  = Carbon::parse($p->date_of_birth);
                    $date = $c->consultation_date ? Carbon::parse($c->consultation_date) : now();
                    $age  = $dob->diffInYears($date);
                } catch (\Throwable $e) {
                    $age = null;
                }
            }

            // Defaults if no triage
            $triageLevel   = $triage->triage_level   ?? 'Non-Urgent';
            $paymentMethod = $triage->payment_method ?? 'Payroll';
            $billType      = $triage->bill_type
                ?? (($age !== null && $age > 16) ? 'Consultation Adult' : 'Consultation Child');

            // Bill class default: dependents → Family; employees → Individual
            $billClass     = $triage->bill_class ?? ($isDependentRow ? 'Family' : 'Individual');

            $iodTick   = $c->injury_on_duty ? '✓' : '';

            // "YYYY-MM-DD HH:MM AM/PM" (date from consultation_date, time from created_at)
            $datePart = $c->consultation_date
                ? Carbon::parse($c->consultation_date)->format('Y-m-d')
                : ($c->created_at ? Carbon::parse($c->created_at)->format('Y-m-d') : '');
            $timePart = $c->created_at
                ? Carbon::parse($c->created_at)->format('H:i A')
                : '';
            $consultationStamp = trim($datePart . ($timePart ? ' ' . $timePart : ''));

            // Title-case names
            $firstName  = $this->toTitle($p->first_name ?? '');
            $middleName = $this->toTitle($p->middle_name ?? '');
            $surname    = $this->toTitle($p->surname ?? '');

            if ($isDependents) {
                $holderFirst  = $this->toTitle($holder->first_name ?? '');
                $holderMiddle = $this->toTitle($holder->middle_name ?? '');
                $holderLast   = $this->toTitle($holder->surname ?? '');
                $holderFull   = trim(collect([$holderFirst, $holderMiddle, $holderLast])->filter()->implode(' '));
                $holderEmp    = $holder->employee_number ?? '';
                $relationship = $p->relationship ?? '';

                // Billed Amount (dependents only)
                $billedAmount = $this->billedAmountForDependent($billClass, $relationship);

                $row = [
                    $consultationStamp,
                    $firstName,
                    $middleName,
                    $surname,
                    $p->gender ?? '',
                    $company,
                    $holderFull,
                    $holderEmp,
                    $relationship,
                    $iodTick,
                    $triageLevel,
                    $billType,
                    $billClass,
                    $paymentMethod,
                    $billedAmount,
                ];
            } else {
                // Employees
                $row = [
                    $consultationStamp,
                    $firstName,
                    $middleName,
                    $surname,
                    $p->gender ?? '',
                    $company,
                    $p->employee_number ?? '',
                    $iodTick,
                    $triageLevel,
                    $billType,
                    $billClass,
                    $paymentMethod,
                ];
            }

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    /* ----------------- helpers ----------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') {
            $name = 'Sheet';
        }
        return mb_substr($name, 0, 31);
    }

    /** Safe filename for ZIP entries (ASCII-ish, dots/underscores, max ~80 chars). */
    private function safeFileName(string $name): string
    {
        // Replace invalid filesystem chars
        $name = preg_replace('/[^\w\-. ]+/u', ' ', $name) ?? $name;
        $name = trim(preg_replace('/\s+/', ' ', $name));
        if ($name === '') $name = 'Report';
        // Ensure .xlsx suffix if missing
        if (!preg_match('/\.xlsx$/i', $name)) {
            $name .= '.xlsx';
        }
        // Limit length to be safe for ZIP viewers
        if (mb_strlen($name) > 80) {
            $ext = '.xlsx';
            $base = mb_substr($name, 0, 80 - mb_strlen($ext));
            $name = $base . $ext;
        }
        return $name;
    }

    /** Title-case a string safely (UTF-8). */
    private function toTitle(?string $val): string
    {
        $v = (string) $val;
        $v = mb_convert_case(mb_strtolower($v, 'UTF-8'), MB_CASE_TITLE, 'UTF-8');
        return trim($v);
    }

    /**
     * Determine billed amount for a dependent from bill class,
     * falling back to relationship when bill class is "Self".
     */
    private function billedAmountForDependent(?string $billClass, ?string $relationship): int
    {
        $normalizedClass = $this->normalize($billClass);

        if ($normalizedClass && $normalizedClass !== 'self') {
            switch ($normalizedClass) {
                case 'parent':
                case 'parents':
                    return 15;
                case 'family':
                case 'immediate family':
                    return 10;
                case 'extended family':
                    return 20;
                case 'other':
                    return 25;
            }
        }

        // Bill class Self (or unknown) -> infer from relationship
        $category = $this->relationshipCategory($relationship);

        switch ($category) {
            case 'parent':    return 15;
            case 'immediate': return 10;
            case 'extended':  return 20;
            default:          return 25; // other/unknown
        }
    }

    /** Infer family category from relationship. */
    private function relationshipCategory(?string $relationship): string
    {
        $r = $this->normalize($relationship);
        if ($r === '') return 'other';

        $parents = ['parent','mother','father','mom','dad','guardian'];
        if (in_array($r, $parents, true)) return 'parent';

        $immediate = [
            'spouse','wife','husband','partner',
            'child','son','daughter','stepchild','stepson','stepdaughter','baby','infant'
        ];
        if (in_array($r, $immediate, true)) return 'immediate';

        $extended = [
            'brother','sister','sibling','cousin',
            'aunt','uncle','niece','nephew',
            'grandmother','grandfather','grandparent','grandma','grandpa',
            'in-law','brother-in-law','sister-in-law','mother-in-law','father-in-law','son-in-law','daughter-in-law'
        ];
        if (in_array($r, $extended, true)) return 'extended';

        return 'other';
    }

    /** Normalize strings (lowercase, trimmed, collapse spaces/hyphens/underscores). */
    private function normalize(?string $val): string
    {
        $v = strtolower(trim((string) $val));
        $v = preg_replace('/[\s\-_]+/', ' ', $v) ?? $v;
        return $v;
    }

    /** Write a simple header block. */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        // spacer
        $writer->addRow(Row::fromValues([]));
    }
}
